package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.subsystem.ocsbridge.util.State.StateChangeListener;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A collection of different state objects.
 *
 * @author tonyj
 */
public class AggregateStatus {

    private final Map<Class<? extends Enum>, State> states = new LinkedHashMap<>();
    private final List<State.StateChangeListener<? extends Enum>> listeners = new CopyOnWriteArrayList<>();
    private final List<FutureStatus> waiters = new CopyOnWriteArrayList<>();
    private final StateChangeListener internalStateChangeListener = new InternalStateChangeListener();

    public void add(CCSTimeStamp when, State<?> state) {
        State oldState = states.get(state.getEnumClass());
        if (oldState == null) {
            states.put(state.getEnumClass(), state);
            state.addStateChangeListener(internalStateChangeListener);
            notifyStateChanged(when, state.getState(), null, null);
        } else {
            oldState.setState(when, state.getState());
        }
    }

    State get(Enum type) {
        return states.get(type.getDeclaringClass());
    }

    /**
     * Test if all given states are present in the aggregate status.
     *
     * @param statesToTest The states to look for
     * @return <code>true</code> if all states are present
     */
    public boolean hasState(Enum... statesToTest) {
        for (Enum e : statesToTest) {
            Class<? extends Enum> enumClass = e.getClass();
            State state = states.get(enumClass);
            if (state == null || !state.isInState(e)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Test if any of the given states are present in the aggregate status.
     *
     * @param statesToTest The states to look for
     * @return <code>true</code> if any one or more states are present
     */
    public boolean anyState(Enum... statesToTest) {
        for (Enum e : statesToTest) {
            Class<? extends Enum> enumClass = e.getClass();
            State state = states.get(enumClass);
            if (state != null && state.isInState(e)) {
                return true;
            }
        }
        return false;
    }

    public Collection<State> getStates() {
        return states.values();
    }

    <T extends Enum> void notifyStateChanged(CCSTimeStamp when, final T currentState, final T oldState, String cause) {
        for (StateChangeListener l : listeners) {
            l.stateChanged(when, currentState, oldState, cause);
        }
        for (FutureStatus waiter : waiters) {
            if (hasState(waiter.state)) {
                waiter.complete(null);
            }
        }
    }

    void addStateChangeListener(State.StateChangeListener<? extends Enum> listener) {
        listeners.add(listener);
    }

    void removeStateChangeListener(State.StateChangeListener<? extends Enum> listener) {
        listeners.remove(listener);
    }

    CompletableFuture<Void> waitForStatus(Enum state) {
        FutureStatus waiter = new FutureStatus(state);
        if (hasState(state)) {
            waiter.complete(null);
        } else {
            waiters.add(waiter);
        }
        return waiter;
    }

    @Override
    public String toString() {
        return "AggregateStatus{" + "states=" + states.values() + '}';
    }

    private class InternalStateChangeListener implements StateChangeListener {

        @Override
        public void stateChanged(CCSTimeStamp when, Enum state, Enum oldState, String cause) {
            notifyStateChanged(when, state, oldState, cause);
        }


    }

    /**
     * An implementation of a future which waits for a particular state.
     */
    private class FutureStatus extends CompletableFuture<Void> {

        private final Enum state;

        FutureStatus(Enum state) {
            this.state = state;
        }

    }

}
