package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.utilities.image.DateUtils;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Generate a shutter motion profile json file to send with the image.
 *
 * @author tonyj
 */
public class ShutterMotionProfileSender {

    public JsonFile processMotionDone(ImageName obsId, MotionDone md, boolean isOpen, ShutterMotionProfileFitter fitter) {

        Map<String, Object> jsonData = new LinkedHashMap<>();
        jsonData.put("startTime", handleTimeStamp(md.startTime()));
        jsonData.put("startPosition", md.startPosition());
        jsonData.put("targetPosition", md.targetPosition());
        jsonData.put("endPosition", md.endPosition());
        jsonData.put("targetDuration", handleDuration(md.targetDuration()));
        jsonData.put("actionDuration", handleDuration(md.actualDuration()));
        jsonData.put("side", md.side().toString());
        jsonData.put("isOpen", isOpen);

        List<Object> encoderSamples = new ArrayList<>();
        for (EncoderSample es : md.encoderSamples()) {
            Map<String, Object> sample = new LinkedHashMap<>();
            sample.put("time", handleTimeStamp(es.getTime()));
            sample.put("position", es.getPosition());
            //System.out.printf("EncoderSample: %d %g\n", es.getTime().getUTCInstant().toEpochMilli(), es.getPosition());
            encoderSamples.add(sample);
        }
        jsonData.put("encodeSamples", encoderSamples);

        List<Object> hallTransitions = new ArrayList<>();
        for (HallTransition ht : md.hallTransitions()) {
            Map<String, Object> transition = new LinkedHashMap<>();
            transition.put("time", handleTimeStamp(ht.getTime()));
            transition.put("position", ht.getSensorId());
            transition.put("sensorId", ht.getPosition());
            transition.put("isOn", ht.isOn());

            //System.out.printf("HallTransition: %d %g %d %s\n", ht.getTime().getUTCInstant().toEpochMilli(), ht.getPosition(), ht.getSensorId(), ht.isOn());
            hallTransitions.add(transition);
        }
        jsonData.put("hallTransitions", hallTransitions);
        if (fitter != null) {
            jsonData.put("fitResult", fitter.doFit(md));
        }
        JsonFile jsonFile = new JsonFile(obsId + "_shutterMotionProfile" + (isOpen ? "Open" : "Close") + ".json", "shutterMotionProfile", obsId, "motionProfile", jsonData);
        return jsonFile;
    }
    private Object handleTimeStamp(CCSTimeStamp time) {
        Map<String, Object> result = new LinkedHashMap<>();
        Instant taiInstant = time.getTAIInstant();
        result.put("tai", DateUtils.convertDateToString(taiInstant));
        result.put("mjd", DateUtils.convertDateToMJD(time.getTAIInstant()));
        return result;
    }

    private Object handleDuration(Duration duration) {
        return duration.toMillis();
    }
}
