package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.UnauthorizedLockException;

/**
 *
 * @author tonyj
 */
public class MCMLockHandler implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(MCMLockHandler.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private MCMSubsystem subsystem;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentLockService agentLockService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentLoginService loginService;

    @ConfigurationParameter(description = "Userid to be used to lock subsystems", isFinal = true)
    private final String userId = "mcm";

    @Override
    public void postStart() {
        loginService.login(userId, "noCredentialsNeeded");

        Predicate<BusMessage<? extends Serializable, ?>> stateChangeFilter
                = BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class);
    }

    @Command(type = CommandType.ACTION, description = "Attempt to lock all subsystems", level = Command.NORMAL)
    public void lockAll() {
        Exception lastException = null;
        for (ControlledSubsystem agent : subsystem.getControlledSubsystems()) {
            try {
                agent.lockAndSwitchToNormal(0);
            } catch (ExecutionException x) {
                lastException = x;
                LOG.log(Level.WARNING, "Failed to lock or switch to normal mode for agent: " + agent, x);
            }
        }
        if (lastException != null) {
            throw new RuntimeException("At least one agent could not be locked, see log for details", lastException);
        }
    }

    @Command(type = CommandType.ACTION, description = "Show lock status of all syubsystems", level = Command.NORMAL)
    public String showLocks() {
        List<String> result = new ArrayList<>();
        for (ControlledSubsystem agent : subsystem.getControlledSubsystems()) {
            String target = agent.getAgentName();
            String owner;
            if (agent.getLock() == null) {
                AgentLock lock = agentLockService.getExistingLockForAgent(target);
                owner = lock == null ? "unlocked" : "locked by "+ lock.getOwner();
            } else {
                owner = "locked";
            }
            result.add(String.format("%s: %s", target, owner));
        }
        return result.stream().collect(Collectors.joining("\n"));
    }

    @Command(type = CommandType.ACTION, description = "Unlock all subsystems", level = Command.NORMAL)
    public void unlockAll() throws UnauthorizedLockException, IOException {
        Exception lastException = null;
        for (ControlledSubsystem agent : subsystem.getControlledSubsystems()) {
            try {
                agent.unlock();
            } catch (ExecutionException x) {
                lastException = x;
                LOG.log(Level.WARNING, "Failed to unlock agent: " + agent, x);
            }
        }
        if (lastException != null) {
            throw new RuntimeException("At least one agent could not be unlocked, see log for details", lastException);
        }
    }
}
