package org.lsst.ccs.subsystem.ocsbridge.xml;

import static java.lang.System.exit;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.SALVariable;

/**
 * Given a data provider dictionary create a map of SALClassDescriptors for a
 * given subsystem.
 *
 * @author farrukh, tonyj
 */
public class SALClassDescriptionMaker {

    private final Map<String, SALClassDescription> SALClassMap = new TreeMap<>();
    private static final boolean SALCLASS_DESCRIPTION_MAKER_DEBUG = false;

    /**
     *
     * @param dataProviderDictionary The data provider dictionary
     * @param cscName The csc name (MTCamera, CCCamera, ATCamera)
     * @param subsystemName CCS subsystem name
     * @param level The level at which conversion should be done
     * @param mapping The regular expression map to use.
     * @param attributeType
     * @param resultType
     */
    public SALClassDescriptionMaker(DataProviderDictionary dataProviderDictionary, String cscName, String subsystemName, int level, Mapping mapping, String attributeType, String resultType) {

        boolean trending = "MONITORING".equals(attributeType);
        List<DataProviderInfo> dpInfoList = dataProviderDictionary.getDataProviderInfos();
        
        for (DataProviderInfo dpi : dpInfoList) {
            if (dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE).trim().equals(attributeType)) {
                String type = dpi.getAttributeValue(DataProviderInfo.Attribute.TYPE);
                String description = dpi.getAttributeValue(DataProviderInfo.Attribute.DESCRIPTION);
                String units = dpi.getAttributeValue(DataProviderInfo.Attribute.UNITS);
                String category = trending ? "" : dpi.getAttributeValue(DataProviderInfo.Attribute.CONFIG_CATEGORY);

                // Still needed?
                if (trending) {
                    type = "double";
                }

                // I've changed getPath() below to getFullPath() because I noticed some of the values were missing when 
                // access was attempted via getPath()..
                String originalPathName = dpi.getFullPath().replace("main/", "").trim();
                // Restore old behaviour where items defined at the base level had a leading slash
                if (!trending && !originalPathName.contains("/")) {
                    originalPathName = "/"+originalPathName;
                }

                Mapping.Match match = mapping.match(originalPathName);
                if (match != null) {
                    String rawSALClassParameterName = match.getPatternName() + "/" + match.getPathAfterMatch();
                    String salClassParameterName = makeSALClassParameterName(rawSALClassParameterName.trim(), level);

                    if (SALCLASS_DESCRIPTION_MAKER_DEBUG) {
                        System.out.println(" **** SALClass Description Maker ****");
                        System.out.println(" Original path name " + originalPathName);
                        System.out.println(" Location " + match.getLocation());
                        System.out.println(" Path after Pattern match " + match.getPathAfterMatch());
                        System.out.println(" Pattern Name " + match.getPatternName());
                        System.out.println(" Raw Class Parameter Name " + rawSALClassParameterName);
                        System.out.println(" Level " + level);
                        System.out.println(" SALClassParameter Name " + salClassParameterName);
                        System.out.println(" ");
                    }
                    
                    String associatedClassName = makeSALClassName(level, cscName, subsystemName, resultType, rawSALClassParameterName, category);
                    SALClassDescription salClassDescription = SALClassMap.computeIfAbsent(associatedClassName, (name) -> new SALClassDescription(associatedClassName, level, category));
                    SALClassDescription.BusVariable busVariable = new SALClassDescription.BusVariable(originalPathName);

                    SALClassDescription.SALVariable existingVariable = salClassDescription.getVariable(salClassParameterName);
                    if (existingVariable instanceof SALClassDescription.PatternMatchedSALVariable) {
                        ((SALClassDescription.PatternMatchedSALVariable) existingVariable).addBusVariable(busVariable);
                    } else {
                        String locationVariableName = match.getPatternName() + "Location";
                        SALClassDescription.LocationVariable locationVariable = (SALClassDescription.LocationVariable) salClassDescription.getVariable(locationVariableName);
                        if (locationVariable == null) {
                            locationVariable = new SALClassDescription.LocationVariable(locationVariableName, match.getPatternName() + " location");
                            salClassDescription.add(locationVariable);
                        }
                        SALClassDescription.PatternMatchedSALVariable var = new SALClassDescription.PatternMatchedSALVariable(match.getPatternName(), match.getLocation(), locationVariable, salClassParameterName, type, units, description);
                        var.addBusVariable(busVariable);
                        salClassDescription.add(var);
                    }
                } else {
                    String associatedClassName = makeSALClassName(level, cscName, subsystemName, resultType, originalPathName, category);
                    SALClassDescription salClassDescription = SALClassMap.computeIfAbsent(associatedClassName, (name) -> new SALClassDescription(associatedClassName, level, category));
                    SALClassDescription.BusVariable busVariable = new SALClassDescription.BusVariable(originalPathName);
                    SALClassDescription.SimpleSALVariable var = new SALClassDescription.SimpleSALVariable(makeSALClassParameterName(originalPathName, level), type, units, description, busVariable);
                    salClassDescription.add(var);
                }
            }
        }

        // If there is only one location variable for a given class, then call it location unless it is already called that.
        for (SALClassDescription classDescription : SALClassMap.values()) {

            // Find all location variables
            List<SALVariable> locationVariables = classDescription.getVariables().stream().filter(v -> v instanceof SALClassDescription.LocationVariable).collect(Collectors.toList());
            if (locationVariables.size() == 1) {
                SALVariable theLocationVariable = locationVariables.get(0);
                classDescription.renameVariable(theLocationVariable, "location");
            }
        }
    }

    public Map<String, SALClassDescription> getSALClassDescriptions() {
        return SALClassMap;
    }

    /**
     * COntrstuct the SALClassName
     * @param level The level at which the conversion is being made
     * @param cscName The CSCName
     * @param subsystemName The CCS subsystem name
     * @param salType The SALType 
     * @param path The path
     * @param category The category
     * @return 
     */
    String makeSALClassName(int level, String cscName, String subsystemName, String salType, String path, String category) {

        if (cscName.toLowerCase().contains("comcam")) {
            cscName = "CCCamera";
        }

        // debug
        String subsystemAndComponent = "";
        if (!cscName.isEmpty()) {

            subsystemAndComponent = cscName.substring(0, 1).toUpperCase() + cscName.substring(1);
        } // there should always be a subsystem 
        subsystemName = subsystemName.trim();

        if (!subsystemName.isEmpty()) {
            subsystemAndComponent += "_" + subsystemName.substring(0, 1).toLowerCase() + subsystemName.substring(1);
        }

        String className;

        if (salType.toLowerCase().contains("settingsapplied")) {
            className = cscName + "_" + "logevent" + "_" + subsystemName.toLowerCase();
        } else {
            className = subsystemAndComponent;
        }

        if (path.contains("main/")) {
            path = path.replace("main/", "");
            className = subsystemAndComponent;// and thats it - 

        } else {// go upto level - 

            for (int i = 0; i < level; i++) {
                final String pathComponentI = path.split("/")[i];
                if (pathComponentI.length() > 0) {
                    className = className + "_" + capitalize(pathComponentI);
                }
            }

        }

        if (!(category == null)) {
            if (!category.isEmpty()) {
                className = className + "_" + category;
            }

        }

        // check to see if last character is underscore if so remove it 
        if (className.endsWith("_")) {
            className = className.substring(0, className.length() - 1);
        }

        return className;
    }

    final String makeSALClassParameterName(String path, int level) {
        path = path.replace("main/", "");
        int pathLength = path.split("/").length;
        String variableName = "";
        path = path.replaceAll("-","_");
       
        // Fix forward slash problem Farrukh August 9, 2022
        if(path.startsWith("/"))path = path.replace("/", "");

        if (path.contains("/")) {
            // level better be less than or at most equal to path-length 
            if (level <= pathLength) {
                for (int i = level; i < pathLength; i++) {

                    variableName = variableName + "_" + path.split("/")[i];
                }

                if (variableName.startsWith("_")) {
                    variableName = variableName.substring(1);

                }
            } else {
                System.out.println("!!You have specified level to be longer than the variable path + name!!");
                exit(-1);

            }
        } else {
            variableName = path;
        }
 // Finally remove illegal character "-" from path Farrukh April 8 2022
//         variableName = variableName.replace('-', '_');

        return uncapitalize(variableName).trim();
    }

    public static String capitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toUpperCase() + in.substring(1);
        } else {
            return in.toUpperCase();
        }
    }

    public static String uncapitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toLowerCase() + in.substring(1);
        } else {
            return in.toLowerCase();
        }
    }

}
