package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Main class for xmlmaker command. Uses command line arguments to invoke
 * xmlmaker.
 *
 * @author tonyj
 */
public class Main {

    @Argument(required = true, usage = "Input (.ser) File")
    private File file;

    @Option(name = "-level", aliases = "-l", usage = "Conversion level")
    private int level;

    @Option(name = "-verbose", aliases = "-v", usage = "Turn on verbose output")
    private boolean verbose;

    @Option(name = "-csc", aliases = "-c", usage = "CSC name (e.g. CCCamera, MTCamera, ATCamera)")
    private String cscName;

    @Option(name = "-map", aliases = "-m", usage = "Use alternate mapping file")
    private File mappingFile;

    @Option(name = "-output", aliases = "-o", usage = "Output file name ")
    private String outputFileName;

    @Option(name = "-settingsApplied", aliases = "-s", usage = "Generate settingsApplied ")
    private boolean generateSettingsApplied;

    @Option(name = "-options", aliases = "-f", usage = "Fiole to read options from")
    private File optionsFile;

    private String subsystemName;
    private String componentName;

    public static void main(String[] args) throws FileNotFoundException, IOException, ClassNotFoundException, ParserConfigurationException, TransformerException, SAXException, XPathExpressionException {
        new Main().doMain(args);
    }

    private void doMain(String[] args) throws IOException, FileNotFoundException, ClassNotFoundException, ParserConfigurationException, TransformerException, SAXException, XPathExpressionException {
        CmdLineParser parser = new CmdLineParser(this);
        try {
            // parse the arguments.
            parser.parseArgument(args);
            Mapping mapping = createMapping(mappingFile);

            if (optionsFile != null) {
                // TEMPORARY for testing -- this is currently hardwired to produce the comcam telemetry xml, it needs to be generalized.
                outputFileName = "/home/tonyj/projects/lsst/ocs-ccs/ts_xml/sal_interfaces/CCCamera/CCCamera_Telemetry.xml";
                cscName = "CCCamera";
                XMLMaker2 maker = new XMLMaker2(false);
                Document document = maker.createDocument(XMLMaker2.SALType.TELEMETRY);
                File file1 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-fcs-status-dictionary.ser");
                StatusDataProviderDictionary dictionary1 = readInputFile(file1);
                maker.addXMLFromMonitoringDictionary(document, dictionary1, "CCCamera", "fcs", 0, mapping);

                File file1a = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-bonn-shutter-status-dictionary.ser");
                StatusDataProviderDictionary dictionary1a = readInputFile(file1a);
                maker.addXMLFromMonitoringDictionary(document, dictionary1a, "CCCamera", "bonn_shutter", 1, mapping);

                File file2 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-daq-monitor-status-dictionary.ser");
                StatusDataProviderDictionary dictionary2 = readInputFile(file2);
                maker.addXMLFromMonitoringDictionary(document, dictionary2, "CCCamera", "daq_monitor", 1, mapping);

                File file3 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-rebpower-status-dictionary.ser");
                StatusDataProviderDictionary dictionary3 = readInputFile(file3);
                maker.addXMLFromMonitoringDictionary(document, dictionary3, "CCCamera", "rebpower", 1, mapping);

                File file4 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-vacuum-status-dictionary.ser");
                StatusDataProviderDictionary dictionary4 = readInputFile(file4);
                maker.addXMLFromMonitoringDictionary(document, dictionary4, "CCCamera", "vacuum", 1, mapping);

                File file5 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-quadbox-status-dictionary.ser");
                StatusDataProviderDictionary dictionary5 = readInputFile(file5);
                maker.addXMLFromMonitoringDictionary(document, dictionary5, "CCCamera", "quadbox", 1, mapping);

                File file6 = new File("src/main/resources/org/lsst/ccs/subsystem/ocsbridge/sim/comcam-fp-status-dictionary.ser");
                StatusDataProviderDictionary dictionary6 = readInputFile(file6);
                maker.addXMLFromMonitoringDictionary(document, dictionary6, "CCCamera", "focal_plane", 1, mapping);

                maker.writeXML(new StreamResult(new File(outputFileName)), document);
                System.out.println("Wrote " + outputFileName);
                // Now do the same for settings applied
                String settingsApplideOutputFileName = "/home/tonyj/projects/lsst/ocs-ccs/ts_xml/sal_interfaces/CCCamera/CCCamera_Events.xml";
                Document settingsAppliedDocument = maker.createDocument(XMLMaker2.SALType.SETTINGS_APPLIED);

                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary1, "CCCamera", "fcs", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary1a, "CCCamera", "bonn_shutter", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary2, "CCCamera", "daq_monitor", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary3, "CCCamera", "rebpower", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary4, "CCCamera", "vacuum", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary5, "CCCamera", "quadbox", 1, mapping);
                maker.addXMLFromConfigDictionary(settingsAppliedDocument, dictionary6, "CCCamera", "focal_plane", 1, mapping);

                DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
//                dbf.setValidating(true);
//                dbf.setIgnoringElementContentWhitespace(true);
                // Remove whitespace, otherwise XML comes out double-spaced
                DocumentBuilder db = dbf.newDocumentBuilder();
                Document doc = db.parse(new File(settingsApplideOutputFileName));
                XPath xp = XPathFactory.newInstance().newXPath();
                NodeList nl = (NodeList) xp.evaluate("//text()[normalize-space(.)='']", doc, XPathConstants.NODESET);
                for (int i = 0; i < nl.getLength(); ++i) {
                    Node node = nl.item(i);
                    node.getParentNode().removeChild(node);
                }
                Element root = doc.getDocumentElement();
                NodeList childNodes = root.getChildNodes();
                int startIndex = 0;
                int endIndex = 0;
                for (int i = 0; i < childNodes.getLength(); i++) {
                    Node node = childNodes.item(i);
                    if (node instanceof Comment) {
                        Comment comment = (Comment) node;
                        if (comment.getData().trim().startsWith("SETTINGSAPPLIED")) {
                            startIndex = i;
                        } else if (comment.getData().trim().startsWith("ENDSETTINGSAPPLIED")) {
                            endIndex = i;
                        }
                    }
                }
                for (int i = startIndex + 1; i < endIndex; i++) {
                    root.removeChild(childNodes.item(startIndex).getNextSibling());
                }
                Node nextSibling = childNodes.item(startIndex).getNextSibling();
                NodeList replacementNodes = settingsAppliedDocument.getDocumentElement().getChildNodes();
                for (int i = 0; i < replacementNodes.getLength(); i++) {
                    Node newNode = doc.importNode(replacementNodes.item(i), true);
                    root.insertBefore(newNode, nextSibling);
                }
                maker.writeXML(new StreamResult(new File(settingsApplideOutputFileName)), doc);
                System.out.println("Wrote " + settingsApplideOutputFileName);

            } else {
                Object readObject = readInputFile(this.file);
                if (verbose) {
                    System.out.println(readObject.getClass().getTypeName());
                }

                if (outputFileName == null) {
                    String inputFileName = file.getName();
                    if (inputFileName.endsWith(".ser")) {
                        outputFileName = inputFileName.replace(".ser", ".xml");
                    } else {
                        outputFileName = inputFileName + ".xml";
                    }
                    if (generateSettingsApplied) {
                        outputFileName = outputFileName.replace(".xml", "SettingsApplied.xml");
                    }

                }

                XMLMaker2 maker = new XMLMaker2(verbose);
                Document document;
                if (readObject instanceof StatusDataProviderDictionary) {
                    final StatusDataProviderDictionary dictionary = (StatusDataProviderDictionary) readObject;
                    computeSubsystemAndComponentNames(dictionary.getOriginAgentInfo(), cscName);
                    if (generateSettingsApplied) {
                        document = maker.makeXMLFromConfigDictionary(dictionary, subsystemName, componentName, level, mapping);
                    } else {
                        document = maker.makeXMLFromDataProviderDictionary(dictionary, subsystemName, componentName, level, mapping);
                    }
                } else {
                    throw new IllegalArgumentException("Invalid .ser file, containts " + readObject.getClass());
                }
                maker.writeXML(new StreamResult(new File(outputFileName)), document);
                System.out.println("Wrote " + outputFileName);
            }

        } catch (CmdLineException e) {

            System.err.println(e.getMessage());
            System.err.println("java XMLMaker [options...] arguments...");
            // print the list of available options
            parser.printUsage(System.err);
            System.err.println();
        }
    }

    static StatusDataProviderDictionary readFromPath(String path) throws IOException, ClassNotFoundException {
        try (InputStream in = MakeAllXML.class.getResourceAsStream("/org/lsst/ccs/subsystem/ocsbridge/sim/"+path)) {
            if (in == null) throw new IOException("Path not found "+path);
            return readInputStream(in);
        }
    }
    
    static StatusDataProviderDictionary readInputFile(File inputFile) throws IOException, ClassNotFoundException {
        try ( FileInputStream fis = new FileInputStream(inputFile)) {  
            return readInputStream(fis);
        }
    }
    
    static StatusDataProviderDictionary readInputStream(InputStream inputStream) throws IOException, ClassNotFoundException {
        try ( BufferedInputStream bis = new BufferedInputStream(inputStream);  
              ObjectInputStream ois = new ObjectInputStream(bis)) {
            return (StatusDataProviderDictionary) ois.readObject();
        }
    }

    private void computeSubsystemAndComponentNames(AgentInfo agentInfo, String cscName) {

        //TODO: Is relying on toString really the best way of handling this?
        String subsystemAndComponent = agentInfo.toString().split(":")[0];

        subsystemName = "";
        componentName = "";

        /*
        if(subsystemAndComponent.contains("comcam-")){
        
            componentName = subsystemAndComponent.replace("comcam-", "");
            subsystemName = "CCCamera";
        } else {
        
          subsystemName = "MTCamera";
          componentName = subsystemAndComponent;
        }*/
 /*
        if (subsystemAndComponent.contains("-")) {
            subsystemName = subsystemAndComponent.split("-")[0];
            componentName = subsystemAndComponent.split("-")[1];
        } else {
            componentName = subsystemAndComponent;
        }*/
        if (cscName != null) {
            subsystemName = cscName;
            componentName = subsystemAndComponent;
        } else {
            if (subsystemAndComponent.contains("comcam-")) {
                subsystemName = "CCCamera";
                componentName = subsystemAndComponent.replace("comcam-", "").trim();
            } else {
                subsystemName = "MTCamera";
                componentName = subsystemAndComponent;
            }
        }

        if (verbose) {
            System.out.println(agentInfo);
            System.out.println(subsystemAndComponent);
            System.out.println(" Subsystem  Name " + subsystemName + " Component Name " + componentName);
        }
    }

    static Mapping createMapping(File mappingFile) throws IOException {
        Mapping mapping;
        if (mappingFile == null) {
            mapping = Mapping.defaultMapping();
        } else {
            try ( InputStream in = new FileInputStream(mappingFile)) {
                mapping = new Mapping(in);
            }
        }
        return mapping;
    }
}
