package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.zip.Checksum;

/**
 *
 * @author tonyj
 */
public class SALClassDescription {

    private final String className;
    private final Map<String, SALVariable> variables = new TreeMap<>(new VariableComparator());
    private final int level;
    private final String category;

    /**
     * Create new SALClassDescription
     * @param className The name of the SAL class
     * @param level The level at which the conversion was done. 
     */
    SALClassDescription(String className, int level, String category) {
        this.className = className;
        this.level = level;
        this.category = category;
    }
    
    SALVariable getVariable(String variableName) {
        return variables.get(variableName);
    }
    
    void add(SALVariable var) {
        variables.put(var.getVariableName(), var);
    }

    public String getClassName() {
        return className;
    }

    public List<SALVariable> getVariables() {
        return new ArrayList(variables.values());
    }

    public int getLevel() {
        return level;
    }

    public String getCategory() {
        return category;
    }

    @Override
    public String toString() {
        return "SALClassDescription{" + "className=" + className + ", level=" + level + ", category=" + category +
               getVariables().stream().map(v->v.toString()).collect(Collectors.joining("\n\t","\n\t","\n"))        
               + '}';
    }
    
    public void updateChecksum(Checksum sum) {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
            try (DataOutputStream dos = new DataOutputStream(bos)) {
                dos.writeUTF(className);
                dos.writeInt(level);
                if (category != null) dos.writeUTF(category);
            }
            bos.flush();
           // Note we don't include units or description in the checksum
            sum.update(bos.toByteArray(), 0, bos.size());
        } catch (IOException x) {
            throw new RuntimeException("Unpexected error computing checksum", x);
        }
        for (SALVariable variable : variables.values()) {
            variable.updateCheckSum(sum);
        }
    }

    /** 
     * Rename the given variable. This is used to rename things like rebLocation to location
     * if there is only one location variable in a class.
     * @param variable The variable to rename
     * @param newName The new name
     */
    void renameVariable(SALVariable variable, String newName) {
        String oldName = variable.getVariableName();
        variables.remove(oldName);
        variable.setName(newName);
        variables.put(newName, variable);
    }
     
    public static abstract class SALVariable {

        private String variableName;
        private final String type;
        private final String units;
        private final String description;

        public SALVariable(String variableName, String type, String units, String description) {
            if (type == null || type.isEmpty()) {
                //throw new RuntimeException("Illegal null/empty type for variableName "+variableName);
                type = "String";
            }
            this.variableName = variableName;
            this.type = type;
            this.units = units;
            this.description = description;
        }

        public String getVariableName() {
            
            return variableName;
        }

        public String getType() {
            return type;
        }

        public String getUnits() {
            return units;
        }

        public String getDescription() {
            return description;
        }
        
        public int getCount() {
            return 1;
        }

        @Override
        public String toString() {
            return "variableName=" + variableName + ", type=" + type + ", units=" + units + ", description=\"" + description+"\"";
        }

        private void setName(String newName) {
            this.variableName = newName;
        }
        
        private void updateCheckSum(Checksum sum) {
            try (ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
                try (DataOutputStream dos = new DataOutputStream(bos)) {
                    dos.writeUTF(variableName);
                    dos.writeUTF(type);
                    // We deliberately do not include description, units or subclass info
                }
                bos.flush();
               // Note we don't include units or description in the checksum
                sum.update(bos.toByteArray(), 0, bos.size());
            } catch (IOException x) {
                throw new RuntimeException("Unexpected error computing checksum", x);
            }
        }
    }

    /**
     * A SAL variable which does not match a pattern
     */
    public static class SimpleSALVariable extends SALVariable {

        private final BusVariable busVariable;

        public SimpleSALVariable(String variableName, String type, String units, String description, BusVariable busVariable) {
            super(variableName, type, units, description);
            this.busVariable = busVariable;
        }

        public BusVariable getBusVariable() {
            return busVariable;
        }

        @Override
        public String toString() {
            return "SimpleSALVariable{" + super.toString() + " busVariable=" + busVariable +  '}';
        }
    }

    /**
     * A SAL variable that has been fabricated to keep track of locations for variables which match patterns. 
     * It does not have a corresponding bus variable
     */
    public static class LocationVariable extends SALVariable {


        public LocationVariable(String variableName, String description) {
            super(variableName, "String", "unitless", description);
        }


        @Override
        public String toString() {
            return "LocationVariable{" + super.toString() + '}';
        }
    }
    
    /**
     * A pattern matched variable is one which matched a pattern, and was converted to an array of values
     * corresponding to all the instance which match the pattern. A corresponding locationVariable tracks the
     * order of elements in the array.
     */
    public static class PatternMatchedSALVariable extends SALVariable {

        private final String patternName;
        private final String pattern;
        private final List<BusVariable> busVariables = new ArrayList<>();
        private final LocationVariable locationVariable;

        public PatternMatchedSALVariable(String patternName, String pattern, LocationVariable location, String variableName, String type, String units, String description) {
            super(variableName, type, units, description);
            this.patternName = patternName;
            this.pattern = pattern;
            this.locationVariable = location;
        }
        
        void addBusVariable(BusVariable var) {
            busVariables.add(var);
        }

        public String getPatternName() {
            return patternName;
        }

        public String getPattern() {
            return pattern;
        }

        public List<BusVariable> getBusVariables() {
            return busVariables;
        }

        public LocationVariable getLocationVariable() {
            return locationVariable;
        }
        
        @Override
        public int getCount() {
            return busVariables.size();
        }

        @Override
        public String toString() {
            return "PatternMatchedSALVariable{"  + super.toString() + " patternName=" + patternName + ", pattern=" + pattern + ", locationVariable=" + locationVariable +
                  busVariables.stream().map(v->v.toString()).collect(Collectors.joining("\n\t\t","\n\t\t","\n\t")) + 
            '}';
        }
    }

    public static class BusVariable {
        private final String pathAndNameOnBuses;

        public BusVariable(String pathAndNameOnBuses) {
            this.pathAndNameOnBuses = pathAndNameOnBuses;
        }

        public String getPathAndNameOnBuses() {
            return pathAndNameOnBuses;
        }

        @Override
        public String toString() {
            return "BusVariable{" + "pathAndNameOnBuses=" + pathAndNameOnBuses + '}';
        }
        
    }
    
  
    /**
     * Compute the maximum count of any PatternMatchedSalVariables which share
     * the same location as this variable
     * @param sv The variable to test
     * @return The maximum count
     */
    int getSharedLocationCount(SALVariable sv) {

        // only enter the if block if it is a PatternMatchedSALVariable
        if (sv instanceof PatternMatchedSALVariable) {
            PatternMatchedSALVariable pmv = (PatternMatchedSALVariable) sv;
            // Find all the patternMatchedVariables which share this location, the calculate the maximum count            
            OptionalInt count = variables.values().stream()
                    .filter(var -> var instanceof PatternMatchedSALVariable)
                    .map(var -> (PatternMatchedSALVariable) var)
                    .filter(var -> var.getLocationVariable() == pmv.getLocationVariable())
                    .mapToInt(var -> var.getCount())
                    .max();
            
            return count.isPresent() ? count.getAsInt() : 0;
        } else {
            return 1;
        }
    }

    /**
     * In earlier versions of CCS, the warnHi and warnLo were called dbandHi and dbandLo
     * Changing this results in the order of all the limits changing, so for now we restore the
     * old ordering to keep the number of changes to the XML manageable.
     */
    private static class VariableComparator implements Comparator<String> {

        @Override
        public int compare(String o1, String o2) {
            o1 = o1.replace("warnHi", "dbandHi");
            o1 = o1.replace("warnLo", "dbandLo");            
            o2 = o2.replace("warnHi", "dbandHi");
            o2 = o2.replace("warnLo", "dbandLo");    
            return o1.compareTo(o2);
        }
    }
}
