package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Interface to shutter used by MCM. 
 *
 * @author tonyj
 */
public class Shutter {

    /**
     * Time needed to move a shutter blade
     */
    static final Duration MOVE_TIME = Duration.ofMillis(980);
    private final MCMConfig config;

    public enum ShutterReadinessState {

        NOT_READY, READY, GETTING_READY
    };

    public enum ShutterState {

        CLOSED, OPENING, OPEN, CLOSING
    };

    private final State shutterReadinessState;
    private final State shutterState;
    private ShutterInterface shutter;

    private final CCS ccs;

    Shutter(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        // If no PREP time, then shutter is always ready
        shutterReadinessState = new State(config.getShutterPrepTime() > 0 ? ShutterReadinessState.NOT_READY : ShutterReadinessState.READY);
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().add(now, shutterReadinessState);
        shutterState = new State(ShutterState.CLOSED);
        ccs.getAggregateStatus().add(now, shutterState);
        // By default assumes simulation
        shutter = new ShutterSimulation(ccs, shutterState, config);
    }

    public void startImageSequence() {
        shutter.setImageSequence(true);
    }

    public void endImageSequence() {
        shutter.setImageSequence(false);
    }


    void prepare() {
        shutter.prepare();
    }

    void expose(Duration exposureTime) throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.expose(exposureTime);
    }

    void start(String configName) throws ExecutionException {
        shutter.start(configName);
    }

    void open() throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.open();
    }

    void close() throws ExecutionException {
        if (!shutterState.isInState(ShutterState.CLOSED)) {
            shutter.close();
        }
    }

    ControlledSubsystem registerMCMSubsystem(MCMSubsystem mcm) {
        final ShutterSubsystemLayer controlledSubsystem = new ShutterSubsystemLayer(mcm, ccs, config);
        shutter = controlledSubsystem;
        return controlledSubsystem;
    }
}
