package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.io.Serializable;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;

/**
 * A base class for subsystems which are actively controlled by the MCM
 *
 * @author tonyj
 */
class ControlledSubsystem {

    private static final Logger LOG = Logger.getLogger(ControlledSubsystem.class.getName());

    protected final MCMConfig config;
    protected final MCMCommandSender commandSender;
    protected final CCS ccs;
    private final String targetSubsystem;
    private final AgentLockService agentLockService;
    private final AlertService alertService;
    private final Alert goneAlert;
    private final Alert faultAlert;
    
    public ControlledSubsystem(Subsystem mcm, String targetSubsystem, CCS ccs, MCMConfig config) {
        this.config = config;
        this.agentLockService = mcm.getAgentService(AgentLockService.class);
        this.alertService = mcm.getAgentService(AlertService.class);
        String goneAlertId = targetSubsystem+"_gone";
        this.goneAlert = new Alert(goneAlertId, "Alert raised when controlled subsystem disappears.");
        alertService.registerAlert(goneAlert); 

        String faultAlertId = targetSubsystem+"_fault";
        this.faultAlert = new Alert(faultAlertId, "Alert raised when controlled subsystem goes into fault state.");
        alertService.registerAlert(faultAlert);
        
        this.commandSender = new MCMCommandSender(targetSubsystem, mcm.getMessagingAccess());
        this.ccs = ccs;
        this.targetSubsystem = targetSubsystem;

        mcm.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(targetSubsystem)) {
                        // TODO: It would be better to get the locally cached state from the core, rather than get it from the
                        // remote subsystem to avoid need to deal with exceptions.
                        try {
                            StateBundle initialState = commandSender.sendCommand(StateBundle.class, "getState");
                            // TODO: Perhaps better to call state change after onConnect?
                            alertService.raiseAlert(goneAlert, AlertState.NOMINAL, String.format("Controlled subsystem %s has connected", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                            onConnect(agent, initialState);
                        } catch (ExecutionException x) {
                            LOG.log(Level.SEVERE, "Failed to get initial state from subsystem "+targetSubsystem, x);
                        }
                    }
                }
            }

            @Override
            public void disconnected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(targetSubsystem)) {
                        if (agentLockService.getExistingLockForAgent(targetSubsystem) != null) {
                            alertService.raiseAlert(goneAlert, AlertState.ALARM, String.format("Controlled subsystem %s has disconnected", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);                            
                        }
                        onDisconnect(agent);
                    }
                }
            }
        });

        Predicate<BusMessage<? extends Serializable, ?>> targetSubsystemFilter = BusMessageFilterFactory.messageOrigin(targetSubsystem);
        mcm.getMessagingAccess().addStatusMessageListener((msg) -> {
            if (msg instanceof StatusStateChangeNotification) {
                boolean isInFault = ((StatusStateChangeNotification) msg).getNewState().isInState(OperationalState.ENGINEERING_FAULT);
                if (isInFault && agentLockService.getExistingLockForAgent(targetSubsystem) != null) {
                    alertService.raiseAlert(faultAlert, AlertState.ALARM, String.format("Controlled subsystem %s has gone into FAULT", targetSubsystem));
                } else if (!isInFault) {
                    alertService.raiseAlert(faultAlert, AlertState.NOMINAL, String.format("Controlled subsystem %s is no longer in FAULT", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                }
                onStateChange((StatusStateChangeNotification) msg);
            } else if (msg instanceof StatusHeartBeat) {
                // Ignored
            } else {
                onEvent(msg);
            }
        }, targetSubsystemFilter);
    }

    public AgentLock lockAndSwitchToNormal(int level) throws ExecutionException {
        try {
            AgentLock lock = agentLockService.getLockForAgent(targetSubsystem);
            if (lock == null) {
                LOG.log(Level.INFO, "Locking agent {0}", targetSubsystem);
                agentLockService.setLevelForAgent(targetSubsystem, level);
            } 
            try {
                commandSender.sendCommand("switchToNormalMode");
            } catch (ExecutionException x) {
                // No point leaving it locked in this case
                agentLockService.unlockAgent(targetSubsystem);
                throw x;
            }
            return lock;
        } catch (IOException | UnauthorizedLevelException | UnauthorizedLockException x) {
            throw new ExecutionException("Failed to lock subsystem " + targetSubsystem, x);
        }
    }
    
    public void start(String configName) throws ExecutionException {
        commandSender.sendCommand("publishConfigurationInfo");
    }

    /**
     * This method is called whenever the target subsystems connects, including
     * initially when the the MCM itself starts up if the target subsystem is
     * already running.
     *
     * @param agent
     */
    protected void onConnect(AgentInfo agent, StateBundle initialState) {

    }

    protected void onDisconnect(AgentInfo agent) {

    }

    protected void onStateChange(StatusStateChangeNotification statusChange) {

    }

    protected void onEvent(StatusMessage msg) {
    }

    String getAgentName() {
        return targetSubsystem;
    }
}
