package org.lsst.ccs.subsystem.ocsbridge;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.subsystem.imagehandling.data.HeaderServiceEnabled;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData.Header;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.sal.SALEvent;

/**
 *
 * @author tonyj
 */
public class HeaderServiceEventHandler implements EventListener<SALEvent>, HasLifecycle, HasDataProviderInfos {

    private static final Logger LOG = Logger.getLogger(HeaderServiceEventHandler.class.getName());
    private final ObjectMapper mapper;
    private final TypeReference<Map<String, List<Header>>> yamlType = new TypeReference<Map<String, List<Header>>>() {
    };

    private volatile boolean headerServiceEnabled = true;
    private final Object hseLock = new Object();
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsystem;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dataProviderDictionaryService;

    public HeaderServiceEventHandler() {
        mapper = new ObjectMapper(new YAMLFactory());
        mapper.findAndRegisterModules();
        mapper.addMixIn(Header.class,HeaderMixIn.class);
    }
    
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(HeaderServiceEnabled.class, HeaderServiceEnabled.EVENT_KEY);
        dataProviderDictionaryService.registerClass(ImageHeaderData.class, ImageHeaderData.EVENT_KEY);
    }
    

    @Override
    public void eventFired(SALEvent event) {
        if (event instanceof org.lsst.sal.atheader.event.LargeFileObjectAvailableEvent) {
            org.lsst.sal.atheader.event.LargeFileObjectAvailableEvent lfoe = (org.lsst.sal.atheader.event.LargeFileObjectAvailableEvent) event;
            String url = lfoe.getUrl();
            handleEvent(url);
        } else if (event instanceof org.lsst.sal.ccheader.event.LargeFileObjectAvailableEvent) {
            org.lsst.sal.ccheader.event.LargeFileObjectAvailableEvent lfoe = (org.lsst.sal.ccheader.event.LargeFileObjectAvailableEvent) event;
            String url = lfoe.getUrl();
            handleEvent(url);
        } else if (event instanceof org.lsst.sal.atheader.states.SummaryStateEvent) {
            org.lsst.sal.atheader.states.SummaryStateEvent sse = (org.lsst.sal.atheader.states.SummaryStateEvent) event;
            LOG.log(Level.INFO, "Header service state changed {0}", sse);
            HeaderServiceEnabled hse = new HeaderServiceEnabled(sse.getSubstate() == org.lsst.sal.atheader.states.SummaryStateEvent.SummaryState.ENABLED); 
            processHeaderServiceEnabled(hse);
        } else if (event instanceof org.lsst.sal.ccheader.states.SummaryStateEvent) {
            org.lsst.sal.ccheader.states.SummaryStateEvent sse = (org.lsst.sal.ccheader.states.SummaryStateEvent) event;
            LOG.log(Level.INFO, "Header service state changed {0}", sse);
            HeaderServiceEnabled hse = new HeaderServiceEnabled(sse.getSubstate() == org.lsst.sal.ccheader.states.SummaryStateEvent.SummaryState.ENABLED); 
            processHeaderServiceEnabled(hse);
        }
    }
    
    //When the provided HeaderServiceEnabled object is null, then we
    //publish the last stored value. This is triggered by an ImageHandler
    //subsystem joining the cluster.
    private void processHeaderServiceEnabled(HeaderServiceEnabled hse) {
        synchronized(hseLock) {
            if ( hse != null ) {
                headerServiceEnabled = hse.isEnabled();
            } else {
                hse = new HeaderServiceEnabled(headerServiceEnabled);
            }
            KeyValueData kvd = new KeyValueData(HeaderServiceEnabled.EVENT_KEY, hse);
            subsystem.publishSubsystemDataOnStatusBus(kvd);
        }
    }
    
    void handleEvent(String url) {
        try {
            LOG.log(Level.INFO, "Header service got {0}", url);
            Map<String, List<Header>> header = mapper.readValue(new URL(url), yamlType);
            sendPrimaryHeaderViaCCS(header.get("PRIMARY"));
        } catch (IOException ex) {
            LOG.log(Level.SEVERE, "Unexpected exception handling header service URL", ex);
        }
    }

    void sendPrimaryHeaderViaCCS(List<Header> data) {
        KeyValueData kvd = new KeyValueData(ImageHeaderData.EVENT_KEY, new ImageHeaderData(data));
        subsystem.publishSubsystemDataOnStatusBus(kvd);
        LOG.log(Level.FINE, "Sent {0}", kvd);
    }

    /// Used to teach Jackson how to create Header class
    public static abstract class HeaderMixIn {

        @JsonCreator
        public HeaderMixIn(@JsonProperty("keyword") String keyword, @JsonProperty("value") String value, @JsonProperty("comment") String comment) {

        }
    }

    @Override
    public void publishDataProviderCurrentData(AgentInfo... agents) {
        //Publish the data only for image handlers
        for ( AgentInfo a : agents ) {
            if ( a.getAgentProperties().getProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"").equals(AgentCategory.IMAGE_HANDLER.name()) ) {
                processHeaderServiceEnabled(null);
                break;                
            }            
        }        
    }

}
