package org.lsst.ccs.subsystem.mmm.shuttersim;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;

import org.lsst.ccs.subsystem.mmm.data.InvalidStateException;
import org.lsst.ccs.subsystem.mmm.data.ShutterReadinessState;
import org.lsst.ccs.subsystem.mmm.data.ShutterState;

// like an enum but with subclasses

public abstract class ShutterInternalState implements Serializable {

	// TODO does not handle both blades moving simultaneously,
	// we don't differentiate here CLOSING and CLOSING_OPENING
	// (we are CLOSING for short exposures, losing track that the leading blade
	// is still moving)

	private static final long serialVersionUID = -5903797676508388449L;

	/**
	 * Time needed to get the shutter ready
	 */
	static private final Duration preparationTime = Duration.ofMillis(150);
	/**
	 * Time shutter can remain ready after prepare or move
	 */
	static private final Duration readyDuration = Duration.ofMillis(4000);
	/**
	 * Time needed to move a shutter blade
	 */
	static private final Duration moveTime = Duration.ofMillis(980);

	protected ShutterInternalState(String value) {
		this.value = value;
		fromValues.put(value, this);
	}

	static public ShutterInternalState valueOf(String n) {
		return fromValues.get(n);
	}

	@Override
	public String toString() {
		return value;
	}

	static private Instant startOpeningTime;
	static private Instant endOpeningTime;
	static private Instant startClosingTime;
	static private Instant endClosingTime;

	public synchronized static void setStartClosingTime(Instant startClosingTime) {
		ShutterInternalState.startClosingTime = startClosingTime;
	}

	public synchronized static void setStartOpeningTime(Instant startOpeningTime) {
		ShutterInternalState.startOpeningTime = startOpeningTime;
	}

	public synchronized static void setEndClosingTime(Instant endClosingTime) {
		ShutterInternalState.endClosingTime = endClosingTime;
	}

	public synchronized static void setEndOpeningTime(Instant endOpeningTime) {
		ShutterInternalState.endOpeningTime = endOpeningTime;
	}

	public long getLastEffectiveExposureTime() {
		throw new InvalidStateException(toString(),
				"cannot get effective exposure time when not closed");
	}

	public long getLastTotalOpeningTime() {
		throw new InvalidStateException(toString(),
				"cannot get total opening time when not closed");
	}

	public ShutterState getPublicState() {
		return ShutterState.valueOf(toString()); // internal states map directly
													// except closed
	}

	public ShutterReadinessState getPublicReadinessState() {
		return ShutterReadinessState.READY; // internal states are READY except
											// some closed
	}

	public ShutterInternalState prepare(ShutterSim s) {
		s.getLogger().error("invalid prepare command, from state " + this);
		throw new InvalidStateException("SHUTTER:" + this.toString(), "prepare");
	}

	public ShutterInternalState expose(ShutterSim s, Duration integrationTime) {
		s.getLogger().error("invalid expose command, from state " + this);
		throw new InvalidStateException("SHUTTER:" + this.toString(), "expose");
	}

	public ShutterInternalState close(ShutterSim s) {
		s.getLogger().error("invalid close command, from state " + this);
		throw new InvalidStateException("SHUTTER:" + this.toString(), "prepare");
	}

	public void enter(ShutterSim s) {
		// default: do nothing
	}

	public void exit(ShutterSim s) {
		// default: do nothing
	}

	public static synchronized ShutterInternalState scheduled(ShutterSim s,
			ShutterInternalState from, ShutterInternalState to) {

		s.getLogger().info(
				"scheduled from " + from + " to " + to + " actual "
						+ s.getState());

		if (s.getState() != from) {
			s.getLogger().info(
					" ignored scheduled: " + to + " because state is now "
							+ s.getState() + " expecting " + from);
			return s.getState();
		}

		s.getLogger().debug(" scheduled: " + to);
		s.setState(to);
		return to;
	}

	public ScheduledFuture<ShutterInternalState> scheduleTransition(
			ShutterSim s, Duration delay, ShutterInternalState from,
			ShutterInternalState to) {
		return s.scheduleTransition(delay, from, to);
	}

	private String value;
	private static Map<String, ShutterInternalState> fromValues = new HashMap<String, ShutterInternalState>();

	public final static ShutterInternalState OPEN = new ShutterInternalState(
			"OPEN") {

		/**
         * 
         */
		private static final long serialVersionUID = -4150157131229089583L;

		@Override
		public ShutterInternalState close(ShutterSim s) {
			return CLOSING;
		};
	};

	public final static ShutterInternalState OPENING = new ShutterInternalState(
			"OPENING") {
		/**
         * 
         */
		private static final long serialVersionUID = -8748630599392814682L;

		@Override
		public void enter(ShutterSim s) {
			setStartOpeningTime(Instant.now());
			scheduleTransition(s, moveTime, OPENING, OPEN);
		};

		@Override
		public void exit(ShutterSim s) {
			setEndOpeningTime(Instant.now());
		};

	};

	public final static ShutterInternalState CLOSING = new ShutterInternalState(
			"CLOSING") {
		/**
         * 
         */
		private static final long serialVersionUID = -5859735401510512904L;

		@Override
		public void enter(ShutterSim s) {
			setStartClosingTime(Instant.now());
			scheduleTransition(s, moveTime, CLOSING, CLOSED_ON);
		};

		@Override
		public void exit(ShutterSim s) {
			setEndClosingTime(Instant.now());
		};

	};

	// common behavior to all CLOSED substates

	public static abstract class ClosedState extends ShutterInternalState {
		/**
         * 
         */
		private static final long serialVersionUID = -1639448622856470993L;

		protected ClosedState(String value) {
			super(value);
		}

		@Override
		public ShutterState getPublicState() {
			return ShutterState.CLOSED;
		}

		@Override
		public long getLastEffectiveExposureTime() {
			return getLastTotalOpeningTime()
					- startOpeningTime.until(endOpeningTime, ChronoUnit.MILLIS)
					- startClosingTime.until(endClosingTime, ChronoUnit.MILLIS);
		}

		@Override
		public long getLastTotalOpeningTime() {
			return startOpeningTime.until(endClosingTime, ChronoUnit.MILLIS);
		}
	}

	// CLOSED substates

	public final static ShutterInternalState CLOSED_OFF = new ClosedState(
			"CLOSED_OFF") {
		/**
         * 
         */
		private static final long serialVersionUID = -1129935528677774917L;

		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.NOT_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			return CLOSED_PREP;
		};

		@Override
		public ShutterInternalState expose(ShutterSim s,
				Duration integrationTime) {
			scheduleTransition(s, preparationTime, CLOSED_PREP_EXPOSE, OPENING);
			scheduleTransition(s,
					integrationTime.plus(preparationTime).plus(moveTime), OPEN,
					CLOSING);
			return CLOSED_PREP_EXPOSE;
		}
	};

	public final static ShutterInternalState CLOSED_PREP = new ClosedState(
			"CLOSED_PREP") {
		/**
         * 
         */
		private static final long serialVersionUID = 6020461273102907209L;

		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.GETTING_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			// accepted, do nothing
			return this;
		};

		@Override
		public ShutterInternalState expose(ShutterSim s,
				Duration integrationTime) {
			// to be more accurate, preparationTime should be remaining time
			// before ready
			Duration remaining = Duration.ofMillis(Instant.now().until(
					readyTime, ChronoUnit.MILLIS));
			s.getLogger().info(
					"ShutterSim: expose while preparing, remaining prep time "
							+ remaining.toMillis() + " ms.");
			remaining = remaining.plus(Duration.ofMillis(5));
			scheduleTransition(s, remaining, CLOSED_ON, OPENING); // check possible race? Do not test from? Schedule also from PREP?
			scheduleTransition(s, integrationTime.plus(remaining)
					.plus(moveTime), OPEN, CLOSING);
			return this;
		}

		@Override
		public void enter(ShutterSim s) {
			scheduleTransition(s, preparationTime, CLOSED_PREP, CLOSED_ON);
			readyTime = Instant.now().plus(preparationTime);
			s.getLogger().info("ShutterSim will be ready at " + readyTime);
		};

		private java.time.Instant readyTime;

	};

	public final static ShutterInternalState CLOSED_PREP_EXPOSE = new ClosedState(
			"CLOSED_PREP_EXPOSE") {
		/**
         * 
         */
		private static final long serialVersionUID = -4697199143872341749L;

		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.GETTING_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			// accepted, do nothing
			return this;
		};

	};

	public final static ShutterInternalState CLOSED_ON = new ClosedState(
			"CLOSED_ON") {

		/**
         * 
         */
		private static final long serialVersionUID = 5636147668726170551L;
		private transient ScheduledFuture<ShutterInternalState> powerOffTransition = null;

		@Override
		public ShutterInternalState expose(ShutterSim s,
				Duration integrationTime) {
			scheduleTransition(s, integrationTime.plus(moveTime), OPEN, CLOSING);
			return OPENING;

		}

		@Override
		public synchronized ShutterInternalState prepare(ShutterSim s) {
			// we stay prepared, reset the timer
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = scheduleTransition(s, readyDuration,
					CLOSED_ON, CLOSED_OFF);
			return CLOSED_ON;
		};

		@Override
		public synchronized void enter(ShutterSim s) {
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = scheduleTransition(s, readyDuration,
					CLOSED_ON, CLOSED_OFF);

			if (endClosingTime != null)
				s.getLogger().info(
						"ShutterSim: exposure time "
								+ super.getLastEffectiveExposureTime()
								+ " total " + super.getLastTotalOpeningTime());
		};

		@Override
		public synchronized void exit(ShutterSim s) {
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = null;
		}

	};

}
