package org.lsst.ccs.subsystem.mmm.raftsim;

import java.io.Serializable;
import java.time.Duration;
import java.util.concurrent.ScheduledFuture;

import org.lsst.ccs.subsystem.mmm.data.InvalidStateException;

// State Machine for rafts
// no second level state => implemented with enum

public enum RaftInternalState implements Serializable {

    CLEARING {
        @Override
        public void enter(RaftSim r) {
            scheduleTransition(r, clearingDuration, QUIESCENT);
        }
    },

    QUIESCENT {
        transient private ScheduledFuture<RaftInternalState> clearTransition = null;

        @Override
        public RaftInternalState clear(RaftSim r) {
            return CLEARING;
        }

        @Override
        public RaftInternalState integrate(RaftSim r) {
            return INTEGRATING;
        }

        @Override
        public void enter(RaftSim r) {
            clearTransition = scheduleTransition(r, clearOkDuration, NEEDS_CLEAR);
        }

        @Override
        public void exit(RaftSim r) {
            if (clearTransition != null)
                clearTransition.cancel(false);
            clearTransition = null;
        }
    },

    INTEGRATING {
        @Override
        public RaftInternalState readout(RaftSim r) {
            return READING_OUT;
        }

        @Override
        public RaftInternalState clear(RaftSim r) {
            return CLEARING;
        }

        public RaftInternalState discardRows(RaftSim r) {
            return this;
        }

    },

    READING_OUT {
        @Override
        public void enter(RaftSim r) {
            scheduleTransition(r, readoutTime, QUIESCENT);
        }

    },

    NEEDS_CLEAR {
        @Override
        public RaftInternalState clear(RaftSim r) {
            return CLEARING;
        }
    };

    static final private Duration clearingDuration = Duration.ofMillis(70);
    static final private Duration clearOkDuration = Duration.ofMillis(4000);
    static final private Duration readoutTime = Duration.ofMillis(2000);

    // default implementation of methods : invalid

    public RaftInternalState clear(RaftSim r) {
        r.getLogger().error("invalid clear command, from state " + this);
        throw new InvalidStateException("RAFTS:" + this.toString(), "clear");
    }

    public RaftInternalState integrate(RaftSim r) {
        r.getLogger().error("invalid integrate command, from state " + this);
        throw new InvalidStateException("RAFTS:" + this.toString(), "integrate");
    }

    public RaftInternalState readout(RaftSim r) {
        r.getLogger().error("invalid readout command, from state " + this);
        throw new InvalidStateException("RAFTS:" + this.toString(), "readout");
    }

    public RaftInternalState discardRows(RaftSim r) {
        r.getLogger().error("invalid discardRows command, from state " + this);
        throw new InvalidStateException("RAFTS:" + this.toString(), "readout");
    }

    public void enter(RaftSim r) {
        // default: do nothing
    }

    public void exit(RaftSim r) {
        // default: do nothing
    }

    public static synchronized RaftInternalState scheduled(RaftSim r, RaftInternalState from, RaftInternalState to) {
        // todo check we are still in from?
        r.getLogger().debug(" scheduled: " + to);
        r.setState(to);
        return to;
    }

    public ScheduledFuture<RaftInternalState> scheduleTransition(RaftSim r, Duration delay, RaftInternalState to) {
        return r.scheduleTransition(delay, to);
    }

}
