package org.lsst.ccs.subsystem.mmm;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.regex.Pattern;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.MutedAlertRequest;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusEnum;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.MessagingService;
import org.lsst.ccs.subsystem.mmm.data.MMMIR2Event;
import org.lsst.ccs.utilities.logging.Logger;

public class GenericMMM
        extends Subsystem implements HasLifecycle {

    protected MMMUtilities mu;

    @ConfigurationParameter(isFinal = true, maxLength = 20)
    protected final Map<MinionGroup,Map<String, Minion>> minions = new HashMap<>();

    @LookupField(strategy = Strategy.TOP)
    Subsystem subsystem;

    @LookupField(strategy = Strategy.CHILDREN)
    protected Map<String, AlertDispatcher> dispatchers = new HashMap<>();

    @LookupField(strategy = Strategy.TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = Strategy.TREE)
    private AgentLockService agentLockService;

    private AgentPresenceManager apm;
    
    @Override
    public void init() {
        mu = new MMMUtilities(subsystem);

        for ( Entry<MinionGroup,Map<String,Minion>> e : minions.entrySet() ) {
            MinionGroup group = e.getKey();
            for ( Entry<String,Minion> e1 : e.getValue().entrySet() ) {
                mu.addMinion(group, e1.getValue(), e1.getKey());
            }
        }
        subsystem.getAgentService(AgentCommandDictionaryService.class).addCommandSetToObject(mu, subsystem);
        
        mu.init();

        apm = getAgentService(MessagingService.class).getMessagingAccess().getAgentPresenceManager();    
    }
        
    @Override
    public void start() {
        mu.start();
        mu.addAlertObserver(this::onAlert);
        mu.activate();
    }

    protected MMMUtilities getMmmUtilities() {
        return mu;
    }

    protected Random random = new Random();
    protected static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mmm");

    public GenericMMM() {
        this("genericMMM");
    }
    
    public GenericMMM(String name) {
        super(name,AgentInfo.AgentType.MMM);
    }


    public void onAlert(AlertNotification notif) {
        log.warn(notif+" "+notif.getOrigin()+" "+notif.getSubsystemType()+" "+notif.getSubsystemGroup());
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            AlertDispatcher ad = e.getValue();
            boolean affectsDispatcher = false;

            //Check if the message is from the MMM and decide if its group applies to this dispatcher
            if ( notif.getOrigin().equals(subsystem.getName()) ) {
                String group = (String)notif.getAlert().getAlertData("group");
                if ( ad.getGroupName().equals(group) ) {
                    affectsDispatcher = true;
                }
            }
            
            if ( affectsDispatcher || ad.getGroup().equals(notif.getSubsystemGroup())) {
                e.getValue().onAlert(notif);
            }
        }
    }

    @Command
    public String[] getAlertDispatchers() {
        return dispatchers.keySet().toArray(new String[0]);
    }
    
    @Command
    public String status() {
        StringBuilder sb = new StringBuilder();
        sb.append("Dispatchers: \n");
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            sb.append("\n");
            sb.append(e.getValue().status("\t"));
        }
        return sb.toString();
    }
    

    public void publishEvent(MMMIR2Event e) {
        StatusEnum<MMMIR2Event> message = new StatusEnum<MMMIR2Event>(e, stateService.getState());
        subsystem.getMessagingAccess().sendStatusMessage(message);
    }

    public Object send(MinionGroup g, Minion dst, String command, Object... parms) throws Exception {
        return mu.send(g, dst, command, parms);
    }

    public Object sendLongCommand(MinionGroup g, Minion dst, long duration, String command, Object... parms) throws Exception {
        return mu.sendLongCommand(g, dst, duration, command, parms);
    }

    public Future<Object> sendAsync(MinionGroup g, Minion dst, String command, Object... parms) {
        return mu.sendAsync(g, dst, command, parms);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> Future<StatusMessage> watchForState(MinionGroup g, Minion sys,
            MinionMMMIR2State state) {
        return mu.watchForState(g, sys, state);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void waitForState(MinionGroup g, Minion sys, MinionMMMIR2State state, long timeout) {
        mu.waitForState(g, sys, state, timeout);
    }

    public void waitMillis(long millis) {
        mu.waitMillis(millis);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void checkState(MinionGroup g, Minion sys, MinionMMMIR2State state) {
        mu.checkState(g, sys, state);
    }

    @SafeVarargs
    public final <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void checkState(MinionGroup g, Minion sys, MinionMMMIR2State... state) {
        mu.checkState(g, sys, state);
    }

    public <MMMIR2State extends Enum<MMMIR2State>> MMMUtilities.ExpectedStateCombination<MMMIR2State> expectingState(
            MinionGroup g, Minion m, MMMIR2State state) {
        return mu.expectingState(g, m, state);
    }

    @SafeVarargs
    public final void setAbortingOnAlarmMinions(MinionGroup g, Minion... m) {
        mu.setAbortingOnAlarmMinions(g, m);
    }

    public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
        return mu.schedule(r, delay);
    }

    public Future<?> execute(Runnable r) {
        return mu.execute(r);
    }

    /**
     * Mute all the alerts for a give subsystem over the specified duration in
     * seconds.
     *
     * During the muted period any alerts raised by the given subsystem will not
     * trigger any actions in the MMM. When the time period has elapsed, the
     * mute request will be cancelled.
     *
     * @param subsystem The name of the subsystem for which the alerts are to be
     * muted.
     * @param durationInSeconds The time period over which the alerts will be
     * muted in seconds.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, autoAck = false)
    public void muteAllAlertsForSubsystem(String subsystem, int durationInSeconds) {
        muteAlertsForSubsystem(subsystem, ".*", durationInSeconds);
        
    }

    /**
     * Mute a set the alerts for a give subsystem over the specified duration in
     * seconds by providing a regular expression to match the alert ids to be
     * muted.
     *
     * During the muted period raised alerts matching the provided regular
     * expression will not trigger any actions in the MMM. When the time period
     * has elapsed, the mute request will be cancelled.
     *
     * @param subsystem The name of the subsystem for which the alerts are to be
     * muted.
     * @param alertIdRegEx A regular expression to match the alert ids to be
     * muted
     * @param durationInSeconds The time period over which the alerts will be
     * muted in seconds.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, autoAck = false)
    public void muteAlertsForSubsystem(String subsystem, String alertIdRegEx, int durationInSeconds) {

        boolean canCompileRegEx = true;
        try {
            Pattern p = Pattern.compile(alertIdRegEx);
        } catch (Exception e) {
            canCompileRegEx = false;
        }
        
        helper().precondition(durationInSeconds > 0, "The duration must be positive.")
                .precondition(durationInSeconds <= maximumMuteDuration,"The duration cannot exceed "+maximumMuteDuration+" seconds.")
                .precondition(canLockTargetSubsystem(subsystem), "To mute alerts of a subsystem, the user must be able to lock the target subsystem.")
                .precondition(canCompileRegEx, "The provided regular expression could not be compiled.")
                .action( () -> {
                    
                    String userName = getCommandUser();
                    MutedAlertRequest mar = new MutedAlertRequest(subsystem, alertIdRegEx, durationInSeconds, userName);
                    addNewMutedAlertRequest(mar);

                });
    }
    
    private boolean canLockTargetSubsystem(String subsystem) {
        if ( subsystem == null ) {
            return false;
        }
        //Check if the subsystem is already locked
        AgentLock existingLock = agentLockService.getExistingLockForAgent(subsystem);
        String userName = getCommandUser();
        //The subsystem can be locked if it is either not locked or if it's locked by the same user
        return existingLock == null ? true : existingLock.getOwner().equals(userName);        
    }
    

    /**
     * Cancel an existing active MutedAlertRequest before its natural expiration
     * by providing its unique id.
     *
     * The unique id is assigned when a MutedAlertRequest is created and can be
     * found by getting list List of the currently active MutedAlertRequest
     * instances.
     *
     * @param mutedAlertRequestId The unique id of the MutedAlertRequest to
     * cancel
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL)
    public void cancelMutedAlertRequest(long mutedAlertRequestId) {
        String subsystemName = getSubsystemNameForMutedAlertRequestId(mutedAlertRequestId);
        helper().precondition(subsystemName != null, "The provided unique id is invalid, we could not find an associated MutedAlertRequest to cancel.")
                .precondition(canLockTargetSubsystem(subsystemName), "To mute alerts of a subsystem, the user must be able to lock the target subsystem.")
                .action( () -> {
                    cancelMutedAlertRequestById(mutedAlertRequestId, true);
                });
    }
    
    private String getSubsystemNameForMutedAlertRequestId(long mutedAlertRequestId) {
        for (MutedAlertRequest mar : mutedAlertRequestList) {
            if (mar.getUniqueId() == mutedAlertRequestId ) {
                return mar.getSubsystemName();
            }
        }
        return null;        
    }

    /**
     * Cancel all the active MutedAlertRequest instances for a given subsystem.
     *
     * @param subsystemName The name of the subsystem for which to cancel all
     * the active MutedAlertRequest instances
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL)
    public void cancelAllMutedAlertRequestForSubsystem(String subsystemName) {
        helper().precondition(canLockTargetSubsystem(subsystemName), "To mute alerts of a subsystem, the user must be able to lock the target subsystem.")
                .action( () -> {
                    List<Long> idsToCancel = new ArrayList<>();
                    for (MutedAlertRequest mar : mutedAlertRequestList) {
                        if (mar.getSubsystemName().equals(subsystemName)) {
                            idsToCancel.add(mar.getUniqueId());
                        }
                    }
                    
                    if ( ! idsToCancel.isEmpty() ) {
                        for ( long id : idsToCancel ) {
                            cancelMutedAlertRequestById(id, false);
                        }
                        publishListOfMutedAlerts();
                    }
                    
                });
    }

    /**
     * Get the list of muted alert for the provided subsystem names. If no
     * subsystem names are provided all the muted alerts will be returned.
     *
     * This command is meant to be used by users using a ccs-shell to get the
     * list of currently active MutedAlertRequest instances.
     *
     * @param subsystemName The subsystem name.
     * @return The list of MutedAlert instances for the provided subsystem
     * names.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL)
    public List<MutedAlertRequest> getListOfMutedAlertRequests(String... subsystemName) {
        if ( subsystemName.length == 0 ) {
            return new ArrayList(mutedAlertRequestList);
        }
        ArrayList<MutedAlertRequest> result = new ArrayList();
        List<String> subsystems = Arrays.asList(subsystemName);
        for ( MutedAlertRequest r : mutedAlertRequestList ) {
            if ( subsystems.contains(r.getSubsystemName()) ) {
                result.add(r);
            }
        }
        return result;
    }

    /**
     * Publish the list of all muted alert instances.
     *
     * This command is meant to be used by ccs-consoles to force the publication
     * of the list of currently active MutedAlertRequest instances.
     *
     * The current list of active MutedAlertRequest instances will be
     * spontaneously published every time there is a change to it: a new request
     * is issued or an existing request is either expired or cancelled.
     *
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL)
    public void publishListOfMutedAlerts() {
        KeyValueData data = new KeyValueData(MutedAlertRequest.PUBLICATION_KEY, new ArrayList(mutedAlertRequestList));
        publishSubsystemDataOnStatusBus(data);
    }
    
    

    private int maximumMuteDuration = 3*60*60;

    private final List<MutedAlertRequest> mutedAlertRequestList = new CopyOnWriteArrayList<>();
    
    private void addNewMutedAlertRequest(MutedAlertRequest mar) {
        
        getScheduler().schedule(
                () -> {
                    cancelMutedAlertRequestById(mar.getUniqueId(), true);
                }, 
                mar.getDurationInSeconds(), TimeUnit.SECONDS);
        
        log.info("Added MutedAlertRequest "+mar);
        mutedAlertRequestList.add(mar);
        publishListOfMutedAlerts();
        
    }


    private void cancelMutedAlertRequestById(long id, boolean publish) {

        Iterator<MutedAlertRequest> iter = mutedAlertRequestList.iterator();
        List<MutedAlertRequest> toRemove = new ArrayList();

        while(iter.hasNext()) {
            MutedAlertRequest mar = iter.next();
            if ( mar.getUniqueId() == id ) {
                toRemove.add(mar);
                break;
            }
        }        
        if ( ! toRemove.isEmpty() ) {
            mutedAlertRequestList.removeAll(toRemove);
            log.info("Cancelled MutedAlertRequest "+toRemove.get(0));
            if ( publish ) {
                publishListOfMutedAlerts();
            }         
        }            
    }
        
    /**
     * Check if a give subsystem Alert is muted.
     * 
     * @param subsystemName
     * @param alert
     * @return true if the provided subsystem alert is muted.
     */
    public boolean isSubsystemAlertMuted(String subsystemName, Alert alert) {
        for ( MutedAlertRequest mar : mutedAlertRequestList ) {
            if ( mar.getSubsystemName().equals(subsystemName) ) {
                if ( mar.isSubsystemAlertMuted(subsystemName, alert) ) {
                    return true;
                }
            }
        }
        return false;
    }
    
}
