package org.lsst.ccs.subsystem.mmm;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.subsystem.mmm.AlertDispatcher.AlertAction;

/**
 * An AlertAction that sends notification to Squadcast.
 *
 * @author The LSST CCS Team
 */
public class SquadcastAlertAction extends AlertAction {

    @ConfigurationParameter(isFinal = true)
    public volatile String payloadUrl = "";
    
    @Persist
    public final List<String> triggeredIncidents = new CopyOnWriteArrayList<>();
    
    private URL url;
    
    @Override
    public void init() {
        super.init();
        try {
            url = new URL(payloadUrl);
        } catch (Exception e) {
            throw new RuntimeException("Failed to initialize Squadcast AlertAction",e);
        }
    }

    @Override
    public String getDescription() {
        return "Squadcast AlertAction";
    }
    
    @Override
    public void accept(AlertNotification notif) {
        SquadcastIncident incident = new SquadcastIncident(notif);
        //Post an Incident to squadcast only if it's an ALARM or of it's NOMINAL and there is already an outstanding triggered incident
        boolean postIncident = false;
        switch(notif.getSeverity()) {
            case ALARM:
                triggeredIncidents.add(incident.alertId);
                postIncident = true;
                break;
            case NOMINAL:
                postIncident = triggeredIncidents.remove(incident.alertId);
                break;
        }
        
        if ( postIncident ) {        
            try {
                HttpURLConnection con = (HttpURLConnection) url.openConnection();
                con.setRequestMethod("POST");
                con.setRequestProperty("Content-Type", "application/json");
                con.setRequestProperty("Accept", "application/json");
                con.setDoOutput(true);
                try ( OutputStream os = con.getOutputStream()) {
                    ObjectMapper mapper = new ObjectMapper();
                    byte[] input = mapper.writeValueAsString(incident).getBytes("utf-8");
                    os.write(input, 0, input.length);
                } catch (JsonProcessingException jpe) {
                    getLogger().log(Level.SEVERE, "Failed to post incident to squadcast", jpe);
                    jpe.printStackTrace();
                }

                StringBuilder content;
                try ( BufferedReader br = new BufferedReader(
                        new InputStreamReader(con.getInputStream()))) {
                    String line;
                    content = new StringBuilder();
                    while ((line = br.readLine()) != null) {
                        content.append(line);
                        content.append(System.lineSeparator());
                    }
                }
                getLogger().log(Level.FINE, "Got response from squadcast", content.toString());

            } catch (Exception e) {
                getLogger().log(Level.SEVERE, "Failed to create connection to squadcast", e);
            }
        }
    }
    
    
    
    static class SquadcastIncident {

        @JsonProperty("event_id")
        public final String alertId;
        public final String message, description, status;

        public final Map<String, Object> tags = new LinkedHashMap();

        SquadcastIncident(AlertNotification notif) {
            alertId = notif.getOrigin() + "/" + notif.getAlert().getAlertId();
            message = notif.getCause();
            description = notif.getAlert().getDescription();
            status = notif.getSeverity() == AlertState.NOMINAL ? "resolve" : "trigger";
            tags.put("alertId", alertId);
            tags.put("severity", new SeverityWithColor(notif.getSeverity()));
            tags.put("group", notif.getSubsystemGroup().name());
            tags.put("cluster", "ir2");
        }
        
    }

    
    static class SeverityWithColor {
        
        public final String value;
        public final String color;
        
        SeverityWithColor(AlertState state) {
            value = state.name();
            switch (state) {
                case ALARM:
                    color = "#FD1802";
                    break;
                case WARNING:
                    color = "#FDE602";
                    break;
                case NOMINAL:
                    color = "#2FD116";
                    break;
                default:
                    color = "";
            }
        }
        
        
    }
    
    
}
