package org.lsst.ccs.subsystem.mmm;

import java.util.Map;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeUtils;

/**
 *
 * An Alert Predicate to filter on alerts
 * 
 * @author The LSST CCS Team
 */
public class AlertPredicate  implements Predicate<Alert> {

    private final Map<String, String> properties;
    private Pattern originPattern;
    private Pattern idPattern;
    private Pattern causePattern;
    private Pattern severityPattern;

    public AlertPredicate(String propertiesStr) {
        properties = (Map) InputConversionEngine.convertArgToType(propertiesStr, Map.class);
        updatePatterns();
    }

    public AlertPredicate(Map<String, String> properties) {
        this.properties = properties;
        updatePatterns();
    }

    private void updatePatterns() {
        for (Map.Entry<String, String> e : properties.entrySet()) {
            String property = e.getKey();
            String propertyValue = e.getValue();
            Pattern p = Pattern.compile(propertyValue);
            switch(property) {
                case "cause" :   
                    causePattern = p;
                    break;
                case "id" :
                    idPattern = p;
                    break;
                case "origin" :
                    originPattern = p;
                    break;
                case "severity" :
                    severityPattern = p;
                    break;
                default:
                    throw new IllegalArgumentException("Unsupported predicate property: "+property);
            }
        }
    }
    
    @Override
    public boolean test(Alert alert) {
        if ( idPattern != null ) {
            if ( idPattern.matcher(alert.getAlertId()).matches() ) {
                return true;
            }
            return false;
        }
        return true;
    }
    
    public boolean testOrigin(String origin) {
        if ( originPattern != null ) {
            if ( originPattern.matcher(origin).matches() ) {
                return true;
            }
            return false;
        }
        return true;        
    }

    public boolean testCause(String cause) {
        if ( causePattern != null ) {
            return causePattern.matcher(cause).matches();
        }
        return true;        
    }

    public boolean testSeverity(String severity) {
        if ( severityPattern != null ) {
            return severityPattern.matcher(severity).matches();
        }
        return true;        
    }

    @Override
    public String toString() {
        return TypeUtils.stringify(properties);
    }
    
}



