package org.lsst.ccs.subsystem.metrology;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import javax.usb.UsbException;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.ad.Ad7747Eval;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to record capacitance data using Analog Devices 7747
 **  evaluation boards
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class CapTest {

   /**
    ***************************************************************************
    **
    **  Fields.
    **
    ***************************************************************************
    */
    /*
    **  Constants
    */
    private final static int
        CHAN_CAP  = 0,
        CHAN_TEMP = 1,
        CHAN_VOLT = 2,
        CHAN_VDD  = 3,
        MONITOR_PERIOD = 500,
        SEL_NONE = 0x100,
        SEL_ALL  = 0x101,
        SEP_ON   = 1,
        SEP_OFF  = 0,
        SHOW_SAMPLE   = 0x01,
        SHOW_RATE     = 0x02,
        SHOW_SELECT   = 0x04,
        SHOW_SEPARATE = 0x08,
        SHOW_PAUSE    = 0x10,
        SHOW_CYCLE    = 0x20;

    private final static String DEFAULT_NAME = "Captest";

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup rateNames;
    static {
        rateNames = new CmndProcess.Lookup(8);
        rateNames.add("45.5", Ad7747Eval.CAPFS_RATE_45);
        rateNames.add("41.9", Ad7747Eval.CAPFS_RATE_42);
        rateNames.add("25.0", Ad7747Eval.CAPFS_RATE_25);
        rateNames.add("13.2", Ad7747Eval.CAPFS_RATE_13);
        rateNames.add("8.1",  Ad7747Eval.CAPFS_RATE_8);
        rateNames.add("6.5",  Ad7747Eval.CAPFS_RATE_7);
        rateNames.add("5.5",  Ad7747Eval.CAPFS_RATE_6);
        rateNames.add("4.6",  Ad7747Eval.CAPFS_RATE_5);
    }

    private final static CmndProcess.Lookup chanNames;
    static {
        chanNames = new CmndProcess.Lookup(4);
        chanNames.add("capacitance", CHAN_CAP);
        chanNames.add("temperature", CHAN_TEMP);
        chanNames.add("voltage",     CHAN_VOLT);
        chanNames.add("vdd",         CHAN_VDD);
    }

    private final static CmndProcess.Lookup selNames;
    static {
        selNames = new CmndProcess.Lookup(2);
        selNames.add("all",  SEL_ALL);
        selNames.add("none", SEL_NONE);
    }

    private final static CmndProcess.Lookup sepNames;
    static {
        sepNames = new CmndProcess.Lookup(2);
        sepNames.add("on",  SEP_ON);
        sepNames.add("off", SEP_OFF);
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess topProc = new CmndProcess();
    private final CmndProcess rptProc = new CmndProcess(true);
    private final MonitorData monData = new MonitorData();
    private final Timer monTimer = new Timer();
    private ConsoleReader reader;
    private int idMask, daqChan, daqNSamp, nEvent, nToss;
    private int nSamp = 1, rate = Ad7747Eval.CAPFS_RATE_5, nCycle = 1, actvMask;
    private boolean sepDaq = true;
    private long pause = 0;
    private ArrayList<Ad7747Eval> boards = new ArrayList<Ad7747Eval>();
    private ArrayBlockingQueue<Integer> doneQueue, rptConsQueue, monConsQueue;
    private int[] id;
    private boolean debug, runActive, monActive, stopMon, repeat;
    private DaqThread[] daq;
    private String logName = DEFAULT_NAME, comment = "";
    private PrintStream logFile;

   /**
    ***************************************************************************
    **
    **  Inner class for running the DAQ on a board.
    **
    ***************************************************************************
    */
    class DaqThread extends Thread {

        private final int index;
        private final Ad7747Eval ad;
        private final ArrayBlockingQueue<Integer> startQueue;
        private int nvalue, nexcptn;
        private double value, mean, sigma, error, drift, temp;
        private Exception excptn;


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public DaqThread(int index, Ad7747Eval ad)
        {
            this.index = index;
            this.ad = ad;
            startQueue = new ArrayBlockingQueue<Integer>(1);
        }


       /**
        ***********************************************************************
        **
        **  Clears the current value
        **
        ***********************************************************************
        */
        public void clearValue()
        {
            value = 0;
        }


       /**
        ***********************************************************************
        **
        **  Gets the sample count
        **
        ***********************************************************************
        */
        public int getSampleCount()
        {
            return nvalue;
        }


       /**
        ***********************************************************************
        **
        **  Gets the current value
        **
        ***********************************************************************
        */
        public double getValue()
        {
            return value;
        }


       /**
        ***********************************************************************
        **
        **  Gets the calculated mean
        **
        ***********************************************************************
        */
        public double getMean()
        {
            return mean;
        }


       /**
        ***********************************************************************
        **
        **  Gets the calculated sigma
        **
        ***********************************************************************
        */
        public double getSigma()
        {
            return sigma;
        }


       /**
        ***********************************************************************
        **
        **  Gets the calculated error
        **
        ***********************************************************************
        */
        public double getError()
        {
            return error;
        }


       /**
        ***********************************************************************
        **
        **  Gets the calculated drift
        **
        ***********************************************************************
        */
        public double getDrift()
        {
            return drift;
        }


       /**
        ***********************************************************************
        **
        **  Gets the measured temperature
        **
        ***********************************************************************
        */
        public double getTemperature()
        {
            return temp;
        }


       /**
        ***********************************************************************
        **
        **  Gets the USB exception count
        **
        ***********************************************************************
        */
        public int getExceptionCount()
        {
            return nexcptn;
        }


       /**
        ***********************************************************************
        **
        **  Gets the first USB exception
        **
        ***********************************************************************
        */
        public Exception getException()
        {
            return excptn;
        }


       /**
        ***********************************************************************
        **
        **  Starts a run
        **
        ***********************************************************************
        */
        public void startRun()
        {
            startQueue.offer(0);
        }


       /**
        ***********************************************************************
        **
        **  Runs the DAQ
        **
        ***********************************************************************
        */
        @Override
        public void run()
        {
            while (true) {
                while (true) {
                    try {
                        startQueue.take();
                        break;
                    }
                    catch (InterruptedException e) {
                        continue;
                    }
                }
                double vsum = 0, vsqsum = 0, tsum = 0, tsqsum = 0, vtsum = 0,
                       temp0 = 0;
                long millis0 = 0;
                nvalue = nexcptn = 0;

                if (runActive) {
                    try {
                        temp0 = readTemperature(ad);
                    }
                    catch (UsbException e) {
                        if (nexcptn++ == 0) excptn = e;
                    }
                }

                for (int j = -nToss; j < daqNSamp; j++) {
                    try {
                        double time;
                        long millis;
                        if (daqChan == CHAN_CAP) {
                            value = ad.readCapacitance(0);
                        }
                        else if (daqChan == CHAN_TEMP) {
                            value = ad.readTemperature(0);
                        }
                        else if (daqChan == CHAN_VOLT) {
                            value = ad.readVoltage(Ad7747Eval.RDO_EXTERN);
                        }
                        else {
                            value = ad.readVoltage(0);
                        }
                        if (j < 0) continue;   // Ignore initial (stale) values
                        millis = System.currentTimeMillis();
                        if (j == 0) {
                            millis0 = millis;
                        }
                        time = (millis - millis0) / 1000.0;
                        vsum += value;
                        vsqsum += value * value;
                        tsum += time;
                        tsqsum += time * time;
                        vtsum += value * time;
                        nvalue++;
                    }
                    catch (UsbException e) {
                        if (nexcptn++ == 0) excptn = e;
                    }
                }

                if (runActive) {
                    try {
                        temp = readTemperature(ad);
                    }
                    catch (UsbException e) {
                        if (nexcptn++ == 0) excptn = e;
                    }
                    temp = 0.5 * (temp + temp0);
                }

                if (nvalue > 0) {
                    double avg = vsum / nvalue;
                    mean = avg;
                    sigma = Math.sqrt(vsqsum / nvalue - avg * avg);
                    error = sigma / nvalue;
                }
                else {
                    mean = 0.0;
                    sigma = 0.0;
                    error = 0.0;
                }
                if (nvalue > 1) {
                    drift = (nvalue * vtsum - vsum * tsum)
                              / (nvalue * tsqsum - tsum * tsum);
                }
                else {
                    drift = 0.0;
                }
                doneQueue.offer(index);
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class running a timer task to monitor current DAQ data.
    **
    ***************************************************************************
    */
    class MonitorData extends TimerTask
    {
        @Override
        public void run()
        {
            if (!monActive || stopMon) return;
            out.print("\r");
            for (int j = 0; j < boards.size(); j++) {
                if ((actvMask & (1 << j)) != 0) {
                    out.format("%s %.5g  ", id[j], daq[j].getValue());
                }
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class for processing console input while repeating.
    **
    ***************************************************************************
    */
    class RptConsThread extends Thread {

        @Override
        public void run()
        {
            while (true) {
                take(rptConsQueue);
                while (repeat) {
                    awaitTerminal();
                    if (!repeat) continue;
                    clearTerminal();
                    stopMon = true;
                    out.println();
                    String line = readLine("*> ");
                    if (line == null || !rptProc.process(line)) {
                        repeat = false;
                    }
                    stopMon = false;
                }
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class for detecting console input while monitoring.
    **
    ***************************************************************************
    */
    class MonConsThread extends Thread {

        @Override
        public void run()
        {
            while (true) {
                take(monConsQueue);
                awaitTerminal();
                clearTerminal();
                stopMon = true;
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class for handling top-level commands.
    **
    ***************************************************************************
    */
    class TopCmnd implements CmndProcess.CmndDisp {

        /*
        **  Top-level command codes
        */
        private final static int
            CMD_SAMPLE   = 0,
            CMD_RATE     = 1,
            CMD_GO       = 2,
            CMD_LOG      = 3,
            CMD_DAC      = 4,
            CMD_MONITOR  = 5,
            CMD_LOAD     = 6,
            CMD_SELECT   = 7,
            CMD_SEPARATE = 8,
            CMD_CYCLE    = 9,
            CMD_PAUSE    = 10,
            CMD_SHOW     = 11,
            NUM_CMDS     = 12;

        /*
        **  Top-level command help text
        */
        private final String[] helpSample = {
            "Show or set the number of samples to be read each DAQ cycle",
            "sample [<count>]",
            "count  The number of samples to set",
        };

        private final String[] helpRate = {
            "Show or set the nominal rate at which samples are acquired",
            "rate [<value>]",
            "value  The rate to set (45.5, 41.9, 25.0, 13.2, 8.1, 6.5, 5.5 or 4.6)",
        };

        private final String[] helpSelect = {
            "Show or select the boards to be used for DAQ",
            "select [<id1> | all | none] [<id2>]... [<id10>]",
            "idn  The id of a board to be used"
        };

        private final String[] helpSeparate = {
            "Show or set separate DAQ mode",
            "separate [<onoff>]",
            "onoff  If 'on', DAQ is performed separately for each board",
            "       If 'off', DAQ is performed on all selected boards simultaneously",
        };

        private final String[] helpCycle = {
            "Set or show the number of DAQ cycles to perform",
            "cycle [<count>]",
            "count   The number of times to perform a cycle (0 = no limit)",
        };

        private final String[] helpPause = {
            "Set or show the time to pause between DAQ cycles",
            "pause [<time>]",
            "time   The time, in seconds, to pause between cycles",
        };

        private final String[] helpGo = {
            "Perform one or more data acquisition cycles",
            "go [<comment>]",
            "comment  A comment to be logged at the end of each cycle",
        };

        private final String[] helpLog = {
            "Close the log file, and optionally set the name for the next one",
            "log  [<name>]",
            "name  The new stem name of the log file: a time stamp is appended",
        };

        private final String[] helpShow = {
            "Display the values of all the parameters",
            "show",
        };

        private final String[] helpDac = {
            "Show or set the state of the capacitance offset DACs",
            "dac <id> [<dac>] [<enab>] [<value>]",
            "id     The ID of the board to use, or -1 for all",   
            "dac    The DAC to use, a or b (default is both when showing)",
            "enab   Enable the DAC if non-zero, disable it if zero",
            "value  The value to set, 0 - 63: 1 unit ~= 0.28 pF",
        };

        private final String[] helpMonitor = {
            "Continuously monitor a sensor on each selected board",
            "monitor [<sensor>]",
            "sensor  The sensor to read: capacitance, temperature, voltage or vdd",
        };

        private final String[] helpLoad = {
            "Load a board with its firmware",
            "load <board>",
            "board  The index of the board to be loaded",
        };

        private CmndProcess.Command cmnd = new CmndProcess.Command(NUM_CMDS);


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public TopCmnd()
        {
            cmnd.add("sample",   CMD_SAMPLE,   helpSample,   "i");
            cmnd.add("rate",     CMD_RATE,     helpRate,     "k",        rateNames);
            cmnd.add("select",   CMD_SELECT,   helpSelect,   "jiiiiiii", selNames);
            cmnd.add("separate", CMD_SEPARATE, helpSeparate, "k",        sepNames);
            cmnd.add("cycle",    CMD_CYCLE,    helpCycle,    "i");
            cmnd.add("pause",    CMD_PAUSE,    helpPause,    "f");
            cmnd.add("go",       CMD_GO,       helpGo,       "e");
            cmnd.add("log",      CMD_LOG,      helpLog,      "s");
            cmnd.add("show",     CMD_SHOW,     helpShow,     "");
            cmnd.add("dac",      CMD_DAC,      helpDac,      "Isii");
            cmnd.add("monitor",  CMD_MONITOR,  helpMonitor,  "k",        chanNames);
            cmnd.add("load",     CMD_LOAD,     helpLoad,     "Is");
        }


       /**
        ***********************************************************************
        **
        **  Returns the top-level command table.
        **
        ***********************************************************************
        */
        @Override
        public CmndProcess.Command getCommand()
        {
            return cmnd;
        }


        /**
        ***********************************************************************
        **
        **  Dispatches top-level command for processing.
        **
        ***********************************************************************
        */
        @Override
        public boolean dispatch(int code, int found, Object[] args)
        {
            try {
                switch (code) {
                case CMD_GO:
                    procGo(found, args); break;
                case CMD_MONITOR:
                    procMonitor(found, args); break;
                case CMD_SAMPLE:
                    procSample(found, args); break;
                case CMD_RATE:
                    procRate(found, args); break;
                case CMD_SELECT:
                    procSelect(found, args); break;
                case CMD_SEPARATE:
                    procSeparate(found, args); break;
                case CMD_CYCLE:
                    procCycle(found, args); break;
                case CMD_PAUSE:
                    procPause(found, args); break;
                case CMD_SHOW:
                    procShow(found, args); break;
                case CMD_DAC:
                    procDac(found, args); break;
                case CMD_LOG:
                    procLog(found, args); break;
                case CMD_LOAD:
                    procLoad(found, args); break;
                default:
                    out.println("Command not fully implemented");
                }
            }
            catch (UsbException e) {
                out.println(e);
            }

            return true;
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class for handling repeat sub-commands.
    **
    ***************************************************************************
    */
    class RptCmnd implements CmndProcess.CmndDisp {

        /*
        **  Repeat sub-command codes
        */
        private final static int
            CMD_STOP     = 0,
            CMD_COMMENT  = 1,
            NUM_CMDS     = 2;

        /*
        **  Repeat sub-command help text
        */
        private final String[] helpStop = {
            "Stop repeating DAQ cycles",
            "stop",
        };

        private final String[] helpComment = {
            "Set the comment for the current DAQ cycle",
            "comment <text>",
            "text  The text of the comment",
        };

        private CmndProcess.Command cmnd = new CmndProcess.Command(NUM_CMDS);


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public RptCmnd()
        {
            cmnd.add("stop",     CMD_STOP,     helpStop,     "e");
            cmnd.add("comment",  CMD_COMMENT,  helpComment,  "e");
        }


       /**
        ***********************************************************************
        **
        **  Returns the repeat sub-command table.
        **
        ***********************************************************************
        */
        @Override
        public CmndProcess.Command getCommand()
        {
            return cmnd;
        }


       /**
        ***********************************************************************
        **
        **  Dispatches repeat sub-command for processing.
        **
        ***********************************************************************
        */
        @Override
        public boolean dispatch(int code, int found, Object[] args)
        {
            switch (code) {

            case CMD_STOP:
                return false;

            case CMD_COMMENT:
                if ((found & 0x01) != 0) {
                    comment = (String)args[0];
                }

            }

            return true;
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public CapTest(int nSamp, int rate, boolean debug)
    {
        this.nSamp = nSamp;
        this.rate = rate;
        this.debug = debug;
        topProc.add(new TopCmnd());
        rptProc.add(new RptCmnd());
    }


   /**
    ***************************************************************************
    **
    **  Main program.
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int nSamp = (args.length >= 1) ? Integer.decode(args[0]) : 1;
        if (nSamp <= 0) {
            nSamp = 1;
        }
        int rate = (args.length >= 2) ? rateNames.encode(args[1], true) : 0;
        if (rate < 0) {
            rate = 0;
        }
        boolean debug = (args.length >= 3);

        try {
            (new CapTest(nSamp, rate, debug)).run();
        }
        catch (Exception e) {
            out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the DAQ program.
    **
    ***************************************************************************
    */
    public void run() throws Exception
    {
        /*
        **  Create console
        */
        reader = new ConsoleReader();

        /*
        **  Find attached boards
        */
        for (int j = 0; true; j++) {
            try {
                boards.add(new Ad7747Eval(j));
            }
            catch (UsbException e) {
                break;
            }
        }
        int nBoard = boards.size();
        if (nBoard == 0) {
            out.println("No AD7747 boards found");
            return;
        }
        out.println(nBoard + " AD7747 board" + (nBoard == 1 ? "" : "s")
                      + " found");
        out.format("Sample count = %s, rate = %s Hz\n", nSamp,
                   rateNames.decode(rate));

        /*
        **  Initialize boards
        */
        id = new int[nBoard];
        daq = new DaqThread[nBoard];
        actvMask = (1 << nBoard) - 1;
        doneQueue = new ArrayBlockingQueue<Integer>(nBoard);

        for (int j = 0; j < nBoard; j++) {
            Ad7747Eval ad = boards.get(j);
            ad.load(false);
            ad.setupStandard();
            ad.setExcEnabled(false);
            ad.setVtEnabled(true);
            id[j] = readBoardId(ad);
            ad.setConvMode(Ad7747Eval.MODE_CONT);
            (daq[j] = new DaqThread(j, ad)).start();
        }
        out.print("Board IDs:");
        for (int j = 0; j < nBoard; j++) {
            out.format(" %s", id[j]);
            idMask |= 1 << id[j];
        }
        out.println();

        /*
        **  Start console and monitor threads
        */
        rptConsQueue = new ArrayBlockingQueue<Integer>(1);
        monConsQueue = new ArrayBlockingQueue<Integer>(1);
        (new RptConsThread()).start();
        (new MonConsThread()).start();
        monTimer.schedule(monData, 0, MONITOR_PERIOD);

        /*
        **  Command processing
        */
        while (true) {
            String line = readLine(">> ");
            if (line == null || !topProc.process(line)) break;
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SAMPLE command
    **
    ***************************************************************************
    */
    private void procSample(int found, Object[] args)
    {
        if (found == 0) {
            doShow(SHOW_SAMPLE);
        }
        else {
            int value = (Integer)args[0];
            if (value > 0) {
                nSamp = value;
            }
            else {
                out.println("Invalid sample count");
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the RATE command
    **
    ***************************************************************************
    */
    private void procRate(int found, Object[] args) throws UsbException
    {
        if (found == 0) {
            doShow(SHOW_RATE);
        }
        else {
            rate = (Integer)args[0];
            for (int j = 0; j < boards.size(); j++) {
                boards.get(j).setCapConvRate(rate);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SELECT command
    **
    ***************************************************************************
    */
    private void procSelect(int found, Object[] args)
    {
        if (found == 0) {
            doShow(SHOW_SELECT);
        }
        else {
            int mask = 0;
            for (int j = 0; j < 16; j++) {
                if ((found & (1 << j)) == 0) continue;
                int value = (Integer)args[j];
                if (j == 0 && (value == SEL_ALL || value == SEL_NONE)) {
                    if ((found & ~0x01) != 0) {
                        out.println("Too many arguments");
                        mask = -1;
                    }
                    else if (value == SEL_ALL) {
                        mask = (1 << boards.size()) - 1;
                    }
                    else {
                        mask = 0;
                    }
                    break;
                }
                if ((idMask & (1 << value)) == 0) {
                    out.println("Invalid board ID: " + value);
                    mask = -1;
                    break;
                }
                mask |= (1 << getBoardIndex(value));
            }
            if (mask >= 0) {
                actvMask = mask;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SEPARATE command
    **
    ***************************************************************************
    */
    private void procSeparate(int found, Object[] args)
    {
        if (found == 0) {
            doShow(SHOW_SEPARATE);
        }
        else {
            sepDaq = (Integer)args[0] == SEP_ON;
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the PAUSE command
    **
    ***************************************************************************
    */
    private void procPause(int found, Object[] args)
    {
        if (found == 0) {
            doShow(SHOW_PAUSE);
        }
        else {
            pause = (long)(1000 * (Float)args[0]);
            if (pause < 0) {
                pause = 0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the CYCLE command
    **
    ***************************************************************************
    */
    private void procCycle(int found, Object[] args)
    {
        if (found == 0) {
            doShow(SHOW_CYCLE);
        }
        else {
            nCycle = (Integer)args[0];
            if (nCycle < 0) nCycle = 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOW command
    **
    ***************************************************************************
    */
    private void procShow(int found, Object[] args)
    {
        doShow(-1);
    }


   /**
    ***************************************************************************
    **
    **  Processes the GO command
    **
    ***************************************************************************
    */
    private void procGo(int found, Object[] args) throws UsbException
    {
        comment = (found != 0) ? (String)args[0] : "";
        stopMon = false;
        runActive = true;

        if (nCycle == 1) {
            doCycle();
        }

        else {
            repeat = true;
            rptConsQueue.offer(0);
            int cycleNum = 0;
            while (repeat) {
                out.format("Starting cycle %s...\n", ++cycleNum);
                doCycle();
                if (repeat && pause != 0) {
                    out.format("Pausing for %s seconds...\n", pause / 1000.0);
                    try {
                        Thread.sleep(pause);
                    }
                    catch (InterruptedException e) {
                    }
                }
                if (cycleNum == nCycle) {
                    repeat = false;
                }
            }
        }

        runActive = false;
    }


   /**
    ***************************************************************************
    **
    **  Processes the MONITOR command
    **
    ***************************************************************************
    */
    private void procMonitor(int found, Object[] args) throws UsbException
    {
        int chan = (found & 0x01) != 0 ? (Integer)args[0] : CHAN_CAP;
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            Ad7747Eval ad = boards.get(j);
            if (chan == CHAN_CAP) {
                if (!sepDaq) {
                    ad.setExcEnabled(true);
                    ad.setCapConvRate(Ad7747Eval.CAPFS_RATE_5);
                }
                else {
                    ad.setCapConvRate(rate);
                }
                ad.setCapEnabled(true);
                ad.setVtEnabled(false);
            }
            else {
                ad.setCapEnabled(false);
                ad.setVtEnabled(true);
                ad.setVtConvRate(Ad7747Eval.VTFS_RATE_8);
                int mode;
                if (chan == CHAN_TEMP) {
                    mode = Ad7747Eval.VTMD_INT_TEMP;
                }
                else if (chan == CHAN_VOLT) {
                    mode = Ad7747Eval.VTMD_EXT_VOLT;
                }
                else {
                    mode = Ad7747Eval.VTMD_VDD_MON;
                }
                ad.setVtMode(mode);
            }
        }
        out.println("Monitoring " + chanNames.decode(chan)
                    + ".  Press any key to stop.");
        stopMon = false;
        monConsQueue.offer(0);
        int[] nexcpt = new int[boards.size()];
        while (!stopMon) {
            if (chan != CHAN_CAP || !sepDaq) {
                nToss = 0;
                doDaq(chan, 1);
            }
            else {
                nToss = 2;
                for (int j = 0; j < boards.size(); j++) {
                    Ad7747Eval ad = boards.get(j);
                    ad.setExcEnabled(true);
                    doDaq(chan, 1, j);
                    ad.setExcEnabled(false);
                }
            }
            if (stopMon) break;
            out.print("\r");
            for (int j = 0; j < boards.size(); j++) {
                if ((actvMask & (1 << j)) == 0) continue;
                DaqThread acq = daq[j];
                out.format("%s %.5g  ", id[j], acq.getMean());
                nexcpt[j] += acq.getExceptionCount();
            }
            if (stopMon) {
                out.println();
            }
        }
        out.println();
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            if (chan == CHAN_CAP && !sepDaq) {
                boards.get(j).setExcEnabled(false);
            }
            if (nexcpt[j] != 0) {
                out.format("Board %s had %s USB exceptions.  First was: %s\n",
                           id[j], nexcpt[j], daq[j].getException());
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOG command
    **
    ***************************************************************************
    */
    private void procLog(int found, Object[] args)
    {
        if ((found & 0x01) != 0) {
            logName = (String)args[0];
        }
        if (nEvent != 0) {
            logFile.close();
            nEvent = 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAC command
    **
    ***************************************************************************
    */
    private void procDac(int found, Object[] args) throws UsbException
    {
        int index = getBoardIndex((Integer)args[0]);
        if (index == -2) return;
        int first = (index == -1) ? 0 : index;
        int last = (index == -1) ? id.length : index + 1;
        int dacs = 3;
        if ((found & 0x02) != 0) {
            String dac = (String)args[1];
            if (dac.equalsIgnoreCase("a")) {
                dacs = 1;
            }
            else if (dac.equalsIgnoreCase("b")) {
                dacs = 2;
            }
            else {
                out.println("Invalid DAC id");
                return;
            }
        }
        if ((found & 0x0c) == 0) {
            for (int j = first; j < last; j++) {
                Ad7747Eval ad = boards.get(j);
                out.print("ID " + id[j] + ":");
                if ((dacs & 1) != 0) {
                    out.print("  DAC A: "
                              + (ad.isDacAEnabled() ? "En" : "Dis")
                              + "abled, value = " + ad.getDacAValue());
                }
                if ((dacs & 2) != 0) {
                    out.print("  DAC B: "
                              + (ad.isDacBEnabled() ? "En" : "Dis")
                              + "abled, value = " + ad.getDacBValue());
                }
                out.println();
            }
        }
        else {
            if (dacs == 3) {
                out.println("DAC id not specified");
                return;
            }
            for (int j = first; j < last; j++) {
                Ad7747Eval ad = boards.get(j);
                if ((found & 0x04) != 0) {
                    boolean enable = (Integer)args[2] != 0;
                    if (dacs == 1) {
                        ad.setDacAEnabled(enable);
                    }
                    else {
                        ad.setDacBEnabled(enable);
                    }
                }
                if ((found & 0x08) != 0) {
                    int value = (Integer)args[3];
                    if (dacs == 1) {
                        ad.setDacAValue(value);
                    }
                    else {
                        ad.setDacBValue(value);
                    }
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOAD command
    **
    ***************************************************************************
    */
    private void procLoad(int found, Object[] args) throws UsbException
    {
        int index = (Integer)args[0];
        Ad7747Eval ad = boards.get(index);
        if ((found & 0x02) == 0) {
            ad.load(false);
        }
        else {
            ad.load((String)args[1], false);
        }
        ad.setupStandard();
        ad.setVtEnabled(true);
        id[index] = readBoardId(ad);
    }


   /**
    ***************************************************************************
    **
    **  Shows parameter values
    **
    ***************************************************************************
    */
    private void doShow(int mask)
    {
        if ((mask & SHOW_SAMPLE) != 0) {
            out.format("Sample count: %s\n", nSamp);
        }
        if ((mask & SHOW_RATE) != 0) {
            out.format("Sampling rate: %s Hz\n", rateNames.decode(rate));
        }
        if ((mask & SHOW_SELECT) != 0) {
            out.print("Selected boards:");
            if (actvMask == 0) {
                out.format(" %s\n", selNames.decode(SEL_NONE));
            }
            else {
                for (int j = 0; j < boards.size(); j++) {
                    if ((actvMask & (1 << j)) != 0) {
                        out.format(" %s", id[j]);
                    }
                }
                if (actvMask == (1 << boards.size()) - 1) {
                    out.format(" (%s)", selNames.decode(SEL_ALL));
                }
                out.println();
            }
        }
        if ((mask & SHOW_SEPARATE) != 0) {
            out.format("Separate DAQ: %s\n",
                       sepNames.decode(sepDaq ? SEP_ON : SEP_OFF));
        }
        if ((mask & SHOW_PAUSE) != 0) {
            out.format("Pause time: %s secs\n", pause / 1000.0);
        }
        if ((mask & SHOW_CYCLE) != 0) {
            out.format("Cycle count: %s%s\n", nCycle,
                       nCycle == 0 ? " (unlimited)" : "");
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets up and performs a DAQ cycle
    **
    ***************************************************************************
    */
    private void doCycle() throws UsbException
    {
        for (int j = 0; j < boards.size(); j++) {
            daq[j].clearValue();
        }
        monActive = true;
        nToss = 2;
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            Ad7747Eval ad = boards.get(j);
            ad.setExcEnabled(true);
            ad.setCapEnabled(true);
            ad.setCapConvRate(rate);
            ad.setVtEnabled(false);
            ad.setVtConvRate(Ad7747Eval.VTFS_RATE_16);
            ad.setVtMode(Ad7747Eval.VTMD_INT_TEMP);
            if (sepDaq) {
                doDaq(CHAN_CAP, nSamp, j);
                ad.setExcEnabled(false);
            }
        }
        if (!sepDaq) {
            doDaq(CHAN_CAP, nSamp);
        }
        monActive = false;
        out.println();
        logEvent();
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            DaqThread acq = daq[j];
            if (!sepDaq) {
                boards.get(j).setExcEnabled(false);
            }
            if (acq.getSampleCount() != 0) {
                out.format("   %s %.6g %.6g", id[j], acq.getMean(),
                           acq.getSigma());
            }
            else {
                out.format("   %s --------- -----------", id[j]);
            }
        }
        out.println();
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            DaqThread acq = daq[j];
            if (acq.getExceptionCount() != 0) {
                out.format("Board %s had %s USB exceptions.  First was: %s\n",
                           id[j], acq.getExceptionCount(), acq.getException());
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs a DAQ cycle on a single board.
    **
    ***************************************************************************
    */
    private void doDaq(int chan, int nSamp, int index)
    {
        if ((actvMask & (1 << index)) == 0) return;
        daqChan = chan;
        daqNSamp = nSamp;
        daq[index].startRun();
        for (int done = 0; done != (1 << index);) {
            try {
                Integer value = doneQueue.poll(10L, TimeUnit.SECONDS);
                if (value != null) {
                    done |= (1 << value);
                }
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs a DAQ cycle on all boards simultaneously.
    **
    ***************************************************************************
    */
    private void doDaq(int chan, int nSamp)
    {
        daqChan = chan;
        daqNSamp = nSamp;
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) != 0) {
                daq[j].startRun();
            }
        }
        for (int done = 0; done != actvMask;) {
            try {
                Integer value = doneQueue.poll(10L, TimeUnit.SECONDS);
                if (value != null) {
                    done |= (1 << value);
                }
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Logs event data.
    **
    ***************************************************************************
    */
    private void logEvent()
    {
        Date ts = new Date();
        if (nEvent == 0) {
            String
              name = logName + "_"
                       + String.format("%tY%<tm%<td_%<tH%<tM%<tS", ts) + ".dat";
            try {
                logFile = new PrintStream(name);
            }
            catch (IOException e) {
                out.println("Cannot open log file: " + e);
                return;
            }
        }

        logFile.format("%tY-%<tm-%<td\t%<tH:%<tM:%<tS\t%s\t%s",
                       ts, comment, rateNames.decode(rate));
        for (int j = 0; j < id.length; j++) {
            DaqThread acq = daq[j];
            logFile.format("\t%s\t%.6g\t%s\t%.6g\t%.6g\t%.6g\t%.6g", id[j],
                           acq.getTemperature(), acq.getSampleCount(),
                           acq.getMean(), acq.getSigma(), acq.getError(),
                           acq.getDrift());
        }
        logFile.println("\r");    // Need <cr> for Windows
        nEvent++;
    }


   /**
    ***************************************************************************
    **
    **  Reads a board's ID
    **
    ***************************************************************************
    */
    private static int readBoardId(Ad7747Eval ad) throws UsbException
    {
        float value = ad.readVoltage(Ad7747Eval.RDO_CONFIG
                                     | Ad7747Eval.RDO_EXTERN
                                     | Ad7747Eval.RDO_SINGLE);
        return (int)(10F * value + 0.5F);
    }


   /**
    ***************************************************************************
    **
    **  Reads a board's temperature
    **
    ***************************************************************************
    */
    private static double readTemperature(Ad7747Eval ad) throws UsbException
    {
        double temp = 0;
        ad.setVtEnabled(true);
        for (int j = 0; j < 2; j++) {
            temp = ad.readTemperature(0);
        }
        ad.setVtEnabled(false);

        return temp;
    }


   /**
    ***************************************************************************
    **
    **  Gets a board's index given its ID
    **
    ***************************************************************************
    */
    private int getBoardIndex(int bid)
    {
        if (bid == -1) return -1;
        for (int j = 0; j < id.length; j++) {
            if (id[j] == bid) return j;
        }
        out.println("Unknown board ID");

        return -2;
    }


   /**
    ***************************************************************************
    **
    **  Reads a line of terminal input in a thread-safe exception-free way
    **
    ***************************************************************************
    */
    private synchronized String readLine(String prompt)
    {
        try {
            return reader.readLine(prompt);
        }
        catch (IOException e) {
            return null;
        }
    }

        
   /**
    ***************************************************************************
    **
    **  Waits for an item to arrive on a queue
    **
    ***************************************************************************
    */
    private static void take(ArrayBlockingQueue queue)
    {
        while (true) {
            try {
                queue.take();
                return;
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Waits for terminal input
    **
    ***************************************************************************
    */
    private static void awaitTerminal()
    {
        while (true) {
            try {
                if (System.in.available() > 0) return;
                try {
                    Thread.sleep(50);
                }
                catch (InterruptedException e) {
                }
            }
            catch (IOException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears terminal input
    **
    ***************************************************************************
    */
    private static void clearTerminal()
    {
        try {
            while (System.in.available() > 0) {
                System.in.read();
            }
        }
        catch (IOException e) {
        }
    }

}
