package org.lsst.ccs.subsystem.metrology;

import org.lsst.ccs.subsystem.metrology.data.MetrologyState;

/**
 * AerotechP165 device class for the metrology subsystem
 *
 * @author: Homer Neal
 */
public class AerotechP165DeviceSim extends AerotechP165Device {

    private double x, y, z;
    private double speed = 100, rampRate;

    /**
     * Performs basic initialization.
     */
    @Override
    protected void initDevice() {
        fullName = "Simulated AerotechP165";
    }

    /**
     * Fully initializes the device.
     */
    @Override
    protected void initialize() {
        setOnline(true);
        initSensors();
        kstate = MetrologyState.pwrstates.OK;
        log.info("Connected to " + fullName);
    }

    /**
     * Closes the connection.
     */
    @Override
    protected void close() {
        kstate = MetrologyState.pwrstates.NOTCONFIGURED;
    }

    /**
     * gets version of the Ensemble controller
     *
     * @return Version string
     */
    @Override
    public String getVersion() {
        return "Simulated 1.0";
    }

    /**
     * abort any motion
     */
    @Override
    public void abort() {
    }

    /**
     * enable axes 
     */
    @Override
    public void enableAxes() {
        log.info("Enabling X axis");
        log.info("Enabling Y axis");
        log.info("Enabling Z axis");
    }

    /**
     * disable axes
     */
    @Override
    public void disableAxes() {
        log.info("Disabling X axis");
        log.info("Disabling Y axis");
        log.info("Disabling Z axis");
    }

    /**
     * setSpeed: step/s
     *
     * @param velocity
     */
    @Override
    public void setSpeed(double velocity) {
        this.speed = velocity;
    }

    /**
     * rampRate: step/s^2
     *
     * @param rate
     */
    @Override
    public void rampRate(double rate) {
        this.rampRate = rate;
    }

    /**
     * incremental move in x
     *
     * @param dx
     */
    @Override
    public void moveIncX(double dx)  {
        long time = Math.abs((long)((dx/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x += dx;
    }

    /**
     * incremental move in y
     *
     * @param dy
     */
    @Override
    public void moveIncY(double dy) {
        long time = Math.abs((long)((dy/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        y += dy;
    }

    /**
     * incremental move in z
     *
     * @param dz
     */
    @Override
    public void moveIncZ(double dz) {
        long time = Math.abs((long)((dz/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        z += dz;
    }

    /**
     * incremental move in x and y
     *
     * @param dx
     * @param dy
     */
    @Override
    public void moveIncXY(double dx, double dy) {
        long time = Math.abs((long)(((dx+dy)/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x += dx;
        y += dy;
    }

    /**
     * incremental move in x, y and z
     *
     * @param dx
     * @param dy
     * @param dz
     */
    @Override
    public void moveIncXYZ(double dx, double dy, double dz) {
        long time = Math.abs((long)(((dx+dy+dz)/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x += dx;
        y += dy;
        z += dz;
    }

    /**
     * absolute move in x
     *
     * @param xpos
     */
    @Override
    public void moveAbsX(double xpos) {
        //System.out.println("Moving from "+x+" to "+xpos+" at speed "+speed);
        long time = Math.abs((long)((Math.abs(xpos-x)/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x = xpos;
    }

    /**
     * absolute move in y
     *
     * @param ypos
     */
    @Override
    public void moveAbsY(double ypos) {
        long time = Math.abs((long)((Math.abs(ypos-y)/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        y = ypos;
    }

    /**
     * absolute move in z
     *
     * @param zpos
     */
    @Override
    public void moveAbsZ(double zpos) {
        long time = Math.abs((long)((Math.abs(zpos-z)/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        z = zpos;
    }

    /**
     * absolute move in x and y
     *
     * @param xpos
     * @param ypos
     */
    @Override
    public void moveAbsXY(double xpos, double ypos) {
        long time = Math.abs((long)(((Math.abs(ypos-y)+Math.abs(xpos-x))/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x = xpos;
        y = ypos;
    }

    /**
     * absolute move in x, y and z
     *
     * @param xpos
     * @param ypos
     * @param zpos
     */
    @Override
    public void moveAbsXYZ(double xpos, double ypos, double zpos) {
        long time = Math.abs((long)(((Math.abs(ypos-y)+Math.abs(xpos-x)+Math.abs(zpos-z))/speed)*1000));
        try {
            //System.out.println("Sleeping "+time);
            Thread.sleep(time/10);
        } catch (InterruptedException e) {}
        x = xpos;
        y = ypos;
        z = zpos;
    }

    /**
     * get x position
     *
     * @return The position
     */
    @Override
    public double getPosX() {
        lastPosx = x - xOffset;
        //System.out.println("lastPosx = "+lastPosx);
        return lastPosx;
    }

    /**
     * get y position
     *
     * @return The position
     */
    @Override
    public double getPosY() {
        lastPosy = y - yOffset;
        //System.out.println("lastPosy = "+lastPosy);
        return lastPosy;
    }

    /**
     * get z position
     *
     * @return The position
     */
    @Override
    public double getPosZ() {
        lastPosz = z - zOffset;
        //System.out.println("lastPosz = "+lastPosz);
        return lastPosz;
    }

    /**
     * get errors
     *
     * @return The error string
     */
    @Override
    public String getError() {
        return "Simulation error";
    }

    /**
     * go to home position
     */
    @Override
    public void goHome() {
        x = 0;
        y = 0;
        z = 0;
    }

    /**
     * setup homing
     */
    @Override
    public void setupHome(int mode) {
    }

    /**
     * setup low position limit
     */
    @Override
    public void softLowLim(String axis, int param) {
    }

    /**
     * setup high position limit
     */
    @Override
    public void softHighLim(String axis, int param) {
    }

    /**
     * acknowledge errors
     */
    @Override
    public void reset() {
    }

    /**
     * reset controller
     */
    @Override
    public void fullreset() {
    }

    /**
     * reconnect
     */
    @Override
    public void reconnect() {
    }

    /**
     * Send command
     *
     * @return The response (echoed command)
     */
    @Override
    public String aerotechChat(String command) {
        return command;
    }

}
