package org.lsst.ccs.subsystem.metrology;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.keyence.KeyenceG5001;
import org.lsst.ccs.subsystem.metrology.data.MetrologyState;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;

/**
 ***************************************************************************
 **
 * KeyenceG5001 device class for the metrology subsystem
 *
 * * @author: Homer Neal
 * **************************************************************************
 */
public class KeyenceG5001Device extends Device implements KeyenceDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POSITION", Channel.TYPE_UNKNOWN);
    }

    protected boolean isconnected = false; // is the device defined

    private KeyenceG5001 keydev = null;

    private String keyence_host = null;

    private int keyence_port = 38400;

    protected MetrologyState.pwrstates kstate = MetrologyState.pwrstates.NOTCONFIGURED;

    protected boolean failedToInitialize = false;
    protected double[] last_read;

    /**
     ***************************************************************************
     **
     ** KeyenceG5001Device constructor - put KeyenceG5001 in initial subsystem
     * state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public KeyenceG5001Device(String host, int port)
            throws DriverException {
        this.last_read = new double[]{0, 0, 0, 0, 0, 0, 0, 0};
        System.out.println("In Keyence initialization class");
        isconnected = true;
        keyence_host = host;
        keyence_port = port;
    }

    /**
     * Initialize the Device instance
     */
    @Override
    protected void initDevice() {
        fullName = "KeyenceG5001 module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (keydev != null) {
                keydev.close();
            }
            kstate = MetrologyState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("KeyenceG5001 device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        try {
            keydev = new KeyenceG5001();
            keydev.openserial(keyence_host, keyence_port);
            System.out.println("Openned connection to Keyence!");
        } catch (DriverException f) {
            System.out.println("Failed to open connection to KeyenceG5001 device!");
            isconnected = false;
        }

        if (isconnected) {
            log.debug("KeyenceG5001 device is already initialized");
            /*
             try {
             log.debug(keydev.getVersion());
             } catch (Exception e) {
             if (!failedToInitialize) {
             log.error("KeyenceG5001 device failed to respond to ident request! The initialization has FAILED.");
             failedToInitialize = true;
             }
             isconnected = false;
             }
             */
            log.debug("Setting device online.");
            setOnline(true);
            kstate = MetrologyState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected Keyence device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** gets version of the Ensemble controller *
     * **************************************************************************
     */
    @Command(description = "return controller version")
    public String getVersion() throws DriverException {
        return keydev.getVersion();
    }

    /**
     ***************************************************************************
     **
     ** read the distance from the Keyence head
     * **************************************************************************
     */
    @Command(name = "read", description = "get keyence head measurement")
    public double read(@Argument(name = "ikey", description = "keyence head number") int ikey
    ) throws DriverException {
        double[] response = this.readAll();
        return (response[ikey - 1]);
    }

    /**
     ***************************************************************************
     **
     ** return the sampling cycle setting being used for the displacement measurements
     * **************************************************************************
     */
    @Command(name = "getcycles", description = "get sampling cycle setting")
    public int getcycles( 
        //@Argument(name = "ikey", description = "keyence head number") int ikey    
            ) throws DriverException {
        return keydev.getnsamps();
    }

    /**
     ***************************************************************************
     **
     ** set the sampling cycle mode to be used for the displacement measurements
     * **************************************************************************
     */
    @Command(name = "setcycles", description = "set number of cycles")
    public void setcycles(
            //@Argument(name = "ikey", description = "keyence head number") int ikey,
            @Argument(name = "cycles", description = "sampling cycle mode") int cycles
    ) throws DriverException {
        keydev.setnsamps(cycles);
    }

    /**
     ***************************************************************************
     **
     ** return the sensing mode
     * **************************************************************************
     */
    @Command(name = "getmeasmode", description = "get mode")
    public int getmeasmode() throws DriverException {
        return -99; // note implemented yet
//               keydev.getmode(ikey);
    }

    /**
     ***************************************************************************
     **
     ** set the sensing mode
     * 
     * 0: Normal, 
     * 1: Translucent object,
     * 2: Transparent object, 
     * 3: Transparent object 2, 
     * 4: Semi opaque
     * 
     * **************************************************************************
     */
    @Command(name = "setmeasmode", description = "set mode")
    public void setmeasmode(@Argument(name = "mode", description = "mode number") int mode
    ) throws DriverException {
        keydev.setmeasmode(mode);
    }

    /**
     ***************************************************************************
     **
     ** set the minimum display unit
     * 
     * 
     * **************************************************************************
     */
    @Command(name = "setmindispunit", description = "set mode")
    public void setmindispunit(@Argument(name = "dspu", description = "(0: 0.01 mm, 1: 0.001 mm, 2: 0.0001 mm, 3: 0.00001 mm, 4: 0.1 um, 5: 0.01 um, 6: 0.001 um)") int dspu
    ) throws DriverException {
        keydev.setmindispunit(dspu);
    }

    /**
     ***************************************************************************
     **
     ** read the distance from the Keyence head
     * **************************************************************************
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(name = "readAll", description = "get keyence head measurement")
    public double[] readAll() throws DriverException {
        double[] response = keydev.readAll();
        for (int i = 0; i < response.length; i++) {
            last_read[i] = response[i];
        }
        return (response);
    }

     /***************************************************************************
     **
     ** communications mode
     *        
     * **************************************************************************
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "communications mode")
    public void commmode() throws DriverException {
        keydev.commmode();
    }

     /***************************************************************************
     **
     ** general mode
     *        
     * **************************************************************************
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "general mode")
    public void genmode() throws DriverException {
        keydev.genmode();
    }

    /**
     ***************************************************************************
     **
     ** get errors
     * **************************************************************************
     */
    @Command(description = "returns any axis fault messages")
    public String getError() throws DriverException {
//        return(keydev.getError());
        return ("not implimented yet");
    }

    /**
     ***************************************************************************
     **
     ** reset
     * **************************************************************************
     */
    @Command(description = "clears all fault mesdsages")
    public void reset() throws DriverException {
        keydev.reset();
        return;
    }

    /**
     ***************************************************************************
     **
     ** laser on/off
     * **************************************************************************
     */
    @Command(description = "sets lasers on/off")
    public void laser(@Argument(name = "laserstate", description = "false for laser off / true for on") boolean laserstate) throws DriverException {
        keydev.laser(laserstate);
        return;
    }

    /**
     ***************************************************************************
     ** Generic write operation followed by a read.
     ** 
     * **************************************************************************
     */
    @Command(description = "generic write", level=10)
    public String keyenceChat(@Argument(name = "command", description = "The raw Keyence command") String command) throws DriverException {
        keydev.writeKeyenceG5001(command);
        return keydev.readKeyenceG5001();
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        /*
         if (iType == null) {
         mon.reportError(name, "type", type);
         } else if (iType != Channel.TYPE_SWITCH) {
         mon.reportError(name, "Wrong channel type specified! type = ", type);
         Exception e;
         }
         */
        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
//            if (type == Channel.TYPE_SWITCH) {
            log.debug("setting channel online - chan=" + chan);
            setOnline(true);
//            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
//        log.info("KeyenceG5001Device readChannel called! chan=" + chan + " type=" + type);
//        System.out.println("KeyenceG5001Device readChannel called! chan=" + chan + " type=" + type);
        try {
            if (chan == 0) {
                value = last_read[0];
            }
            if (chan == 1) {
                value = last_read[1];
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    public double[] getLast_read() {
        return last_read;
    }

    @Command(name = "setstate", description = "set KeyenceG5001 device status")
    public void setState(int istate) {
        kstate = MetrologyState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get KeyenceG5001 device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
