package org.lsst.ccs.subsystem.metrology;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.aerotech.AerotechPro165;
import org.lsst.ccs.subsystem.metrology.data.MetrologyState;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;

/**
 * AerotechP165 device class for the metrology subsystem
 *
 * @author: Homer Neal
 */
public class AerotechP165Device extends Device implements AerotechDevice {

    /*
     * Fields set from Groovy file
     */
    protected String host = null;
    protected int port = 0;  // Driver uses default

    /*
     * Fields used by simulation code
     */
    MetrologyState.pwrstates kstate = MetrologyState.pwrstates.NOTCONFIGURED;
    double xOffset = -300.0, yOffset = -200.0, zOffset = -100.0;
    double lastPosx, lastPosy, lastPosz;

    /*
     * Private fields
     */
    private final AerotechPro165 devaero = new AerotechPro165();

    /**
     * Performs basic initialization.
     */
    @Override
    protected void initDevice() {
        if (host == null) {
            MonitorLogUtils.reportConfigError(log, name, "host", "not defined");
        }
        fullName = "AerotechP165 (" + host +")";
    }

    /**
     * Fully initializes the device.
     */
    @Override
    protected void initialize() {
        try {
            devaero.opennet(host, port);
            setOnline(true);
            initSensors();
            kstate = MetrologyState.pwrstates.OK;
            log.info("Connected to " + fullName);
        } catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    protected void close() {
        try {
            devaero.close();
            kstate = MetrologyState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return Encoded type & subtype
     * @throws DriverException
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= 3) {
            MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return The channel value
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
//        log.info("AerotechP165Device readChannel called! chan=" + chan + " type=" + type);
//        System.out.println("AerotechP165Device readChannel called! chan=" + chan + " type=" + type);

//        try {
        switch (chan) {
            case 0: // pos_x
                value = lastPosx;
                break;
            case 1: // pos_y
                value = lastPosy;
                break;
            case 2: // pos_z
                value = lastPosz;
                break;
        }
//                oldval[chan+2] = value;
            /*
        } catch (DriverLockedException f) {
            System.out.print("A"); // just to indicate that we are waiting
        } catch (DriverException e) {
            log.error("Error reading channel type " + chan + ": " + e);
            log.error("trying to reopen the connection");
            try {
                devaero.opennet(aerotech_host, aerotech_port);
            } catch (DriverException ex) {
                log.error("failed attempt to reopen connection");
            }

        }
*/
        return value;
    }

    @Override
    @Command(description="return the x offset")
    public double getxOffset() {
        return xOffset;
    }

    @Override
    @Command(description="set the x offset")
    public void setxOffset(@Argument(description="offset in user units") double xoff) {
        this.xOffset = xoff;
    }

    @Override
    @Command(description="return the y offset")
    public double getyOffset() {
        return yOffset;
    }

    @Override
    @Command(description="set the y offset")
    public void setyOffset(@Argument(description="offset in user units") double yoff) {
        this.yOffset = yoff;
    }

    @Override
    @Command(description="return the z offset")
    public double getzOffset() {
        return zOffset;
    }

    @Override
    @Command(description="set the z offset")
    public void setzOffset(@Argument(description="offset in user units") double zoff) {
        this.zOffset = zoff;
    }

    /**
     * gets version of the Ensemble controller
     *
     * @return The version
     * @throws DriverException
     */
    @Override
    @Command(description = "return controller version")
    public String getVersion() throws DriverException {
        return devaero.getVersion();
    }

    /**
     * abort any motion
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "abort motion")
    public void abort() throws DriverException {
        devaero.abort();
    }

    /**
     * enable axes
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "enable axes")
    public void enableAxes() throws DriverException {
        log.debug("Enabling X axis");
        devaero.enableAxis('X');
        sleep(1.0);
       log.debug("Enabling Y axis");
        devaero.enableAxis('Y');
        sleep(1.0);
        log.debug("Enabling Z axis");
        devaero.enableAxis('Z');
    }

    /**
     * disable axes
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "disable axes")
    public void disableAxes() throws DriverException {
        log.debug("Disabling X axis");
        devaero.disableAxis('X');
        sleep(1.0);
        log.debug("Disabling Y axis");
        devaero.disableAxis('Y');
        sleep(1.0);
        log.debug("Disabling Z axis");
        devaero.disableAxis('Z');
    }

    /**
     * Sleep - what a waste
     *
     * @param secs The sleep period
     */
    private void sleep(double secs) {
        try {
            long millis = (long) (secs * 1000);
            log.debug("Sleeping for " + millis + " millis");
            Thread.sleep(millis);
        } catch (InterruptedException ex) {
            log.error("Rude awakening!" + ex);
        }
    }

    /**
     * setSpeed: step/s
     *
     * @param velocity
     * @throws DriverException
     */
    @Override
    @Command(description = "setSpeed")
    public void setSpeed(@Argument(description = "step / s") double velocity) throws DriverException {
 //       devaero.rampMode("RATE");
        devaero.setSpeed(velocity);
    }

    /**
     * rampRate: step/s^2
     *
     * @param rate
     * @throws DriverException
     */
    @Override
    @Command(description = "rampRate")
    public void rampRate(@Argument(description = "step / s2") double rate) throws DriverException {
        devaero.rampMode("RATE");
        devaero.rampRate(rate);
    }

    /**
     * incremental move in x
     *
     * @param dx
     * @throws DriverException
     */
    @Override
    @Command(description = "move incrementally in x")
    public void moveIncX(@Argument(description = "amount to move in user units") double dx) throws DriverException {
        devaero.moveIncX(dx);
    }

    /**
     * incremental move in y
     * 
     * @param dy
     * @throws DriverException
     */
    @Override
    @Command(description = "move incrementally in y")
    public void moveIncY(@Argument(description = "amount to move in user units") double dy) throws DriverException {
        devaero.moveIncY(dy);
    }

    /**
     * incremental move in z
     *
     * @param dz
     * @throws DriverException
     */
    @Override
    @Command(description = "move incrementally in z")
    public void moveIncZ(@Argument(description = "amount to move in user units") double dz) throws DriverException {
        devaero.moveIncZ(dz);
    }

    /**
     * incremental move in x and y
     *
     * @param dx
     * @param dy
     * @throws DriverException
     */
    @Override
    @Command(description = "move incrementally in x and y")
    public void moveIncXY(@Argument(description = "amount to move in user units") double dx,
                          @Argument(description = "amount to move in user units") double dy) throws DriverException {
        devaero.moveInc(dx, dy);
    }

    /**
     * moveInx_xyz: incremental move in x, y and z
     *
     * @param dx
     * @param dy
     * @param dz
     * @throws DriverException
     */
    @Override
    @Command(description = "move incrementally in x, y and z")
    public void moveIncXYZ(@Argument(description = "amount to move in user units") double dx,
                           @Argument(description = "amount to move in user units") double dy,
                           @Argument(description = "amount to move in user units") double dz) throws DriverException {
        devaero.moveInc(dx, dy, dz);
    }

    /**
     * absolute move
     *
     * @param x
     * @throws DriverException
     */
    @Override
    @Command(description = "move absolutely in x")
    public void moveAbsX(@Argument(description = "x position in user units") double x) throws DriverException {
        devaero.moveAbsX(x);// +this.xOffset);
    }

    /**
     * absolute move
     *
     * @param y
     * @throws DriverException
     */
    @Override
    @Command(description = "move absolutely in y")
    public void moveAbsY(@Argument(description = "y position in user units") double y) throws DriverException {
        devaero.moveAbsY(y); //+this.yOffset);
    }

    /**
     * absolute move
     *
     * @param z
     * @throws DriverException
     */
    @Override
    @Command(description = "move absolutely in z")
    public void moveAbsZ(@Argument(description = "z position in user units") double z) throws DriverException {
        devaero.moveAbsZ(z); // +this.zOffset);
    }

    /**
     * absolute move in x and y
     *
     * @param x
     * @param y
     * @throws DriverException
     */
    @Override
    @Command(description = "move absolutely in x and y")
    public void moveAbsXY(@Argument(description = "x position in user units") double x,
                          @Argument(description = "y position in user units") double y) throws DriverException {
        devaero.moveAbs(x, y); // xpos+this.xOffset, ypos+this.yOffset);
    }

    /**
     * absolute move in x, y and z
     *
     * @param x
     * @param y
     * @param z
     * @throws DriverException
     */
    @Override
    @Command(description = "move absolutely in x, y and z")
    public void moveAbsXYZ(@Argument(description = "x position in user units") double x,
                           @Argument(description = "y position in user units") double y,
                           @Argument(description = "z position in user units") double z) throws DriverException {
        devaero.moveAbs(x, y, z); //xpos+this.xOffset, ypos+this.yOffset, zpos+this.zOffset);
    }

    /**
     * get xyz position
     *
     * @return position array
     * @throws DriverException
     */
    @Override
    @Command(description = "returns position in x, y and z")
    public double[] getPosXYZ() throws DriverException {
/*
        double p[] = devaero.getPos_xyz();
*/
        double p[] = new double[3];
        p[0] = this.getPosX();
        p[1] = this.getPosY();
        p[2] = this.getPosZ();

        lastPosx = p[0];
        lastPosy = p[1];
        lastPosz = p[2];

        //System.out.println("lastPosx = "+lastPosx);
        //System.out.println("lastPosy = "+lastPosy);
        //System.out.println("lastPosz = "+lastPosz);
        return p;
    }

    /**
     * get x position
     *
     * @return x position
     * @throws DriverException
     */
    @Override
    @Command(description = "returns position in x")
    public double getPosX() throws DriverException {
        double xmeas = Double.NaN;
        while (Double.isNaN(xmeas)) {
            xmeas = devaero.getPosX();
            if (Double.isNaN(xmeas)) {
                log.debug("Waiting for x position value to become available");
            }
        }
        xmeas -= this.xOffset;
        lastPosx = xmeas;
        //System.out.println("lastPosx = "+lastPosx);
        return lastPosx;
    }

    /**
     * get y position
     *
     * @return y position
     * @throws DriverException
     */
    @Override
    @Command(description = "returns position in y")
    public double getPosY() throws DriverException {
        double ymeas = Double.NaN;
        while (Double.isNaN(ymeas)) {
            ymeas = devaero.getPosY();
            if (Double.isNaN(ymeas)) {
                log.debug("Waiting for y position value to become available");
            }
        }
        ymeas -= this.yOffset;
        lastPosy = ymeas;
        //System.out.println("lastPosy = "+lastPosy);
        return lastPosy;
    }

    /**
     * get z position
     *
     * @return z position
     * @throws DriverException
     */
    @Override
    @Command(description = "returns position in z")
    public double getPosZ() throws DriverException {
        double zmeas = Double.NaN;
        while (Double.isNaN(zmeas)) {
            zmeas = devaero.getPosZ();
            if (Double.isNaN(zmeas)) {
                log.debug("Waiting for z position value to become available");
            }
        }
        zmeas -= this.zOffset;
        lastPosz = zmeas;
        //System.out.println("lastPosz = "+lastPosz);
        return lastPosz;
    }

    /**
     * Sends command to Aerotech
     *
     * @param command
     * @return The response
     * @throws DriverException
     */
    @Command(description = "generic write", level = 10)
    public String aerotechChat(@Argument(name = "command", description = "The raw Aerotech command")
                               String command) throws DriverException {
        return devaero.read(command);
    }

    /**
     * return last x position
     *
     * @return The position
     */
    @Override
    public double getLastPosX() {
        return lastPosx;
    }

    /**
     * return last y position
     *
     * @return The position
     */
    @Override
    public double getLastPosY() {
        return lastPosy;
    }

    /**
     * return last z position
     *
     * @return The position
     */
    @Override
    public double getLastPosZ() {
        return lastPosz;
    }


    /**
     * get errors
     *
     * @return The error string
     * @throws DriverException
     */
    @Override
    @Command(description = "returns any axis fault messages")
    public String getError() throws DriverException {
        return devaero.getError();
    }

    /**
     * go to home position
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "goes to home position")
    public void goHome() throws DriverException {

        devaero.goHome();

        /*
        Sleep(10.0); // wait for motion to settle
        double new_xOffset = this.getPos_x() + this.xOffset;
        Sleep(1.0); // pause
        double new_yOffset = this.getPos_y() + this.yOffset;
        Sleep(1.0); // pause
        double new_zOffset = this.getPos_z() + this.zOffset;
        */
        double new_xOffset = this.xOffset;
        double new_yOffset = this.yOffset;
        double new_zOffset = this.zOffset;
        
        this.xOffset = new_xOffset;
        this.yOffset = new_yOffset;
        this.zOffset = new_zOffset;
        
        log.error("Offsets determined from home position are ("+
                this.xOffset+","+this.yOffset+","+this.zOffset+")");
    }

    /**
     * setup homing
     *
     * @param mode The homing mode
     * @throws DriverException
     */
    @Override
    @Command(description = "setup homing mode")
    public void setupHome(int mode) throws DriverException {
        devaero.setupHome(mode);
    }

    /**
     * setup low position limit
     *
     * @param axis The axis name
     * @param param The limit
     * @throws DriverException
     */
    @Override
    @Command(description = "setup low position limit")
    public void softLowLim(String axis, int param) throws DriverException {
        devaero.softLowLim(axis,param);
    }

    /**
     * setup high position limit
     *
     * @param axis The axis name
     * @param param The limit
     * @throws DriverException
     */
    @Override
    @Command(description = "setup high position limit")
    public void softHighLim(String axis, int param) throws DriverException {
        devaero.softHighLim(axis,param);
    }

    /**
     * acknowledge errors
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "clears all fault messages")
    public void reset() throws DriverException {
        devaero.reset();
    }

    /**
     * reset controller
     *
     * @throws DriverException
     */
    @Override
    @Command(description = "reset controller")
    public void fullreset() throws DriverException {
        devaero.fullreset();
        log.error("trying to reopen the connection");
        reconnect();
    }

    /**
     * Reconnects.
     *
     * Uses the monitoring system reconnection feature
     */
    @Override
    @Command(description = "reconnect to controller")
    public void reconnect() {
        setOnline(false);
    }

    /**
     * Sets the state.
     *
     * @param istate The state
     */
    @Command(name = "setstate", description = "set AerotechP165 device status")
    public void setState(int istate) {
        kstate = MetrologyState.pwrstates.values()[istate];
    }

    /**
     * Gets the state.
     *
     * @return The state
     */
    @Command(name = "getstate", description = "get AerotechP165 device status")
    public int getState() {
        return kstate.ordinal();
    }

}
