package org.lsst.ccs.subsystem.metrology;

import hep.aida.*;
import hep.aida.util.XMLUtils;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.BusMessage;
//import org.lsst.ccs.bus.messages.AidaPlots;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.metrology.data.MetrologyConfig;
import org.lsst.ccs.subsystem.metrology.data.MetrologyFullState;
import org.lsst.ccs.subsystem.metrology.data.MetrologyState;
import org.lsst.ccs.subsystem.metrology.status.MetrologyStateStatus;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.data.MonitorFullState;
import org.lsst.ccs.utilities.logging.Logger;

import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.ConfigurationServiceException;
import org.lsst.ccs.subsystem.metrology.data.MetrologyAgentProperties;

/**
 * Implements the metrology subsystem for use with Aerotech positioners
 * subsystem.
 *
 * @author Homer Neal
 */
public class MetrologySubSys implements HasLifecycle, Monitor.AlarmHandler, StatusMessageListener {

    List<Device> devcList = new ArrayList<>();
    String stateCorrelId;
    MetrologyConfig metrologyc = new MetrologyConfig();
//    MetrologyConfigCatcher atcc = new MetrologyConfigCatcher(metrologyc);
    private ConcurrentMessagingUtils cmu;
    private static final Logger LOGGER = Logger.getLogger("org.lsst.ccs.subsystem.metrology.main");
    private static final String METROLOGY_DEST = System.getProperty("lsst.ccs.metrology.metrologyguidest", "metrology");
    private String teststand_dest = "ts2";
    private final String ts8_dest = "ts8";

    public static final String BROADCAST_TASK = "publish-data";

    @LookupField(strategy=LookupField.Strategy.TOP)
    Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupName
    private String name;
    
    private long broadcastMillis = 10000;
    
    @ConfigurationParameter
    private boolean monitorCCDTemp = true;

    
    /**
     * Data fields.
     */
    int state = 0;

    boolean last_systemOK = true;  // last system check status
    int nwarn = 0;             // number of consequtive warnings
    double last_vac = 0.;         // last pressure reading

    int disp_mode = 0;  // default displacement mode of adding both displacement measurements

    /* define references to the TS devices */
//    SensorControllerDevice ccdDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    AerotechDevice aeroDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    KeyenceDevice keyDevc;

    IDataPointSetFactory dpsf;
    //Create analysis factory
    IAnalysisFactory af = IAnalysisFactory.create();
    List<Double> plot_posx = new ArrayList<>();
    List<Double> plot_posy = new ArrayList<>();
    List<Double> plot_posz = new ArrayList<>();

    List<Double> move_posx = new ArrayList<>();
    List<Double> move_posy = new ArrayList<>();

    private boolean abort_action = false;
    private double last_temp_a = -999.;
    private double last_temp_b = -999.;
    private double last_temp_c = -999.;
    private double last_temp_d = -999.;
    private boolean got_reb_temps = false;
    private double last_vqmpressure = -999.;

    ConfigurationInfo configInfo = null;

    /**
     * Perform scan
     *
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void scan() throws DriverException, InterruptedException {
        scanfl(
                "/home/ts5prod/Metrology_Scan_Data.csv"
        );
    }

    /**
     * Perform scan to a specified file
     *
     * @param filename
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region to the specified file")
    public void scanfl(@Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename
    ) throws DriverException, InterruptedException {
        configInfo = subsys.getConfigurationService().getConfigurationInfo();
        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        scan(
                filename,
                Double.parseDouble(configForComponent.get("startx")),
                Double.parseDouble(configForComponent.get("stopx")),
                Double.parseDouble(configForComponent.get("dx")),
                Double.parseDouble(configForComponent.get("starty")),
                Double.parseDouble(configForComponent.get("stopy")),
                Double.parseDouble(configForComponent.get("dy")),
                Double.parseDouble(configForComponent.get("z"))
        );
    }

    /**
     * Perform scan
     *
     * @param filename
     * @param startx
     * @param stopx
     * @param starty
     * @param stopy
     * @param dx
     * @param dy
     * @param z
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void scan(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename, @Argument(name = "startx", description = "x start position") double startx,
            @Argument(name = "stopx", description = "x stop position") double stopx,
            @Argument(name = "dx", description = "x step size") double dx,
            @Argument(name = "starty", description = "y start position") double starty,
            @Argument(name = "stopy", description = "y stop position") double stopy,
            @Argument(name = "dy", description = "y step size") double dy,
            @Argument(name = "z", description = "z position") double z
    ) throws DriverException, InterruptedException {
        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        scan(filename, startx, stopx, dx, starty, stopy, dy, z,
                Double.parseDouble(configForComponent.get("speed")));
    }

    /**
     * Perform scan
     *
     * @param filename
     * @param startx
     * @param stopx
     * @param starty
     * @param stopy
     * @param dx
     * @param dy
     * @param z
     * @param speed
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void scan(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename,
            @Argument(name = "startx", description = "x start position") double startx,
            @Argument(name = "stopx", description = "x stop position") double stopx,
            @Argument(name = "dx", description = "x step size") double dx,
            @Argument(name = "starty", description = "y start position") double starty,
            @Argument(name = "stopy", description = "y stop position") double stopy,
            @Argument(name = "dy", description = "y step size") double dy,
            @Argument(name = "z", description = "z position") double z,
            @Argument(name = "speed", description = "speed") double speed
    ) throws DriverException, InterruptedException {

        LOGGER.info("Scan requested with parameters:\n"
                + " x: " + startx + "mm ->" + stopx + "mm  in steps of " + dx + "mm \n"
                + " y: " + starty + "mm ->" + stopy + "mm  in steps of " + dy + "mm \n"
                + " output to: " + filename
        );
        metrologyc.setOperatingState(MetrologyConfig.operating_states.SCANNING.ordinal());

        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        aeroDevc.moveAbsZ(z);
        double accel = Double.parseDouble(configForComponent.get("acceleration"));
        int nsamps = Integer.decode(configForComponent.get("nsamples"));
        int measmode = Integer.decode(configForComponent.get("measmode"));
        LOGGER.debug("Setting the acceleration to " + accel);
        aeroDevc.rampRate(accel);
        LOGGER.debug("Setting the speed to " + speed);
        aeroDevc.setSpeed(speed);

        // these can only occur in communications mode
        keyDevc.commmode();
        LOGGER.debug("Setting cycle mode to " + nsamps);
        keyDevc.setcycles(nsamps);

        LOGGER.debug("Setting surface measurement mode to " + measmode);
        keyDevc.setmeasmode(measmode);
        keyDevc.genmode();

        plot_posx.clear();
        plot_posy.clear();
        plot_posz.clear();
        move_posx.clear();
        move_posy.clear();

        if (!filename.isEmpty()) {
            FileWriter fstream = null;
            BufferedWriter out = null;
            File pdFl = new File(filename);

            try {
                if (pdFl.exists()) {
                    LOGGER.error("File already exists!!! ABORTING!");
                    return;
                }
                pdFl.createNewFile();
            } catch (Exception e) {
                LOGGER.error("Unable to create file (" + filename + ") for reason " + e);
            }
            try {
                fstream = new FileWriter(pdFl);
            } catch (IOException e) {
                LOGGER.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
            }
            try {
                if (fstream != null) {
                    out = new BufferedWriter(fstream);
                }
            } catch (Exception e) {
                LOGGER.error("Failed to open buffered writer stream for file (" + filename + ") for reason " + e);
            }

            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# filename: " + filename + "\n"
                            + "# start time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n"
                            + "#  x: " + startx + "mm ->" + stopx + "mm  in steps of " + dx + "mm \n"
                            + "#  y: " + starty + "mm ->" + stopy + "mm  in steps of " + dy + "mm \n"
                            + "#  z: " + z + "mm \n"
                            + "#  speed: " + speed + "\n"
                            + "#  acceleration: " + accel + "\n"
                            + "#  cycle mode: " + nsamps + "\n"
                            + "#  measurement mode: " + measmode + "\n"
                            + "#  displacement summing mode: " + MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode] + "\n"
                            + "# #######################################################################\n"
                    );
                } catch (IOException ex) {
                    LOGGER.error(ex);
                }
            }

            int i = 0;
            double y = starty;
            for (double x = startx; x <= stopx; x += dx) {
//                for (double y = starty; y <= stopy; y += dy) {
                while (y >= starty && y <= stopy) {
                    if (abort_action) {
                        break;
                    }
                    aeroDevc.moveAbsXY(x, y);
                    sleep(0.2);
                    double[] disps = keyDevc.readAll();
//                    double displacement = keyDevc.read(2) + keyDevc.read(1);
                    double displacement = -999.;
//                    switch (MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode]) {
                    if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.NormalDispSum.ordinal()) {
                        if (disps[0] > -900. && disps[1] > -900.) {
                            displacement = disps[1] + disps[0];
                        }
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead1Only.ordinal()) {
                        displacement = disps[0];
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead2Only.ordinal()) {
                        displacement = disps[1];
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementDiff.ordinal()) {
                        if (disps[0] > -900. && disps[1] > -900.) {
                            displacement = disps[1] - disps[0];
                        }
                    } else {
                        LOGGER.error("BAD disp_mode!");
                    }
                    double xmeas = Double.NaN;
                    double ymeas = Double.NaN;
                    double zmeas = Double.NaN;

                    while (Double.isNaN(xmeas) || Double.isNaN(ymeas) || Double.isNaN(zmeas)) {
                        double p[] = aeroDevc.getPosXYZ();
                        xmeas = p[0];
                        ymeas = p[1];
                        zmeas = p[2];
                        //System.out.println("Waiting for position information to become available.");
                    }

                    sleep(0.2);
                    System.out.println("x= " + x + " y= " + y + " displacement= " + displacement + " xmeas= " + xmeas
                            + " ymeas= " + ymeas + " disp1= " + disps[0] + " disp2= " + disps[1]
                            + " z= " + z + " zmeas= " + zmeas
                            + " temp_a= " + last_temp_a
                            + " temp_b= " + last_temp_b
                            + " temp_c= " + last_temp_c
                            + " temp_d= " + last_temp_d
                            + " vqmpressure= " + last_vqmpressure
                            + " tstamp= " + Long.toString((long) System.currentTimeMillis())
                    );

                    if (displacement > -900.) {
                        plot_posx.add(xmeas);
                        plot_posy.add(ymeas);
                        plot_posz.add(displacement);

                        if (out != null) {
                            String line = x + "," + y + "," + displacement + "," + xmeas + "," + ymeas + "," + disps[0] + "," + disps[1] + ","
                                    + z + "," + zmeas
                                    + "," + last_temp_a
                                    + "," + last_temp_b
                                    + "," + last_temp_c
                                    + "," + last_temp_d
                                    + "," + last_vqmpressure
                                    + "," + Long.toString((long) System.currentTimeMillis());

                            try {
                                out.write(line);
                                out.newLine();
                                out.flush();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    }

                    i++;
                    y += dy;
                }
                dy = -dy; // flip directions and step back in bounds
                y += dy;
            }
            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# end time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n");
                    out.close();
                    if (System.getenv("HOSTNAME").contains("ts5.inst.bnl")) {
                        Runtime r = Runtime.getRuntime();
                        try {
                            LOGGER.warning("Copying scan data to BNL astro GPFS");
                            Process p1 = r.exec("scp -rp " + filename + " ccdtest@astroracf:/gpfs01/astro/workarea/ccdtest/ts5/DATA/");
                            LOGGER.warning("The scan has been copied to: /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                            Process p2 = r.exec("ssh ccdtest@astroracf ls -l /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                        } catch (IOException ex) {
                            LOGGER.error("Scan copy failed with error " + ex);
                        }
                    }
                } catch (IOException e) {
                    LOGGER.error("Failed to close file!");
                }
            }

            if (!abort_action) {
                plotscan(180., 80., 90.);
            }

        }

        metrologyc.setOperatingState(MetrologyConfig.operating_states.IDLE.ordinal());

        abort_action = false;

    }

    /**
     * Perform PointList scan
     *
     * @param filename
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void scanPL(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename
    ) throws DriverException, InterruptedException {
        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());

        scanPL(filename, Double.parseDouble(configForComponent.get("speed")));
    }

    /**
     * Perform PointList scan
     *
     * @param filename
     * @param speed
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void scanPL(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename,
            @Argument(name = "speed", description = "speed") double speed
    ) throws DriverException, InterruptedException {

        LOGGER.info("Point List Scan requested with parameters:\n"
                + " output to: " + filename
        );
        metrologyc.setOperatingState(MetrologyConfig.operating_states.SCANNING.ordinal());

        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());

        double accel = Double.parseDouble(configForComponent.get("acceleration"));
        int nsamps = Integer.decode(configForComponent.get("nsamples"));
        int measmode = Integer.decode(configForComponent.get("measmode"));
        LOGGER.debug("Setting the acceleration to " + accel);
        aeroDevc.rampRate(accel);
        LOGGER.debug("Setting the speed to " + speed);
        aeroDevc.setSpeed(speed);

        double z = aeroDevc.getLastPosZ();

        // these can only occur in communications mode
        keyDevc.commmode();
        LOGGER.debug("Setting cycle mode to " + nsamps);
        keyDevc.setcycles(nsamps);

        LOGGER.debug("Setting surface measurement mode to " + measmode);
        keyDevc.setmeasmode(measmode);
        keyDevc.genmode();

        plot_posx.clear();
        plot_posy.clear();
        plot_posz.clear();
        move_posx.clear();
        move_posy.clear();

        if (!filename.isEmpty()) {
            FileWriter fstream = null;
            BufferedWriter out = null;

            File pdFl = new File(filename);

            try {
                if (pdFl.exists()) {
                    LOGGER.error("File already exists!!! ABORTING!");
                    return;
                }
                pdFl.createNewFile();
            } catch (Exception e) {
                LOGGER.error("Unable to create file (" + filename + ") for reason " + e);
            }
            try {
                fstream = new FileWriter(pdFl);
            } catch (IOException e) {
                LOGGER.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
            }
            try {
                if (fstream != null) {
                    out = new BufferedWriter(fstream);
                }
            } catch (Exception e) {
                LOGGER.error("Failed to open buffered writer stream for file (" + filename + ") for reason " + e);
            }

            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# filename: " + filename + "\n"
                            + "# start time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n"
                            + "#  speed: " + speed + "\n"
                            + "#  acceleration: " + accel + "\n"
                            + "#  cycle mode: " + nsamps + "\n"
                            + "#  measurement mode: " + measmode + "\n"
                            + "#  displacement summing mode: " + MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode] + "\n"
                            + "# #######################################################################\n"
                    );
                } catch (IOException ex) {
                    LOGGER.error(ex);
                }
            }

            int i = 0;
            for (MetrologyConfig.pointset_states componentState : MetrologyConfig.pointset_states.values()) {
                String componentName = componentState.name();
                Map<String, String> PSconfigForComponent = configInfo.getCurrentValuesFor(componentName);

                double PointX = Double.parseDouble(PSconfigForComponent.get("PointX"));
                double PointY = Double.parseDouble(PSconfigForComponent.get("PointY"));
                double PointZ = Double.parseDouble(PSconfigForComponent.get("PointZ"));
                if (abort_action) {
                    break;
                }
                if (PointX<-900. || PointY<-900. || PointZ<-900.) {
                    continue;
                }
                aeroDevc.moveAbsXYZ(PointX, PointY, PointZ);
                sleep(0.2);
                double[] disps = keyDevc.readAll();
//                    double displacement = keyDevc.read(2) + keyDevc.read(1);
                double displacement = -999.;
//                    switch (MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode]) {
                if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.NormalDispSum.ordinal()) {
                    if (disps[0] > -900. && disps[1] > -900.) {
                        displacement = disps[1] + disps[0];
                    }
                } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead1Only.ordinal()) {
                    displacement = disps[0];
                } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead2Only.ordinal()) {
                    displacement = disps[1];
                } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementDiff.ordinal()) {
                    if (disps[0] > -900. && disps[1] > -900.) {
                        displacement = disps[1] - disps[0];
                    }
                } else {
                    LOGGER.error("BAD disp_mode!");
                }
                double xmeas = Double.NaN;
                double ymeas = Double.NaN;
                double zmeas = Double.NaN;

                while (Double.isNaN(xmeas) || Double.isNaN(ymeas) || Double.isNaN(zmeas)) {
                    double p[] = aeroDevc.getPosXYZ();
                    xmeas = p[0];
                    ymeas = p[1];
                    zmeas = p[2];
                    //System.out.println("Waiting for position information to become available.");
                }

                sleep(0.2);
                System.out.println("x= " + PointX + " y= " + PointY + " displacement= " + displacement + " xmeas= " + xmeas
                        + " ymeas= " + ymeas + " disp1= " + disps[0] + " disp2= " + disps[1]
                        + " z= " + PointZ + " zmeas= " + zmeas
                        + " temp_a= " + last_temp_a
                        + " temp_b= " + last_temp_b
                        + " temp_c= " + last_temp_c
                        + " temp_d= " + last_temp_d
                        + " vqmpressure= " + last_vqmpressure
                        + " tstamp= " + Long.toString((long) System.currentTimeMillis())
                );

                if (displacement > -900.) {
                    plot_posx.add(xmeas);
                    plot_posy.add(ymeas);
                    plot_posz.add(displacement);

                    if (out != null) {
                        String line = PointX + "," + PointY + "," + displacement + "," + xmeas + "," + ymeas + "," + disps[0] + "," + disps[1] + ","
                                + PointZ + "," + zmeas
                                + "," + last_temp_a
                                + "," + last_temp_b
                                + "," + last_temp_c
                                + "," + last_temp_d
                                + "," + last_vqmpressure
                                + "," + Long.toString((long) System.currentTimeMillis());

                        try {
                            out.write(line);
                            out.newLine();
                            out.flush();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                }

                i++;
            }
            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# end time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n");
                    out.close();
                    if (System.getenv("HOSTNAME").contains("ts5.inst.bnl")) {
                        Runtime r = Runtime.getRuntime();
                        try {
                            LOGGER.warning("Copying scan data to BNL astro GPFS");
                            Process p1 = r.exec("scp -rp " + filename + " ccdtest@astroracf:/gpfs01/astro/workarea/ccdtest/ts5/DATA/");
                            LOGGER.warning("The scan has been copied to: /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                            Process p2 = r.exec("ssh ccdtest@astroracf ls -l /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                        } catch (IOException ex) {
                            LOGGER.error("Scan copy failed with error " + ex);
                        }
                    }
                } catch (IOException e) {
                    LOGGER.error("Failed to close file!");
                }
            }

//            if (!abort_action) {
//                plotscan(180., 80., 90.);
//            }
        }

        metrologyc.setOperatingState(MetrologyConfig.operating_states.IDLE.ordinal());

        abort_action = false;

    }

    /**
     * Perform smooth scan
     *
     * @param filename
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void noStepScan(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename) throws DriverException, InterruptedException {

        configInfo = subsys.getConfigurationService().getConfigurationInfo();

        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        noStepScan(filename,
                Double.parseDouble(configForComponent.get("startx")),
                Double.parseDouble(configForComponent.get("stopx")),
                Double.parseDouble(configForComponent.get("dx")),
                Double.parseDouble(configForComponent.get("starty")),
                Double.parseDouble(configForComponent.get("stopy")),
                Double.parseDouble(configForComponent.get("dy")),
                Double.parseDouble(configForComponent.get("z"))
        );
    }

    /**
     * Perform scan
     *
     * @param filename
     * @param startx
     * @param stopx
     * @param starty
     * @param stopy
     * @param dx
     * @param dy
     * @param z
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    public void noStepScan(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename,
            @Argument(name = "startx", description = "x start position") double startx,
            @Argument(name = "stopx", description = "x stop position") double stopx,
            @Argument(name = "dx", description = "x step size") double dx,
            @Argument(name = "starty", description = "y start position") double starty,
            @Argument(name = "stopy", description = "y stop position") double stopy,
            @Argument(name = "dy", description = "y step size") double dy,
            @Argument(name = "z", description = "z position") double z
    ) throws DriverException, InterruptedException {

        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        noStepScan(filename, startx, stopx, dx, starty, stopy, dy, z,
                Double.parseDouble(configForComponent.get("speed")));
    }

    /**
     * Perform scan
     *
     * @param filename
     * @param startx
     * @param stopx
     * @param starty
     * @param stopy
     * @param dx
     * @param dy
     * @param z
     * @param speed
     * @throws DriverException
     * @throws InterruptedException
     */
    @Command(type = CommandType.QUERY, description = "scan a region")
    @SuppressWarnings("empty-statement")
    public void noStepScan(
            @Argument(name = "filename", description = "filename for storing measurement (.csv)") String filename,
            @Argument(name = "startx", description = "x start position") double startx,
            @Argument(name = "stopx", description = "x stop position") double stopx,
            @Argument(name = "dx", description = "x step size") double dx,
            @Argument(name = "starty", description = "y start position") double starty,
            @Argument(name = "stopy", description = "y stop position") double stopy,
            @Argument(name = "dy", description = "y step size") double dy,
            @Argument(name = "z", description = "z position") double z,
            @Argument(name = "speed", description = "speed") double speed
    ) throws DriverException, InterruptedException {

        LOGGER.info("Scan requested with parameters:\n"
                + " x: " + startx + "mm -> " + stopx + "mm  in steps of " + dx + "mm \n"
                + " y: " + starty + "mm -> " + stopy + "mm  in steps of " + dy + "mm \n"
                + " output to: " + filename
        );
        metrologyc.setOperatingState(MetrologyConfig.operating_states.SCANNING.ordinal());

        Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
        aeroDevc.moveAbsZ(z);
        double accel = Double.parseDouble(configForComponent.get("acceleration"));
        int nsamps = Integer.decode(configForComponent.get("nsamples"));
        int measmode = Integer.decode(configForComponent.get("measmode"));

        LOGGER.debug("Setting the acceleration to " + accel);
        aeroDevc.rampRate(accel);
        LOGGER.debug("Setting the speed to " + speed);
        aeroDevc.setSpeed(speed);

        // these can only occur in communications mode
        keyDevc.commmode();
        LOGGER.debug("Setting cycle mode to " + nsamps);
        keyDevc.setcycles(nsamps);

        LOGGER.debug("Setting surface measurement mode to " + measmode);
        keyDevc.setmeasmode(measmode);
        keyDevc.genmode();

        plot_posx.clear();
        plot_posy.clear();
        plot_posz.clear();
        move_posx.clear();
        move_posy.clear();

        if (!filename.isEmpty()) {
            FileWriter fstream = null;
            BufferedWriter out = null;

            File pdFl = new File(filename);

            try {
                if (pdFl.exists()) {
                    pdFl.delete();
                }
                pdFl.createNewFile();
            } catch (Exception e) {
                LOGGER.error("Unable to create file (" + filename + ") for reason " + e);
            }
            try {
                fstream = new FileWriter(pdFl);
            } catch (IOException e) {
                LOGGER.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
            }
            try {
                if (fstream != null) {
                    out = new BufferedWriter(fstream);
                }
            } catch (Exception e) {
                LOGGER.error("Failed to open buffered writer stream for file (" + filename + ") for reason " + e);
            }

            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# filename: " + filename + "\n"
                            + "# start time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n"
                            + "#  x: " + startx + "mm ->" + stopx + "mm  in steps of " + dx + "mm \n"
                            + "#  y: " + starty + "mm ->" + stopy + "mm  in steps of " + dy + "mm \n"
                            + "#  z: " + z + "mm \n"
                            + "#  speed: " + speed + "\n"
                            + "#  acceleration: " + accel + "\n"
                            + "#  cycle mode: " + nsamps + "\n"
                            + "#  measurement mode: " + measmode + "\n"
                            + "#  displacement summing mode: " + MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode] + "\n"
                            + "# #######################################################################\n"
                    );
                } catch (IOException ex) {
                    LOGGER.error(ex);
                }
            }

            // move to start point
            if (!abort_action) {
                aeroDevc.moveAbsXY(startx, starty);
            }
            // record start time
            double preScanStartTime_fwd = System.currentTimeMillis();
            // move to stop point
            aeroDevc.moveAbsXY(startx, stopy);
            // record stop time
            double preScanStopTime_fwd = System.currentTimeMillis();

            // move to return start point
            if (!abort_action) {
                aeroDevc.moveAbsXY(stopx, stopy);
            }
            // record start time
            double preScanStartTime_bwd = System.currentTimeMillis();
            // move to return stop point
            if (!abort_action) {
                aeroDevc.moveAbsXY(stopx, starty);
            }
            // record stop time
            double preScanStopTime_bwd = System.currentTimeMillis();

            // move to the start point
            if (!abort_action) {
                aeroDevc.moveAbsXY(startx, starty);
            }

            final double tm_starty = starty;
            final double tm_stopy = stopy;

            if (abort_action) {
                return;
            }

            int i = 0;
            for (double x = startx; x <= stopx; x += dx) {

                double y = starty;
                final double tm_x = x;
                final double tm_dy = dy;
// move to beginning of next scan line
                if (tm_dy > 0.) {
                    String msg = "move to scan line start point x,y = " + tm_x + " ," + tm_starty;
                    LOGGER.debug(msg);
                    //System.out.println(msg);
                    if (!abort_action) {
                        aeroDevc.moveAbsXY(tm_x, tm_starty);
                        move_posx.add(tm_x);
                        move_posy.add(tm_starty);
                    }
                } else {
                    String msg = "move to scan line start point x,y = " + tm_x + " ," + tm_stopy;
                    LOGGER.debug(msg);
                    //System.out.println(msg);
                    if (!abort_action) {
                        aeroDevc.moveAbsXY(tm_x, tm_stopy);
                        move_posx.add(tm_x);
                        move_posy.add(tm_stopy);
                    }
                }
// measure this position
                double xmeas = Double.NaN;
                double ymeas = Double.NaN;
                double zmeas = Double.NaN;

                while (Double.isNaN(xmeas) || Double.isNaN(ymeas) || Double.isNaN(zmeas)) {
                    double p[] = aeroDevc.getPosXYZ();
                    xmeas = p[0];
                    ymeas = p[1];
                    zmeas = p[2];
                    //System.out.println("Waiting for position information to become available.");
                }
                sleep(0.2);

                long runStartTime = System.currentTimeMillis();
                //System.out.println("Y-runStartTime = " + runStartTime);
                //System.out.println("Starting move to other endpoint.");
                Runnable tmove = new Runnable() {

                    @Override
                    public void run() {
                        try {
                            if (tm_dy > 0.) {
                                String msg = "thread move started to x,y" + tm_x + " ," + tm_stopy;
                                LOGGER.debug(msg);
                                //System.out.println(msg);
                                move_posx.add(tm_x);
                                move_posy.add(tm_stopy);
                                aeroDevc.moveAbsXY(tm_x, tm_stopy);
                            } else {
                                String msg = "thread move started to x,y" + tm_x + " ," + tm_starty;
                                LOGGER.debug(msg);
                                //System.out.println(msg);
                                move_posx.add(tm_x);
                                move_posy.add(tm_starty);
                                aeroDevc.moveAbsXY(tm_x, tm_starty);
                            }

                            LOGGER.debug("move thread terminated");
                            //System.out.println("move thread terminated");
                        } catch (DriverException e) {
                            LOGGER.error("driver exception in tmove queue", e);
                        }
                    }
                };

                Thread motion = new Thread(tmove);
                if (!abort_action) {
                    motion.start();
                }

                while (y >= starty && y <= stopy) {
                    long timeMark;
                    if (dy > 0.) {
                        timeMark = (long) ((y - starty) / (stopy - starty) * (preScanStopTime_fwd - preScanStartTime_fwd) + runStartTime);
                    } else {
                        timeMark = (long) ((y - stopy) / (starty - stopy) * (preScanStopTime_bwd - preScanStartTime_bwd) + runStartTime);
                    }
                    //System.out.println("next timeMark is " + timeMark);
                    // check if an abort has been requested
                    if (abort_action) {
                        motion.interrupt(); // stop the move
                        break;
                    }

                    // wait for the moment to read
                    //System.out.print("waiting for time to make next read");
                    while (System.currentTimeMillis() < timeMark);
                    // get the time before the displacement measurement

                    long time_before_key = System.currentTimeMillis();

                    // read displacement
                    double[] disps = keyDevc.readAll();

                    // get the time after the displacement measurement
                    long time_after_key = System.currentTimeMillis();

                    // corrected displacement
//                    double displacement = disps[1] + disps[0];
                    double displacement = -999.;
//                    switch (MetrologyConfig.Displacement_Summing_Modes.values()[disp_mode]) {
                    if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.NormalDispSum.ordinal()) {
                        if (disps[0] > -900. && disps[1] > -900.) {
                            displacement = disps[1] + disps[0];
                        }
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead1Only.ordinal()) {
                        displacement = disps[0];
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementHead2Only.ordinal()) {
                        displacement = disps[1];
                    } else if (disp_mode == MetrologyConfig.Displacement_Summing_Modes.DisplacementDiff.ordinal()) {
                        if (disps[0] > -900. && disps[1] > -900.) {
                            displacement = disps[1] - disps[0];
                        }
                    } else {
                        LOGGER.error("BAD disp_mode!");
                    }

                    // make a better estimate of the position from the actual read time
//                    double actual_time = (time_before_key + time_after_key) / 2.0; // double? Fractions of millisecs important?
                    double actual_time = time_before_key; // double? Fractions of millisecs important?
                    if (dy > 0.) {
                        // just reverse the timeMark calc. with timeMark replaced by actual_time
                        // timeMark = (long) ((y - starty) / (stopy - starty) * (preScanStopTime_fwd - preScanStartTime_fwd) + runStartTime);
                        ymeas = (actual_time - runStartTime) / (preScanStopTime_fwd - preScanStartTime_fwd) * (stopy - starty) + starty;
                    } else {
                        // timeMark = (long) ((y - stopy) / (starty - stopy) * (preScanStopTime_bwd - preScanStartTime_bwd) + runStartTime);
                        ymeas = (actual_time - runStartTime) / (preScanStopTime_bwd - preScanStartTime_bwd) * (starty - stopy) + stopy;
                    }

                    System.out.println("x= " + x + " y= " + y + " displacement= " + displacement + " xmeas= " + xmeas
                            + " ymeas= " + ymeas + " disp1= " + disps[0] + " disp2= " + disps[1]
                            + " z= " + z + " zmeas= " + zmeas
                            + " temp_a= " + last_temp_a
                            + " temp_b= " + last_temp_b
                            + " temp_c= " + last_temp_c
                            + " temp_d= " + last_temp_d
                            + " vqmpressure= " + last_vqmpressure
                            + " tstamp= " + Long.toString((long) actual_time)
                    );

                    if (displacement > -900.) {
                        plot_posx.add(xmeas);
                        plot_posy.add(ymeas);
                        plot_posz.add(displacement);

                        if (out != null) {
                            String line;
                            line = x + "," + y + "," + displacement + "," + xmeas + "," + ymeas + "," + disps[0] + "," + disps[1] + "," + z + "," + zmeas
                                    + "," + last_temp_a
                                    + "," + last_temp_b
                                    + "," + last_temp_c
                                    + "," + last_temp_d
                                    + "," + last_vqmpressure
                                    + "," + Long.toString((long) actual_time);
                            try {
                                out.write(line);
                                out.newLine();
                                out.flush();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    }

                    i++;
                    y += dy;
                }
                                
                motion.join();
// hn - commenting out so that scans are always in the same direction
//                dy = -dy; // flip directions and step back in bounds
//                y += dy;
            }
            if (out != null) {
                try {
                    out.write("# #######################################################################\n"
                            + "# end time: " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                                    System.currentTimeMillis())) + "\n");
                    out.close();
                    if (System.getenv("HOSTNAME").contains("ts5.inst.bnl")) {
                        Runtime r = Runtime.getRuntime();
                        try {
                            LOGGER.warning("Copying scan data to BNL astro GPFS");
                            Process p1 = r.exec("scp -rp " + filename + " ccdtest@astroracf:/gpfs01/astro/workarea/ccdtest/ts5/DATA/");
                            LOGGER.warning("The scan has been copied to: /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                            Process p2 = r.exec("ssh ccdtest@astroracf ls -l /gpfs01/astro/workarea/ccdtest/ts5/DATA/" + filename.split("/")[filename.split("/").length - 1]);
                        } catch (IOException ex) {
                            LOGGER.error("Scan copy failed with error " + ex);
                        }
                    }
                } catch (IOException e) {
                    LOGGER.error("Failed to close file!");
                }
            }

            if (!abort_action) {
                plotscan(180., 80., 90.);
            }

        }

        metrologyc.setOperatingState(MetrologyConfig.operating_states.IDLE.ordinal());

        abort_action = false;

    }

    /**
     * Perform repetitions - repeated measurements at the same positions
     *
     * @param filename
     * @param nreps
     * @param period
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, description = "do repeated measurements at the same positions")
    public void doreps(
            @Argument(name = "filename", description = "filename for storing measurement ") String filename,
            @Argument(name = "nreps", description = "number of repetitions") int nreps,
            @Argument(name = "period", description = "period in seconds") double period
    ) throws DriverException {

        LOGGER.info("Reps requested with parameters:\n"
                + " nreps: " + nreps + "\n"
                + " period:" + period + "\n"
                + " output to: " + filename
        );
        metrologyc.setOperatingState(MetrologyConfig.operating_states.SCANNING.ordinal());

        if (!filename.isEmpty()) {
            FileWriter fstream = null;
            BufferedWriter out = null;

            File pdFl = new File(filename);

            try {
                if (pdFl.exists()) {
                    pdFl.delete();
                }
                pdFl.createNewFile();
            } catch (Exception e) {
                LOGGER.error("Unable to create file (" + filename + ") for reason " + e);
            }
            try {
                fstream = new FileWriter(pdFl);
            } catch (IOException e) {
                LOGGER.error("Failed to open writer stream for file (" + filename + ") for reason " + e);
            }
            try {
                if (fstream != null) {
                    out = new BufferedWriter(fstream);
                }
            } catch (Exception e) {
                LOGGER.error("Failed to open buffered writer stream for file (" + filename + ") for reason " + e);
            }

            for (int i = 0; i < nreps; i++) {
                if (abort_action) {
                    break;
                }
                double[] disps = keyDevc.readAll();
//                    double displacement = keyDevc.read(2) + keyDevc.read(1);
                double displacement = disps[1] + disps[0];
                double p[] = aeroDevc.getPosXYZ();
                sleep(period);
                if (out != null) {
                    String line = i + ","
                            + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(System.currentTimeMillis()))
                            + "," + p[0] + "," + p[1] + "," + p[2]
                            + "," + disps[0] + "," + disps[1] + "," + displacement;
                    try {
                        out.write(line);
                        out.newLine();
                        out.flush();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }

            }
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                    LOGGER.error("Failed to close file!");
                }
            }

        }

        abort_action = false;
        metrologyc.setOperatingState(MetrologyConfig.operating_states.IDLE.ordinal());

    }

    /**
     * Plot the scan data
     *
     * @throws DriverException
     */
    @Command(description = "plot the scan data")
    public void plotscan() throws DriverException {
        plotscan(80., 180., 90.);
    }

    /**
     * Plot the scan data
     *
     * @throws DriverException
     */
    @Command(description = "abort motion")
    public void abortMotion() throws DriverException {
        LOGGER.error("SETTING ABORT ACTION TRUE!");
        abort_action = true; // aborts the scan
        LOGGER.error("SENDING ABORT COMMAND TO AEROTECH!");
        aeroDevc.abort(); // stop any motion
        LOGGER.error("DISABLING AXES!");
        aeroDevc.disableAxes(); // disable Axes
        LOGGER.error("ABORTING ANY ACTIVE SCRIPT!");
        sendSyncMetrologyCommand("abortInterpreter", "ts");
    }

    protected Object sendSyncMetrologyCommand(String name, Object... params) {
        try {
//            return cmu.sendSynchronousCommand(new CommandRequest(METROLOGY_DEST, name, params), 30000);
            return cmu.sendSynchronousCommand(new CommandRequest(METROLOGY_DEST, name), Duration.ofMillis(30000));

        } catch (Exception e) {
            LOGGER.warning("Unable to perform jgroup communication with destination + "
                             + METROLOGY_DEST + " - Exception " + e);
            return null;
        }
    }

    /*
     *
     * Plot the scan data
     *
     * @param rotation_x
     * @param rotation_y
     * @param rotation_z
     * @throws DriverException
     */
    @Command(description = "plot the scan data")
    public void plotscan(
            @Argument(name = "rotation_x", description = "rotation about the x axis") double rotx,
            @Argument(name = "rotation_y", description = "rotation about the y axis") double roty,
            @Argument(name = "rotation_z", description = "rotation about the z axis") double rotz
    //            @Argument(name = "plot_name", description = "name of the plot") String plotname
    ) throws DriverException {

        double vx, vy, vz;
//        int i = 0;
        double rotation_xaxis = rotx / 180. * Math.PI;
        double rotation_yaxis = roty / 180. * Math.PI;
        double rotation_zaxis = rotz / 180. * Math.PI;

        LOGGER.debug("doing new aidaplots");
//        AidaPlots plots = new AidaPlots();
        KeyValueDataList plots = new KeyValueDataList();
        LOGGER.debug("declaring pdthst");
//        plotidx = (plotidx % 3) + 1;

        IDataPointSet pdthst = dpsf.create("FlatnessScan", "TS5 Surface Scan - Date:" + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                System.currentTimeMillis())), 2);
        IDataPointSet motion = dpsf.create("Motion", "Motion profile - Date:" + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                System.currentTimeMillis())), 2);
        //("surface","surface",3,10000,"");
//        ICloud3D pdthst = cf.createCloud3D("surface");
//        ICloud2D pdthst = cf.createCloud2D("");

        Iterator<Double> posy = plot_posy.iterator();
        Iterator<Double> posz = plot_posz.iterator();
        LOGGER.debug("starting iteration over data points");
        for (Iterator<Double> posx = plot_posx.iterator(); posx.hasNext();) {
            double x = posx.next();
            double y = posy.next();
            double z = posz.next();

            LOGGER.debug("plotscan: " + x + " , " + y + " , " + z);
//            pdthst.fill(x, y, z);

            IDataPoint point = pdthst.addPoint();

//            // rotate about the y axis
//            double x1 = Math.cos(rotation_yaxis) * x + Math.sin(rotation_yaxis) * z;
//            double y1 = y;
//            double z1 = -Math.sin(rotation_yaxis) * x + Math.cos(rotation_yaxis) * z;
//            // rotate about the z axis
//            double x2 = Math.cos(rotation_zaxis) * x1 + Math.sin(rotation_zaxis) * y1;
//            double y2 = -Math.sin(rotation_zaxis) * x1 + Math.cos(rotation_zaxis) * y1;
//            double z2 = z1;
//            // rotate about the x axis
//            vx = x2;
//            vy = Math.cos(rotation_xaxis) * y2 + Math.sin(rotation_xaxis) * z2;
//            vz = -Math.sin(rotation_xaxis) * y2 + Math.cos(rotation_xaxis) * z2;
//
//            LOGGER.debug("setting coordinates");
//
//            point.coordinate(0).setValue(vx);
//            point.coordinate(0).setErrorPlus(0.);
//            point.coordinate(1).setValue(vy);
//            point.coordinate(1).setErrorPlus(0.);
//            point.coordinate(1).setErrorMinus(0.);
            
            
            LOGGER.debug("setting coordinates");

            point.coordinate(0).setValue(x);
            point.coordinate(0).setErrorPlus(0.);
            point.coordinate(1).setValue(y);
            point.coordinate(1).setErrorPlus(0.);
            point.coordinate(1).setErrorMinus(0.);
            
            //            i++;
//            LOGGER.debug("added rotated point: " + vx + " , " + vy + " , " + vz);
//             */
        }
        Iterator<Double> movey = move_posy.iterator();
        for (Iterator<Double> movex = move_posx.iterator(); movex.hasNext();) {
            IDataPoint point = motion.addPoint();
            point.coordinate(0).setValue(movex.next());
            point.coordinate(1).setValue(movey.next());            
        }

        try {
            pdthst.setTitle("Flatness Plot");
            LOGGER.debug("doing new addplots");

            //Fill the AidaPlot object with their XML representation
//            plots.addPlot("surface", XMLUtils.createXMLString((IManagedObject) pdthst));
            plots.addData("surface", XMLUtils.createXMLString((IManagedObject) pdthst), System.currentTimeMillis(), KeyValueData.KeyValueDataType.KeyValuePlotData);
            plots.addData("motion", XMLUtils.createXMLString((IManagedObject) motion), System.currentTimeMillis(), KeyValueData.KeyValueDataType.KeyValuePlotData);
        } catch (IOException ex) {
            LOGGER.error("Unable to create XML rep of AidaPlot object!", ex);
        }
        // Display the results
        IPlotter plotter = af.createPlotterFactory().create("Plot IDataPointSets");
        plotter.createRegions(1, 2);
        IPlotterStyle regionStyle = plotter.region(0).style();
        plotter.region(0).plot(pdthst);
        plotter.region(1).plot(motion);
//        regionStyle.dataStyle().lineStyle().setParameter("color", "none");
        regionStyle.dataStyle().lineStyle().setParameter("setOpacity", "0.0");
//        plotter.createRegion(.66, 0, .33, .5).plot(pdthst.projectionX("X Projection", h2d));
//        plotter.createRegion(.66, .5, .33, .5).plot(hf.projectionY("Y Projection", h2d));
//        plotter.show();

//Publish the plots somewhere in your subsystem
        LOGGER.debug("publishing plot");
//        subsys.publishStatus("ScanPlots", plots);
        KeyValueData kd = new KeyValueData("ScanPlots", plots);
        subsys.publishSubsystemDataOnStatusBus(kd);
    }

    /**
     * Initializes the aerotech subsystem. ccdDevc; BiasDevice biasDevc;
     * PhotoDiodeDevice pdDevc; CryoDevice cryoDevc; VacuumGaugeDevice vacDevc;
     * MonochromatorDevice monoDevc; Fe55Device fe55Devc; EnviroDevice envDevc;
     */
    
    @Override
    public void build() {
        
        //Define the Runnable object to be invoked periodically
        Runnable myRunnable = new Runnable() {
            @Override
            public void run() {
                //The code to be executed
                periodicBroadcast();
            }
        };

        //Create an AgentPeriodicTask that will execute the above Runnable instance
        //every second at a fixed rate
        AgentPeriodicTask periodicTask = new AgentPeriodicTask(BROADCAST_TASK, myRunnable).withPeriod(Duration.ofMillis(broadcastMillis));

        //Schedule the periodic task with the AgentPeriodicTaskService
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
        
    }
    
    @Override
    public void postInit() {
 
        // Add AgentInfo properties
        subsys.setAgentProperty(MetrologyAgentProperties.METROLOGY_TYPE_AGENT_PROPERTY,
                                MetrologySubSys.class.getCanonicalName());
        
        
        //Initialize DataPointSet factory.
        dpsf = af.createDataPointSetFactory(null);
        LOGGER.info("DPSF and Cloud Factory started.");

        teststand_dest = System.getProperty("lsst.ccs.teststand.tsguidest",
                "ts2");
        state |= MetrologyState.READY;

    }

    @Override
    public void postStart() {
        cmu = new ConcurrentMessagingUtils(subsys.getMessagingAccess());
        
        try {
            LOGGER.info("Enabling Aerotech axes");
            aeroDevc.enableAxes(); // enable axes
        } catch (DriverException ex) {
            java.util.logging.Logger.getLogger(MetrologySubSys.class.getName()).log(Level.SEVERE, null, ex);
        }

        try {
            keyDevc.commmode();
            keyDevc.setmindispunit(2); // 0.0001
            keyDevc.genmode();
        } catch (DriverException ex) {
            java.util.logging.Logger.getLogger(MetrologySubSys.class.getName()).log(Level.SEVERE, null, ex);
        }        
    }
    
    @Override
    public void start() {
       /*
         *  Initialize all configuration data
        // start the subsystem
         */
        try {
            startSubSys();
        } catch (Exception e) {
            LOGGER.error("Error starting Aerotech subsystem: " + e);
        }

        try {
            metrologyc.setOperatingState(MetrologyConfig.operating_states.IDLE.ordinal());
//            this.setStateFlags(MetrologyConfig.operating_states.RTM);
        } catch (Exception ee) {
            LOGGER.error("Exception while setting state flags:", ee);
        }
        
        
        Predicate<BusMessage<? extends Serializable, ?>> filter = BusMessageFilterFactory.messageOrigin(teststand_dest).or(BusMessageFilterFactory.messageOrigin(ts8_dest)).and(BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
        subsys.getMessagingAccess().addStatusMessageListener(this, filter);

    }

    /**
     * Performs periodic trending data broadcast.
     */
    protected void periodicBroadcast() {
        /*
         *  Broadcast the state
         */
        //System.out.print(".");

        if ((state & MetrologyState.SHUTDOWN) != 0) {
            System.exit(0); // gone in a tick
        }
        publishState();

    }

    /**
     * Handles alarm events.
     *
     * @param event
     * @param parm
     */

    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        String alarm_ev = null;
        switch (event) {
            case Alarm.EVENT_TRIP:
                alarm_ev = "TRIP";
                break;
            case Alarm.EVENT_RESET:
                alarm_ev = "RESET";
                break;
            default:
                alarm_ev = "Unknown";
        }

        LOGGER.error("Received alarm for " + alarm_ev + " event, where trip=" + Alarm.EVENT_TRIP + " and reset=" + Alarm.EVENT_RESET);
        LOGGER.error("Alarm is from " + MetrologyConfig.EVENT_ID.values()[parm]);
        return false;
    }

    @Command(name = "soundAlarm")
    public void soundAlarm() {
        Runtime r = Runtime.getRuntime();
        try {
            LOGGER.error("Sending warning sound as requested by " + name);
            Process p = r.exec("./make-warning-sound");
        } catch (IOException ex) {
            LOGGER.error("Failed to exec shell command to send warning sound " + ex);
        }
    }

    @Command(name = "soundBleep")
    public void soundBleep() {
        Runtime r = Runtime.getRuntime();
        try {
            LOGGER.severe("Sending bleep sound as requested by " + name);
            Process p = r.exec("./make-bleep-sound");
        } catch (IOException ex) {
            LOGGER.severe("Failed to exec shell command to send bleep sound " + ex);
        }
    }

    @Command(name = "resetTrip", description = "reset trips etc...")
    public void resetTrip() {
    }

    /**
     * startSubSys:
     *
     * @throws Exception
     */
    @Command(type = CommandType.ACTION, description = "Start the metrology subsystem")
    public void startSubSys() throws Exception {
        state |= MetrologyState.STARTUP;
        LOGGER.info("Starting Metrology SubSystem.");

        String param = null;
        //        Configurable.Environment env = getEnvironment();
        configInfo = subsys.getConfigurationService().getConfigurationInfo();
//        atcc.initCatcher(configInfo);
//        agentMessagingLayer = Agent.getEnvironmentMessagingAccess();
//        Future<Object> future = cmu.sendAsynchronousCommand(new CommandRequest(METROLOGY_DEST, "getConfigurationInfo"));
//        configInfo = (ConfigurationInfo) future.get();

        state |= MetrologyState.PWRDEVC_ON;

        /*
         *  Initialize the hardware
         */
        publishState();

    }

    /**
     * stopSubSys: put TS sub system into the ready state 
     */
    @Command(name = "stopsubsys", type = CommandType.ACTION,
            description = "Stop and set the Aerotech sub system into the ready state")
    public void stopSubSys() {
//        state |= MetrologyState.SHUTDOWN;
        LOGGER.info("Stop Sub System called.");

        state &= ~MetrologyState.READY;
        publishState();

        state |= MetrologyState.IDLE;
        publishState();
    }

    /**
     * Saves the configuration data.
     *
     * @throws IOException
     */
    @Command(name = "saveconfig", type = CommandType.ACTION,
            description = "Saves the current configuration")
    public void saveConfiguration() throws IOException {

        try {
            subsys.getConfigurationService().saveAllChanges();
        } catch (ConfigurationServiceException ex) {
            LOGGER.warning("Failed to save all changes " + ex);
        }

        LOGGER.info("Saved configuration.");

    }

    /**
     * Gets the full state of the aerotech module.
     *
     * This is intended to be called by GUIs during initialization
     *
     * @return The full state
     */
    @Command(type = CommandType.QUERY, description = "Gets the full aerotech state")
    public MetrologyFullState getFullState() {
        MetrologyState metrologyState = new MetrologyState(state, metrologyc.getCfgstate(), (int)periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis(), 0.0, 0.0, 0.0, 0.0, 0.0, 0, 0, 0.0, 0.0);
        MonitorFullState monState = subsys.getMonitor().getFullState();
        return new MetrologyFullState(metrologyState, monState);
    }

    /**
     * Prints the full state of aerotech
     *
     * @return The full state as a string
     */
    @Command(name = "printfullstate", type = CommandType.QUERY,
            description = "Prints the full aerotech state")
    public String printFullState() {
        MetrologyState aerotechState = new MetrologyState(state, metrologyc.getCfgstate(), (int)periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis());
//        getDevStates(aerotechState);

        MetrologyStateStatus aerotechs = new MetrologyStateStatus(aerotechState);
        return (aerotechs.toString());
    }

    /**
     * Gets the operating state word.
     *
     * @return The current value of the operating state
     */
    @Command(name = "getstate", type = CommandType.QUERY,
            description = "Returns the current value of the TS state")
    public int getState() {
        return state;
    }

    @Command(description = "Sets the configuration state.")
    public void setCfgState(int cstate) {
        LOGGER.info("Setting subsystem configuration state to " + MetrologyConfig.configuration_states.values()[cstate]);
        metrologyc.setCfgState(cstate);
    }

    @Command(description = "Sets the configuration state.")
    public void setCfgStateByName(String selectedstate) {
        int cfgstate = MetrologyConfig.configuration_states.valueOf(selectedstate).ordinal();
        LOGGER.info("Setting subsystem configuration state to " + MetrologyConfig.configuration_states.values()[cfgstate]);
        metrologyc.setCfgState(cfgstate);
    }

    /**
     * Sets the Metrology Subsystem state word.
     *
     * @param state
     */
    @Command(name = "setstate", type = CommandType.QUERY,
            description = "sets the current value of the TS state")
    public void setState(int state) {
        this.state = state;
        publishState();
    }

    /**
     * Sleep - what a waste
     *
     * @param secs The sleep time
     */
    private void sleep(double secs) {
        try {
            long millis = (long) (secs * 1000);
            LOGGER.debug("Sleeping for " + millis + " millis");
            Thread.sleep(millis);
        } catch (InterruptedException ex) {
            LOGGER.error("Rude awakening!" + ex);
        }
    }

    /**
     * Publishes the state of the aerotech This is intended to be called
     * whenever any element of the state is changed.
     */
    @Command(name = "publishState", type = CommandType.QUERY, description = "publishes the Aerotech state")
    public void publishState() {
        LOGGER.debug("Entering pubishState");
        //System.out.println("publishing");

        MetrologyState metrologyState = new MetrologyState(state, metrologyc.getOperatingState(), (int)periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis());
        getDevStates(metrologyState);
        KeyValueData kd = new KeyValueData(MetrologyState.KEY, metrologyState);
        subsys.publishSubsystemDataOnStatusBus(kd);
//        subsys.publishStatus(MetrologyState.KEY, aerotechState);
//        publish(MetrologyState.KEY, aerotechState);
        LOGGER.debug("published state");
        //System.out.println("published");
    }

    /**
     * Gets device states
     *
     * @param tst
     */
    void getDevStates(MetrologyState tst) {
        if (aeroDevc != null) {
            double posx = Double.NaN;
            double posy = Double.NaN;
            double posz = Double.NaN;
            if (metrologyc.getOperatingState() != MetrologyConfig.operating_states.SCANNING.ordinal()) {
                try {
                    while (Double.isNaN(posx) || Double.isNaN(posy) || Double.isNaN(posz)) {
                        double p[] = aeroDevc.getPosXYZ();
                        posx = p[0];
                        posy = p[1];
                        posz = p[2];
                        //System.out.println("Waiting for position information to become available.");
                        sleep(0.2);
                    }
                } catch (DriverException e) {
                    LOGGER.error("Error reading Aerotech position: " + e);
                    aeroDevc.reconnect();
                }
            } else {
                posx = aeroDevc.getLastPosX(); // avoid active query during scan                
                posy = aeroDevc.getLastPosY(); // avoid active query during scan                
                posz = aeroDevc.getLastPosZ(); // avoid active query during scan                
            }
            //System.out.println("in getdevstates: pos = " + posx + " , " + posy + " , " + posz);
            tst.setPosx(posx);
            tst.setPosy(posy);
            tst.setPosz(posz);
        }

        if (keyDevc != null) {
            if (metrologyc.getOperatingState() != MetrologyConfig.operating_states.SCANNING.ordinal()) {
                try {
                    tst.setDisplacement1(keyDevc.read(1));
                    tst.setDisplacement2(keyDevc.read(2));
                } catch (Exception e) {
                    LOGGER.debug("failed to retrieve Keyence displacement for publishing!");
                }
            } else {
                tst.setDisplacement1(keyDevc.getLast_read()[0]);
                tst.setDisplacement2(keyDevc.getLast_read()[1]);
            }
        }

        try {
            Map<String, String> configForComponent = configInfo.getCurrentValuesFor(MetrologyConfig.configuration_states.values()[metrologyc.getCfgstate()].name());
            double accel = Double.parseDouble(configForComponent.get("acceleration"));
            int nsamps = Integer.decode(configForComponent.get("nsamples"));
            int measmode = Integer.decode(configForComponent.get("measmode"));

            tst.setAcceleration(accel);
            tst.setNsamples(nsamps);
            tst.setMeasmode(measmode);
        } catch (NullPointerException ex) { // just note this exception and continue
            LOGGER.error("Null Pointer Exception - config not ready: " + ex);
        }

    }

    /**
     * Handles received status messages
     *
     * @param msg
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {

        
        if (monitorCCDTemp) {
            try {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                Object msgObject = ssd.getSubsystemData();
                if (msgObject instanceof KeyValueDataList) {
                    for (KeyValueData d : (KeyValueDataList) msgObject) {
//                    KeyValueData d = (KeyValueData) msgObject;
                        LOGGER.debug("In onDataArrival method. KEY=" + d.getKey());
                        switch (d.getKey()) {
                            case "temp_a":
                                if (!got_reb_temps) {
                                    last_temp_a = (double) d.getValue();
                                }
                                break;
                            case "temp_b":
                                if (!got_reb_temps) {
                                    last_temp_b = (double) d.getValue();
                                }
                                break;
                            case "temp_c":
                                if (!got_reb_temps) {
                                    last_temp_c = (double) d.getValue();
                                } else {
                                    last_temp_d = (double) d.getValue();
                                }
                                break;
                            case "temp_d":
                                if (!got_reb_temps) {
                                    last_temp_d = (double) d.getValue();
                                }
                                break;
                            case "vqmpressure":
                                last_vqmpressure = (double) d.getValue();
                                break;
                            case "R00.Reb0.CCDTemp1":
                                last_temp_a = (double) d.getValue();
                                got_reb_temps = true;
                                break;
                            case "R00.Reb1.CCDTemp1":
                                last_temp_b = (double) d.getValue();
                                got_reb_temps = true;
                                break;
                            case "R00.Reb2.CCDTemp1":
                                last_temp_c = (double) d.getValue();
                                got_reb_temps = true;
                                break;
                            default:
                                break;
                        }

                    }
                }

            } catch (RuntimeException e) {
                if (!e.toString().contains("de-serializing")) {
                    LOGGER.info("Problem unpacking message:" + e);
                }
            }
        }

        if (msg.getOriginAgentInfo().getName().equals(METROLOGY_DEST)) {
            //                 System.out.println("msg type = "+(msg.getObject()).toString());                         
            if (msg.getObject() instanceof ConfigurationInfo) {
                configInfo = (ConfigurationInfo) msg.getObject();
            }
        }

    }

    /**
     * Gets displacement summing mode
     *
     * @return
     */
    @Command(type = CommandType.QUERY, description = "get displacememt summing mode")
    public int getDisp_mode() {
        return disp_mode;
    }

    /**
     * Sets displacement summing mode
     *
     * @param disp_mode
     */
    @Command(type = CommandType.QUERY, description = "set displacememt summing mode")
    public void setDisp_mode(int disp_mode) {
        this.disp_mode = disp_mode;
    }

}
