package org.lsst.ccs.subsystem.mcm.shuttersim;

import java.time.Duration;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.utilities.logging.Logger;

public class ShutterSim implements HasLifecycle {

	private ShutterInternalState state = ShutterInternalState.CLOSED_OFF;

	private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(4);

	protected static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mcm");

	@LookupField(strategy = Strategy.TOP)
	Subsystem subsystem;

	@Override
	public void init() {
		subsystem.updateAgentState(state.getPublicState(), state.getPublicReadinessState());
	}

	@Command
	public void prepare() {
		log.info("ShutterSim command: prepare");
		setState(state.prepare(this));
	}

	@Command
	public void expose(long millis) {
		log.info("ShutterSim command: expose " + millis);
		setState(state.expose(this, Duration.ofMillis(millis)));
	}

	@Command
	public void close() {
		log.info("ShutterSim command: close");
		setState(state.close(this));
	}

	public long getLastEffectiveExposureTime() {
		return state.getLastEffectiveExposureTime();
	}

	public long getLastTotalOpeningTime() {
		return state.getLastTotalOpeningTime();
	}

	public void setState(ShutterInternalState s) {
		if (state == s)
			return;

		state.exit(this);
		state = s;
		state.enter(this);

		subsystem.updateAgentState(state.getPublicState(), state.getPublicReadinessState());

		log.info("ShutterSim state: " + state.getPublicState() + " " + state.getPublicReadinessState());

	}

	public ShutterInternalState getState() {
		return state;
	}

	public ScheduledFuture<ShutterInternalState> scheduleTransition(Duration delay, ShutterInternalState from,
			ShutterInternalState to) {
		log.info(" scheduling to " + to + " delay " + delay + " from " + from);
		return scheduler.schedule(() -> ShutterInternalState.scheduled(this, from, to), delay.toMillis(),
				TimeUnit.MILLISECONDS);
	}

	public void clearSchedule() {
		for (Runnable r : scheduler.getQueue()) {
			if (r instanceof ScheduledFuture<?>) {
				((ScheduledFuture) r).cancel(false);
			}
		}
	}

	Logger getLogger() {
		return log;
	}

}
