package org.lsst.ccs.subsystem.mcm.shuttersim;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;

import org.lsst.ccs.subsystem.mcm.data.InvalidStateException;
import org.lsst.ccs.subsystem.mcm.data.ShutterReadinessState;
import org.lsst.ccs.subsystem.mcm.data.ShutterState;

// like an enum but with subclasses

public abstract class ShutterInternalState implements Serializable {

	// TODO does not handle both blades moving simultaneously,
	// we don't differentiate here CLOSING and CLOSING_OPENING
	// (we are CLOSING for short exposures, losing track that the leading blade
	// is still moving)


	private static final long serialVersionUID = -5903797676508388449L;

	/**
	 * Time needed to get the shutter ready
	 */
	static private final Duration preparationTime = Duration.ofMillis(150);
	/**
	 * Time shutter can remain ready after prepare or move
	 */
	static private final Duration readyDuration = Duration.ofMillis(4000);
	/**
	 * Time needed to move a shutter blade
	 */
	static private final Duration moveTime = Duration.ofMillis(980);

	protected ShutterInternalState(String value) {
		this.value = value;
		fromValues.put(value, this);
	}

	static public ShutterInternalState valueOf(String n) {
		return fromValues.get(n);
	}

	@Override
	public String toString() {
		return value;
	}

	static private Instant startOpeningTime;
	static private Instant endOpeningTime;
	static private Instant startClosingTime;
	static private Instant endClosingTime;

	public synchronized static void setStartClosingTime(Instant startClosingTime) {
		ShutterInternalState.startClosingTime = startClosingTime;
	}

	public synchronized static void setStartOpeningTime(Instant startOpeningTime) {
		ShutterInternalState.startOpeningTime = startOpeningTime;
	}

	public synchronized static void setEndClosingTime(Instant endClosingTime) {
		ShutterInternalState.endClosingTime = endClosingTime;
	}

	public synchronized static void setEndOpeningTime(Instant endOpeningTime) {
		ShutterInternalState.endOpeningTime = endOpeningTime;
	}

	public long getLastEffectiveExposureTime() {
		throw new InvalidStateException(toString(), "cannot get effective exposure time when not closed");
	}

	public long getLastTotalOpeningTime() {
		throw new InvalidStateException(toString(), "cannot get total opening time when not closed");
	}

	public ShutterState getPublicState() {
		return ShutterState.valueOf(toString()); // internal states map directly
													// except closed
	}

	public ShutterReadinessState getPublicReadinessState() {
		return ShutterReadinessState.READY; // internal states are READY except
											// some closed
	}

	public ShutterInternalState prepare(ShutterSim s) {
		s.getLogger().error("invalid prepare command, from state " + this);
		throw new InvalidStateException("SHUTTER:" + this.toString(), "prepare");
	}

	public ShutterInternalState expose(ShutterSim s, Duration integrationTime) {
		s.getLogger().error("invalid expose command, from state " + this);
		throw new InvalidStateException("SHUTTER:" + this.toString(), "expose");
	}

	public void enter(ShutterSim s) {
		// default: do nothing
	}

	public void exit(ShutterSim s) {
		// default: do nothing
	}

	public static synchronized ShutterInternalState scheduled(ShutterSim s, ShutterInternalState from,
			ShutterInternalState to) {
		// todo check we are still in from?
		s.getLogger().debug(" scheduled: " + to);
		s.setState(to);
		return to;
	}

	public ScheduledFuture<ShutterInternalState> scheduleTransition(ShutterSim s, Duration delay,
			ShutterInternalState to) {
		return s.scheduleTransition(delay, to);
	}

	private String value;
	private static Map<String, ShutterInternalState> fromValues = new HashMap<String, ShutterInternalState>();

	public final static ShutterInternalState OPEN = new ShutterInternalState("OPEN") {
		// all generic behavior
	};

	public final static ShutterInternalState OPENING = new ShutterInternalState("OPENING") {
		@Override
		public void enter(ShutterSim s) {
			setStartOpeningTime(Instant.now());
			scheduleTransition(s, moveTime, OPEN);
		};

		@Override
		public void exit(ShutterSim s) {
			setEndOpeningTime(Instant.now());
		};

	};

	public final static ShutterInternalState CLOSING = new ShutterInternalState("CLOSING") {
		@Override
		public void enter(ShutterSim s) {
			setStartClosingTime(Instant.now());
			scheduleTransition(s, moveTime, CLOSED_ON);
		};

		@Override
		public void exit(ShutterSim s) {
			setEndClosingTime(Instant.now());
		};

	};

	// common behavior to all CLOSED substates

	public static abstract class ClosedState extends ShutterInternalState {
		protected ClosedState(String value) {
			super(value);
		}

		@Override
		public ShutterState getPublicState() {
			return ShutterState.CLOSED;
		}

		@Override
		public long getLastEffectiveExposureTime() {
			return getLastTotalOpeningTime() - startOpeningTime.until(endOpeningTime, ChronoUnit.MILLIS)
					- startClosingTime.until(endClosingTime, ChronoUnit.MILLIS);
		}

		@Override
		public long getLastTotalOpeningTime() {
			return startOpeningTime.until(endClosingTime, ChronoUnit.MILLIS);
		}
	}

	// CLOSED substates

	public final static ShutterInternalState CLOSED_OFF = new ClosedState("CLOSED_OFF") {
		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.NOT_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			return CLOSED_PREP;
		};

		@Override
		public ShutterInternalState expose(ShutterSim s, Duration integrationTime) {
			scheduleTransition(s, preparationTime, OPENING);
			scheduleTransition(s, integrationTime.plus(preparationTime).plus(moveTime), CLOSING);
			return CLOSED_PREP_EXPOSE;
		}
	};

	public final static ShutterInternalState CLOSED_PREP = new ClosedState("CLOSED_PREP") {
		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.GETTING_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			// accepted, do nothing
			return this;
		};

		@Override
		public ShutterInternalState expose(ShutterSim s, Duration integrationTime) {
			// to be more accurate, preparationTime should be remaining time
			// before ready
			Duration remaining = Duration.ofMillis(Instant.now().until(readyTime, ChronoUnit.MILLIS));
			s.getLogger()
					.info("ShutterSim: expose while preparing, remaining prep time " + remaining.toMillis() + " ms.");
			remaining = remaining.plus(Duration.ofMillis(5));
			scheduleTransition(s, remaining, OPENING);
			scheduleTransition(s, integrationTime.plus(remaining).plus(moveTime), CLOSING);
			return this;
		}

		@Override
		public void enter(ShutterSim s) {
			scheduleTransition(s, preparationTime, CLOSED_ON);
			readyTime = Instant.now().plus(preparationTime);
			s.getLogger().info("ShutterSim will be ready at " + readyTime);
		};

		private java.time.Instant readyTime;

	};

	public final static ShutterInternalState CLOSED_PREP_EXPOSE = new ClosedState("CLOSED_PREP_EXPOSE") {
		@Override
		public ShutterReadinessState getPublicReadinessState() {
			return ShutterReadinessState.GETTING_READY;
		};

		@Override
		public ShutterInternalState prepare(ShutterSim s) {
			// accepted, do nothing
			return this;
		};

	};

	public final static ShutterInternalState CLOSED_ON = new ClosedState("CLOSED_ON") {

		private transient ScheduledFuture<ShutterInternalState> powerOffTransition = null;

		@Override
		public ShutterInternalState expose(ShutterSim s, Duration integrationTime) {
			scheduleTransition(s, integrationTime.plus(moveTime), CLOSING);
			return OPENING;

		}

		@Override
		public synchronized ShutterInternalState prepare(ShutterSim s) {
			// we stay prepared, reset the timer
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = scheduleTransition(s, readyDuration, CLOSED_OFF);
			return CLOSED_ON;
		};

		@Override
		public synchronized void enter(ShutterSim s) {
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = scheduleTransition(s, readyDuration, CLOSED_OFF);

			if (endClosingTime != null)
				s.getLogger().info("ShutterSim: exposure time " + super.getLastEffectiveExposureTime() + " total "
						+ super.getLastTotalOpeningTime());
		};

		@Override
		public synchronized void exit(ShutterSim s) {
			if (powerOffTransition != null)
				powerOffTransition.cancel(false);
			powerOffTransition = null;
		}

	};

}
