package org.lsst.ccs.subsystem.mcm.raftsim;

import java.time.Duration;
import java.util.Random;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.mcm.data.RaftState;
import org.lsst.ccs.subsystem.mcm.data.RaftStatus;
import org.lsst.ccs.utilities.logging.Logger;

public class RaftSim extends Module {

	private RaftInternalState state = RaftInternalState.NEEDS_CLEAR;

	private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(
			4);

	protected final static Logger log = Logger
			.getLogger("org.lsst.ccs.subsystem.mcm");

	public RaftSim(String name, int tickMillis) {
		super(name, tickMillis);
	}

	@Override
	public void initModule() {
		getSubsystem().updateAgentState("", getPublicState());
	}

	// commands are delegated to the states (GoF State pattern)

	@Command
	public void clear() {
		log.info("RaftSim command: clear");
		setState(state.clear(this));
	}

	@Command
	public void integrate() {
		log.info("RaftSim command: integrate");
		setState(state.integrate(this));
	}

	@Command
	public void readout() {
		log.info("RaftSim command: readout");
		setState(state.readout(this));
	}

	@Command
	public void triggerAlarm() {
		getSubsystem().raiseAlert(new Alert("RAFT001", "raft alarm"),
				AlertState.ALARM);
	}

	@Command
	public void clearAlarm() {
		getSubsystem().clearAlerts("RAFT001");
	}

	// setting a state

	RaftState getPublicState() {
		return RaftState.valueOf(state.toString());
	}

	public void setState(RaftInternalState s) {
		if (state == s)
			return;

		state.exit(this);
		state = s;
		state.enter(this);

		getSubsystem().updateAgentState("", state, getPublicState());

		log.info("RaftSim state: " + getPublicState());
	}

	public ScheduledFuture<RaftInternalState> scheduleTransition(
			Duration delay, RaftInternalState to) {
		log.debug(" scheduling to " + to + " delay " + delay);
		return scheduler.schedule(
				() -> RaftInternalState.scheduled(this, state, to),
				delay.toMillis(), TimeUnit.MILLISECONDS);
	}

	Logger getLogger() {
		return log;
	}

	private double temp = -100;
	private double dtemp = 0;
	private double d2temp = 0;

	private Random random = new Random();

	@Override
	public void tick() {
		d2temp = 1 - 2 * random.nextDouble();
		d2temp /= 5;
		if (temp < -110) {
			d2temp += .2;
			dtemp = 0;
		} else if (temp > -90) {
			d2temp += -.2;
			dtemp = 0;
		}

		dtemp += d2temp;

		temp += dtemp;
		RaftStatus s = new RaftStatus();
		s.setTemperature(temp);
		log.debug("publish temp " + temp + "(" + dtemp + "," + d2temp + ")");
                KeyValueData d = new KeyValueData("raftStatus",s);
		getSubsystem().publishSubsystemDataOnStatusBus(d);

	}
}
