package org.lsst.ccs.subsystem.mcm;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusEnum;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.mcm.data.MCMIR2Event;
import org.lsst.ccs.utilities.logging.Logger;

public abstract class GenericMCM<MinionT extends Enum<MinionT>, GroupT extends Enum<GroupT>, EventT extends Enum<EventT>, StateT extends Enum<StateT>, CommandT extends Enum<CommandT>>
        extends Subsystem implements HasLifecycle {

    protected MCMUtilities<MinionT, GroupT, EventT> mu;

    Map<StateT, Set<CommandT>> allowedTransition = new HashMap<>();
    protected Map<GroupT,Map<String, MinionT>> minions;

    @LookupField(strategy = Strategy.TOP)
    Subsystem subsystem;

    @LookupField(strategy = Strategy.CHILDREN)
    protected Map<String, AlertDispatcher> dispatchers = new HashMap<>();

    @Override
    public void init() {
        mu = new MCMUtilities<>(subsystem);

        initAllowedTransitions();
        
        for ( Entry<GroupT,Map<String,MinionT>> e : minions.entrySet() ) {
            GroupT group = e.getKey();
            for ( Entry<String,MinionT> e1 : e.getValue().entrySet() ) {
                mu.addMinion(group, e1.getValue(), e1.getKey());
            }
        }
        
        subsystem.addCommandsFromObject(mu, "");
        
        initMCM();
        
    }
        
    @Override
    public void start() {
        mu.init();
        mu.addAlertObserver(this::onAlert);
        mu.activate();
    }

    protected MCMUtilities getMcmUtilities() {
        return mu;
    }

    protected abstract void initAllowedTransitions();

    protected abstract void initMCM();

    protected Random random = new Random();
    protected static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mcm");

    protected void setState(StateT s) {
        log.info("MCM State " + s);
        subsystem.updateAgentState(s);
//        mu.addAlertObserver(this::onAlert);
//        mu.activate();
    }

    protected abstract Class<StateT> getStateClass();

    protected void checkCommandValidity(CommandT cmd) {
        StateT state = (StateT) subsystem.getState(getStateClass());
        if (!allowedTransition.get(state).contains(cmd)) {
            log.error("Command " + cmd + " not allowed in state " + state);
            throw new RuntimeException("Command " + cmd + " not allowed in state " + state);
        }
    }

    public GenericMCM() {
        super("genericMCM",AgentInfo.AgentType.MCM);
    }

    public void onAlert(AlertNotification notif) {
        log.warn(notif+" "+notif.getOrigin()+" "+notif.getSubsystemType()+" "+notif.getSubsystemGroup());
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            AlertDispatcher ad = e.getValue();
            boolean affectsDispatcher = false;

            //Check if the message is from the MCM and decide if its group applies to this dispatcher
            if ( notif.getOrigin().equals(subsystem.getName()) ) {
                String group = (String)notif.getAlert().getAlertData("group");
                if ( ad.getGroupName().equals(group) ) {
                    affectsDispatcher = true;
                }
            }
            
            if ( affectsDispatcher || ad.getGroup().equals(notif.getSubsystemGroup())) {
                e.getValue().onAlert(notif);
            }
        }
    }

    @Command
    public String[] getAlertDispatchers() {
        return dispatchers.keySet().toArray(new String[0]);
    }
    
    @Command
    public String status() {
        StringBuilder sb = new StringBuilder();
        sb.append("Dispatchers: \n");
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            sb.append("\n");
            sb.append(e.getValue().status("\t"));
        }
        return sb.toString();
    }
    

    public void publishEvent(MCMIR2Event e) {
        StatusEnum<MCMIR2Event> message = new StatusEnum<MCMIR2Event>(e, subsystem.getState());
        subsystem.getMessagingAccess().sendStatusMessage(message);
    }

    public Object send(GroupT g, MinionT dst, String command, Object... parms) throws Exception {
        return mu.send(g, dst, command, parms);
    }

    public Object sendLongCommand(GroupT g, MinionT dst, long duration, String command, Object... parms) throws Exception {
        return mu.sendLongCommand(g, dst, duration, command, parms);
    }

    public Future<Object> sendAsync(GroupT g, MinionT dst, String command, Object... parms) {
        return mu.sendAsync(g, dst, command, parms);
    }

    public <MinionStateT extends Enum<MinionStateT>> Future<StatusMessage> watchForState(GroupT g, MinionT sys,
            MinionStateT state) {
        return mu.watchForState(g, sys, state);
    }

    public <MinionStateT extends Enum<MinionStateT>> void waitForState(GroupT g, MinionT sys, MinionStateT state, long timeout) {
        mu.waitForState(g, sys, state, timeout);
    }

    public void waitMillis(long millis) {
        mu.waitMillis(millis);
    }

    public <MinionStateT extends Enum<MinionStateT>> void checkState(GroupT g, MinionT sys, MinionStateT state) {
        mu.checkState(g, sys, state);
    }

    @SafeVarargs
    public final <MinionStateT extends Enum<MinionStateT>> void checkState(GroupT g, MinionT sys, MinionStateT... state) {
        mu.checkState(g, sys, state);
    }

    public <MinionStateT extends Enum<MinionStateT>> boolean isInState(GroupT g, MinionT sys, MinionStateT state) {
        return mu.isInState(g, sys, state);
    }

    public <MinionStateT extends Enum<MinionStateT>> MCMUtilities<MinionT, GroupT, EventT>.ExpectedStateCombination<MinionStateT> expectingState(
            GroupT g, MinionT m, MinionStateT state) {
        return mu.expectingState(g, m, state);
    }

    @SafeVarargs
    public final void setAbortingOnAlarmMinions(GroupT g, MinionT... m) {
        mu.setAbortingOnAlarmMinions(g, m);
    }

    public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
        return mu.schedule(r, delay);
    }

    public Future<?> execute(Runnable r) {
        return mu.execute(r);
    }

}
