package org.lsst.ccs.subsystem.mcm.ir2.actions;

import java.util.Map;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.subsystem.mcm.AlertDispatcher;
import org.lsst.ccs.subsystem.mcm.AlertNotification;
import org.lsst.ccs.subsystem.mcm.MCM_IR2;
import org.lsst.ccs.subsystem.mcm.MinionIR2;
import org.lsst.ccs.subsystem.rafts.alerts.RaftAlert;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 * An Alert Action to deal with Reb Temperature getting too high
 *
 * @author The LSST CCS Team
 */
public class FocalPlaneRebTemperatureTooHighAction extends AlertDispatcher.AlertAction {

    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    AlertDispatcher dispatcher;
    
    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    MCM_IR2 mcm;
        
    @Override
    public void accept(AlertNotification t) {
        if (t.getSubsystemType() == MinionIR2.FOCAL_PLANE) {
            String alertId = t.getAlert().getAlertId();
            if (alertId.equals(RaftAlert.ASPICS_TEMPERATURE_TOO_HIGH.getAlertId())
                    || alertId.equals(RaftAlert.REB_BOARD_TEMPERATURE_TOO_HIGH.getAlertId())) {
                getLogger().info("REB temperature too high action required at level: " + t.getSeverity()+" ("+t.getCause()+")");
                

                String destination = dispatcher.getMcmUtilities().getDestination(dispatcher.getGroup(), MinionIR2.POWER);
                
                StateBundle powerStateBundle = dispatcher.getMcmUtilities().getState(dispatcher.getGroup(), MinionIR2.POWER);
                if ( powerStateBundle != null ) {
                Map<String,RebPowerState> powerStates = powerStateBundle.getComponentsWithState(RebPowerState.class);
                getLogger().info("Overall Power state: "+powerStateBundle.getState(RebPowerState.class));
                getLogger().info("Component Power state "+powerStates);
                
                if ( t.getSeverity() == AlertState.ALARM ) {
                    getLogger().warn("********** Powering off the rebs for group "+dispatcher.getGroupName());

                    for ( int i = 0; i < 6; i++ ) {
                        String command = "sequencePower "+i+" false";
                        getLogger().warn("Sending command \""+command+"\" to "+destination);
                        try {
                            dispatcher.getMcmUtilities().send(destination, "sequencePower", i, false);
                        } catch (Exception e) {
                            getLogger().severe("FAILED TO TURN OFF REB "+i,e);
                        }
                    }
                }                
                } else {
                    getLogger().warning("No state information is available for "+destination+". The subsystem might not be available on the buses.");
                    getLogger().warning("The Reb Temperature Too High Action is skipped!");
                }
            }
        }
    }

    @Override
    public String getDescription() {
        return "Powers off the Rebs upon ALARM by sending sequencePower i false to "+dispatcher.getMcmUtilities().getDestination(dispatcher.getGroup(), MinionIR2.POWER);
    }
    
    
}
