package org.lsst.ccs.subsystem.mcm;

import java.util.ArrayList;
import java.util.logging.Level;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertEvent.AlertEventType;
import org.lsst.ccs.services.alert.AlertService;

public class AlertNotifier<MinionT extends Enum<MinionT>, GroupT extends Enum<GroupT>> {

    static Logger log = Logger.getLogger("lsst.ccs.bus");

    // Map containing the most recend RaisedAlertSummary for all agents on the buses
    private final Object updateInProgress = new Object();

    private final MCMUtilities<MinionT, GroupT, ?> mcmUtils;
    private final AlertService alertService;

    public AlertNotifier(MCMUtilities mcmUtils, Agent agent) {
        this.mcmUtils = mcmUtils;
        alertService = agent.getAgentService(AlertService.class);
        alertService.addListener((event) -> processEvent(event));
    }

    private void processEvent(AlertEvent event) {
        if (event.getType()==AlertEventType.ALERT_RAISED) {
            String origin = event.getSource();

            // is this me?
            // FIXME find a better way
            if (origin.contains("MCM")) {
                log.info("Got a raised alert message from "+origin+" -- ignored, source = MCM");
                return;
            }
            GroupT g = mcmUtils.getSubsystemGroup(origin);
            try {
                MinionT m = mcmUtils.getSubsystemType(origin);
                if ( m != null ) {
                Alert alert = event.getAlert();
                log.info("Got a raised alert message from "+origin);

                RaisedAlertHistory hist = event.getSummary().getRaisedAlert(alert.getAlertId());
                notifyObservers(new AlertNotification(origin, g, m, hist.getLatestAlertCause(), hist.getLatestAlert(),
                        hist.getLatestAlertState(), hist.getLatestAlertCCSTimeStamp().getUTCInstant().toEpochMilli()));
                } else {
                    log.log(Level.WARNING,"Cannot process alert from origin {0} ({1})",new Object[]{origin, event.getAlert()});
                }
            } catch (MCMUtilities.UnknownMinionException e) {
                log.warning("Could not process event "+event, e);
            }
        }
    }

    public static interface Observer {
        public void alertRaised(AlertNotification notif);
    }

    ArrayList<Observer> observers = new ArrayList<>();

    public void addObserver(Observer o) {
        synchronized (updateInProgress) {
            observers.add(o);
        }
    }

    public void deleteObserver(Observer o) {
        synchronized (updateInProgress) {
            observers.remove(o);
        }
    }

    private void notifyObservers(AlertNotification notif) {
        ArrayList<Observer> obs;
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, notif);
            });
        }
    }

    private void notifyObservers(String origin, RaisedAlertSummary summary) {
        ArrayList<Observer> obs;
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, origin, summary);
            });
        }
    }

    private void notifyObserver(Observer o, AlertNotification notif) {
        o.alertRaised(notif);
    }

    private void notifyObserver(Observer o, String origin, RaisedAlertSummary summary) {
        synchronized (updateInProgress) {
            for (RaisedAlertHistory hist : summary.getAllRaisedAlertHistories()) {
                if (hist.getLatestAlertState()!=AlertState.NOMINAL) {
                    try {
                        GroupT g = mcmUtils.getSubsystemGroup(origin);
                        MinionT m = mcmUtils.getSubsystemType(origin);
                        notifyObserver(o, new AlertNotification(origin, g, m, hist.getLatestAlertCause(),
                                hist.getLatestAlert(), hist.getLatestAlertState(), hist.getLatestAlertCCSTimeStamp().getUTCInstant().toEpochMilli()));
                    } catch (Exception e) {
                        log.warning("Could not process alert notification from "+origin, e);
                    }
                }
            }
        }
    }
}
