package org.lsst.ccs.subsystem.mcm;

import java.util.ArrayList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

public class AlertNotifier<MinionT extends Enum<MinionT>, GroupT extends Enum<GroupT>> implements StatusMessageListener, AgentPresenceListener {

    static Logger log = Logger.getLogger("lsst.ccs.bus");

    //Map containing the most recend RaisedAlertSummary for all agents on the buses
    private Map<String,RaisedAlertSummary> raisedAlertMap = new ConcurrentHashMap<>();
    private final Object updateInProgress = new Object(); 

    private final ConcurrentMessagingUtils cmu;
    private final Agent a;
    private final MCMUtilities<MinionT,GroupT,?> mcmUtils;
    
    public AlertNotifier(MCMUtilities mcmUtils, Agent agent) {
        this.cmu = new ConcurrentMessagingUtils(agent.getMessagingAccess());
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        this.a = agent;
        this.mcmUtils = mcmUtils;
    }
    
    
    public static interface Observer {
        public void alertRaised(AlertNotification notif);
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {

        //Ignore messages coming from consoles or listeners
        if ( msg.getOriginAgentInfo().getType().compareTo(AgentInfo.AgentType.WORKER) < 0 ) {
            return;
        }
        
        //Check if we have already all the information we need for
        //the agent that published the message
        String origin = msg.getOriginAgentInfo().getName();        
        synchronized(updateInProgress) {
            if (!raisedAlertMap.containsKey(origin)) {
                //No information for this agent. 
                //It needs to be updated to cover the case in which the this agent
                //just started and there are other subsystem with outstanding uncleared
                //alerts
                log.info("Need to fetch alert information for " + origin);
                RaisedAlertSummary summary = null;
                if (origin.equals(a.getName())) {
                    summary = a.getAlertService().getRaisedAlertSummary();
                } else {
                    CommandRequest cmd = new CommandRequest(origin, "getRaisedAlertSummary", null);
                    try {
                        summary = (RaisedAlertSummary) cmu.sendSynchronousCommand(cmd);
                    } catch (Exception e) {
                        log.severe("Problem fetching raised alert summary for " + origin, e);
                    }
                }
                if ( summary != null ) {
                    raisedAlertMap.put(origin, summary);
                    //Disabling this for now to avoid notifications to be sent on past
                    //history. We should have a way to tell which observers want
                    //this kind of notification to be activated.
//                    notifyObservers(origin, summary);                    
                } 
            }

            //If the message is a StatusAlert we have to update the
            //RaisedAlert information we are storing
            if (msg instanceof StatusAlert) {
                raisedAlertMap.put(origin, ((StatusAlert) msg).getRaisedAlertSummary());
            }
        }
        
        //If the message is not a StatusRaisedAlert continue
        if (!(msg instanceof StatusRaisedAlert))
            return;
        Object o = msg.getObject();

        log.info("Got a raised alert message from "+origin);
        
        if (!(o instanceof Alert)) {
            log.severe("How did we get here? Why is the content of a StatusRaisedAlert not an Alert???? "+o.getClass());
            return;
        }


        Alert alert = (Alert) o;
        AlertState severity = ((StatusRaisedAlert) msg).getRaisedAlertSummary().getRaisedAlert(alert.getAlertId())
                .getLatestAlertState();
        String cause = ((StatusRaisedAlert) msg).getCause();

        try {
            GroupT g = mcmUtils.getSubsystemGroup(origin);
            MinionT m = mcmUtils.getSubsystemType(origin);
            notifyObservers(new AlertNotification(origin, g, m, cause, alert, severity, msg.getTimeStamp()));
        } catch (Exception e) {
            log.warning("Could not process alert notification from " + origin, e);
        }


    }

    @Override
    public void connecting(AgentInfo agent) {
    }

    //Remove an agent's information once it disconnects
    @Override
    public void disconnecting(AgentInfo agent) {
        synchronized(updateInProgress) {
            log.info("Agent "+agent.getName()+" has disconnected. Removing the corresponding alert history");
            raisedAlertMap.remove(agent.getName());
        }
    }

    
    
    
    ArrayList<Observer> observers = new ArrayList<>();

    public void addObserver(Observer o) {
        synchronized(updateInProgress) {
            observers.add(o);
                    //Disabling this for now to avoid notifications to be sent on past
                    //history. We should have a way to tell which observers want
                    //this kind of notification to be activated.
//            updateObserver(o);
        }
    }

    public void deleteObserver(Observer o) {
        synchronized(updateInProgress) {
            observers.remove(o);
        }
    }

    private void notifyObservers(AlertNotification notif) {
        ArrayList<Observer> obs;        
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, notif);
            });
        }
    }
    
    private void notifyObservers(String origin, RaisedAlertSummary summary) {
        ArrayList<Observer> obs;        
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, origin, summary);
            });
        }
    }
    
    
    private void notifyObserver( Observer o, AlertNotification notif) {
        o.alertRaised(notif);
    }

    private void notifyObserver( Observer o, String origin, RaisedAlertSummary summary) {
        synchronized(updateInProgress) {
            for ( RaisedAlertHistory hist: summary.getAllRaisedAlertHistories() )  {
                if ( hist.getLatestAlertState() != AlertState.NOMINAL ) {
                    try {
                        GroupT g = mcmUtils.getSubsystemGroup(origin);
                        MinionT m = mcmUtils.getSubsystemType(origin);
                        notifyObserver(o, new AlertNotification(origin, g, m, hist.getLatestAlertCause(), hist.getLatestAlert(), hist.getLatestAlertState(), hist.getLatestAlertTimestamp()));
                    } catch (Exception e) {
                        log.warning("Could not process alert notification from "+origin, e);                        
                    }
                }
            }
        }
    }

    
    private void updateObserver(Observer o) {
        synchronized(updateInProgress) {
            for ( Entry<String, RaisedAlertSummary> e : raisedAlertMap.entrySet() ) {
                notifyObserver(o, e.getKey(), e.getValue());
            }
        }
    }
    

}
