package org.lsst.ccs.subsystem.mcm;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;

import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusEnum;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StateBundleAggregator;
import org.lsst.ccs.messaging.StatusAggregator;
import org.lsst.ccs.subsystem.mcm.data.CCSOperationalState;
import org.lsst.ccs.subsystem.mcm.data.CCSSalState;
import org.lsst.ccs.subsystem.mcm.data.CameraOperationalState;
import org.lsst.ccs.subsystem.mcm.data.InvalidStateException;
import org.lsst.ccs.subsystem.mcm.data.MCMEvent;
import org.lsst.ccs.subsystem.mcm.data.OperationTimeoutException;
import org.lsst.ccs.subsystem.mcm.data.RaftState;
import org.lsst.ccs.subsystem.mcm.data.ShutterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.utilities.logging.Logger;

public final class MCMUtilities implements AgentPresenceListener {

	private ConcurrentMessagingUtils cmu;
	private AgentPresenceManager apm;
	private StatusAggregator sa = new StatusAggregator();
	private StateBundleAggregator sba = new StateBundleAggregator();

	public StatusAggregator getStatusAggregator() {
		return sa;
	}

	// As a subsystem, the MCM has its own set of states, OperationalState for
	// instance.
	// It also publishes an aggregate OperationalState for the whole camera.

	// todo map to states expected by OCS, to be translated by the OCS bridge

	private long defaultTimeout = 1000L;

	private Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mcm");

	private Agent mcm;
	private AgentMessagingLayer agentMessagingLayer;

	public MCMUtilities(Agent agent) {
		mcm = agent;
		agentMessagingLayer = agent.getMessagingAccess();
		cmu = new ConcurrentMessagingUtils(agentMessagingLayer);
		apm = agentMessagingLayer.getAgentPresenceManager();
		apm.addAgentPresenceListener(this);
		agentMessagingLayer.addStatusMessageListener(sa);
		agentMessagingLayer.addStatusMessageListener(sba);

		// todo configure this in a more dynamical way
		minionNames.put(Minion.RAFTS, "raftsim");
		minionNames.put(Minion.SHUTTER, "shuttersim");
		minionNames.put(Minion.FILTER, "filtersim");

		for (String s : minionNames.values()) {
			sba.addOrigin(s);
		}

		for (Map.Entry<Minion, String> e : minionNames.entrySet()) {
			name2minion.put(e.getValue(), e.getKey());
		}

		for (AgentInfo i : apm.listConnectedAgents()) {
			Minion m = name2minion.get(i.getName());
			if (m != null)
				presentMinions.add(m);
		}

		// todo/tocheck we should not too quickly report an error when we miss
		// information
		execute(() -> {
			waitMillis(5000);
			checkCamOpState();
			checkPresence();
		});

		sba.addObserver((source, old, change) -> minionStateChange(source, old, change));

		initEventRules();

		sa.setAggregate("raftsim/temperature", 10000, 60000);
	}

	// todo keep a list of the subsystems we are dealing with, along with their
	// name

	private final Map<Minion, String> minionNames = new EnumMap<Minion, String>(Minion.class);
	private final Map<String, Minion> name2minion = new HashMap<String, Minion>();

	private final Set<Minion> presentMinions = new HashSet<Minion>();

	// todo if a required subsystem is not present it should be reflected in our
	// state

	// todo more specific exceptions?

	public Object send(Minion dst, String command, Object... parms) throws Exception {
		CommandRequest cmd = new CommandRequest(minionNames.get(dst), command, parms);
		return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(defaultTimeout));
	}

	/**
	 * To send a command which lasts more than defaultTimeout. Used to send
	 * command to fcs subsystem (FILTER).
	 * 
	 * @param dst
	 * @param command
	 * @param timeout
	 * @param parms
	 * @return
	 * @throws Exception
	 */
	public Object sendLongCommand(Minion dst, long timeout, String command, Object... parms) throws Exception {
		CommandRequest cmd = new CommandRequest(minionNames.get(dst), command, parms);
		return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(timeout));
	}

	public Future<Object> sendAsync(Minion dst, String command, Object... parms) {
		CommandRequest cmd = new CommandRequest(minionNames.get(dst), command, parms);
		return cmu.sendAsynchronousCommand(cmd);
	}

	private final Set<Minion> abortingOnAlarmMinions = new HashSet<Minion>();

	public void setAbortingOnAlarmMinions(Minion... m) {
		abortingOnAlarmMinions.clear();
		abortingOnAlarmMinions.addAll(Arrays.asList(m));
		createAlarmFilter();
	}

	private Predicate<BusMessage<? extends Serializable, ?>> alarmFilter;

	private void createAlarmFilter() {
		alarmFilter = abortingOnAlarmMinions.stream()
				.map(mm -> BusMessageFilterFactory.messageOrigin(minionNames.get(mm))).reduce(Predicate::and)
				.orElse(x -> false).and(BusMessageFilterFactory.embeddedObjectClass(Alert.class));
	}

	private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(4);

	public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
		return scheduler.schedule(r, delay.toMillis(), TimeUnit.MILLISECONDS);
	}

	public Future<?> execute(Runnable r) {
		return scheduler.submit(r);
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public Future<StatusMessage> watchForState(Minion sys, Enum state) {
		Predicate<BusMessage<? extends Serializable, ?>> f = BusMessageFilterFactory.messageOrigin(minionNames.get(sys))
				.and(BusMessageFilterFactory.messageClass(StatusMessage.class));

		Predicate<BusMessage<? extends Serializable, ?>> p = (m) -> ((StatusMessage) m).getState().isInState(state);
		f = f.and(p);

		for (Minion m : abortingOnAlarmMinions) {
			Predicate<BusMessage<? extends Serializable, ?>> x = BusMessageFilterFactory
					.embeddedObjectClass(Alert.class).and(BusMessageFilterFactory.messageOrigin(minionNames.get(m)));
			f = f.or(x);
		}

		return cmu.startListeningForStatusBusMessage(f);
	}

	public <T extends Enum<T>> void checkState(Minion sys, T state) {
		StateBundle current = sba.getState(minionNames.get(sys));
		if (current == null || !current.isInState(state)) {
			throw new InvalidStateException(current == null ? "null" : current.toString(), " expecting " + state);
		}
	}

	@SafeVarargs
	public final <T extends Enum<T>> void checkState(Minion sys, T... state) {
		StateBundle current = sba.getState(minionNames.get(sys));
		if (current != null) {
			for (T s : state) {
				if (current.isInState(s))
					return;
			}
		}
		StringBuilder expected = new StringBuilder("current state ");
		expected.append(current == null ? "null" : current.toString());
		expected.append(" expecting one of [");
		for (int i = 0; i < state.length; i++) {
			expected.append(state[i].toString());
			if (i < state.length - 1)
				expected.append(", ");
		}
		expected.append("]");
		throw new InvalidStateException(expected.toString());
	}

	public <T extends Enum<T>> boolean isInState(Minion sys, T state) {
		StateBundle current = sba.getState(minionNames.get(sys));
		if (current == null)
			return false;
		return current.isInState(state);
	}

	@SuppressWarnings("rawtypes")
	public <T extends Enum<T>> void waitForState(Minion sys, T state, long timeout) {
		Future<StatusMessage> f = watchForState(sys, state);

		StateBundle current = sba.getState(minionNames.get(sys));
		if (current != null && current.isInState(state)) {
			log.info("state for " + sys + " already ok for " + state);
			return;
		}
		log.debug("current state for " + sys + " : " + current);
		try {
			StatusMessage m = f.get(timeout, TimeUnit.MILLISECONDS);
			if (m != null)
				log.debug("received " + m.getState());
			if (m == null) {
				if (sba.getState(minionNames.get(sys)).isInState(state)) {
					log.info("null future, but in state");
				} else {
					log.info("null future, not in state");
					log.info(sba.getState(minionNames.get(sys)));
					throw new OperationTimeoutException("waiting for state " + state + " on " + sys);
				}
			}
			if (m instanceof StatusRaisedAlert) {
				throw new AlarmException("interrupted waitForState", ((StatusRaisedAlert) m).getObject());
			}
		} catch (TimeoutException e) {
			throw new OperationTimeoutException("waiting for state " + state + " on " + sys, e);
		} catch (InterruptedException | ExecutionException e) {
			throw new RuntimeException(e);
		}
	}

	public <T extends Enum<T>> ExpectedStateCombination expectingState(Minion m, T state) {
		return new ExpectedStateCombination(m, state);
	}

	public final class ExpectedStateCombination {
		@SuppressWarnings("rawtypes")
		private Map<Minion, Enum> states = new EnumMap<Minion, Enum>(Minion.class);

		private ExpectedStateCombination(Minion m, Enum<?> state) {
			states.put(m, state);
		}

		private ExpectedStateCombination(ExpectedStateCombination src, Minion m, Enum<?> state) {
			states.putAll(src.states);
			states.put(m, state);
		}

		public ExpectedStateCombination expectingState(Minion m, Enum<?> state) {
			return new ExpectedStateCombination(this, m, state);
		}

		private <T extends Enum<T>> Predicate<BusMessage<? extends Serializable, ?>> filter(Minion sys, T state) {
			return BusMessageFilterFactory.messageOrigin(minionNames.get(sys))
					.and(m -> ((StatusMessage<?, ?>) m).getState().isInState(state));
		}

		private Predicate<BusMessage<? extends Serializable, ?>> createFilter(Minion sys) {
			return filter(sys, states.get(sys)).or(alarmFilter);
		}

		public void waitForAllStatesHappening(long timeout) {

			Set<Minion> waitingFor = new HashSet<Minion>(states.keySet());

			for (Map.Entry<Minion, Enum> e : states.entrySet()) {
				if (sba.getState(minionNames.get(e.getKey())).isInState(e.getValue())) {
					waitingFor.remove(e.getKey());
				}
			}

			log.debug("waiting for subsystems " + waitingFor + " timeout " + timeout);

			if (waitingFor.isEmpty()) {
				log.debug("waitForAllStatesHappening: already all in state");
				return;
			}

			int timeoutseconds = timeout > 1000 ? (int) timeout / 1000 : 1;
			// TODO the cmu should have timeouts in milliseconds, or use
			// Duration

			Set<Future<StatusMessage>> futures = new HashSet<Future<StatusMessage>>();
			for (Minion m : waitingFor) {
				Predicate<BusMessage<? extends Serializable, ?>> f = createFilter(m);
				futures.add(cmu.startListeningForStatusBusMessage(f, Duration.ofMillis(timeoutseconds)));
			}

			// futures.stream().parallel().forEach(f -> f.get()); // Damn
			// checked exceptions...

			for (Future<StatusMessage> f : futures) {
				try {
					/* StatusMessage m = */f.get();
				} catch (InterruptedException | ExecutionException e) {
					log.warn("interrupted");
					throw new RuntimeException(e);
				}
			}
		}

		public void waitForAllStates(long timeout) {
			long endTimeout = System.currentTimeMillis() + timeout;
			outer: while (System.currentTimeMillis() < endTimeout) {
				waitForAllStatesHappening(endTimeout - System.currentTimeMillis());

				for (Map.Entry<Minion, Enum> e : states.entrySet()) {
					if (!sba.getState(minionNames.get(e.getKey())).isInState(e.getValue()))
						continue outer;
				}

				return;
			}
			throw new OperationTimeoutException("waitForAllStates");
		}

	}

	public void waitMillis(long millis) {
		try {
			Thread.sleep(millis);
		} catch (InterruptedException e) {
			log.error("wait interrupted ", e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public void connecting(AgentInfo agent) {
		Minion m = name2minion.get(agent.getName());
		if (m != null) {
			presentMinions.add(m);
			checkPresence();
		}
	}

	@Override
	public void disconnecting(AgentInfo agent) {
		Minion m = name2minion.get(agent.getName());
		if (m != null) {
			presentMinions.remove(m);

			// should we clear from the SBA because the missing subsystem state
			// is
			// at best unknown?

			checkPresence();
		}
	}

	private abstract class StateChangeToEventRule {
		protected MCMEvent event;

		public StateChangeToEventRule(MCMEvent e) {
			event = e;
		}

		public MCMEvent getEvent() {
			return event;
		}

		public abstract int matches(StateBundle out, StateBundle changes); // returns
																			// level,
																			// 0
		// for no match
	}

	private class DefaultStateChangeToEventRule extends StateChangeToEventRule {
		Minion minion;
		Class<FilterState> stateClass;

		public DefaultStateChangeToEventRule(Minion m, Class<FilterState> s, MCMEvent e) {
			super(e);
			stateClass = s;
			minion = m;
		}

		@Override
		public int matches(StateBundle out, StateBundle changes) {
			return changes.getState(stateClass) == null ? 0 : 1;
		}
	}

	private class SingleStateChangeToEventRule extends StateChangeToEventRule {
		Minion minion;
		Enum state;

		public SingleStateChangeToEventRule(Minion m, Enum s, MCMEvent e) {
			super(e);
			minion = m;
			state = s;
		}

		@Override
		public int matches(StateBundle out, StateBundle changes) {
			return changes.isInState(state) ? 2 : 0;
		}
	}

	private class SingleStateChangeOutToEventRule extends StateChangeToEventRule {
		Minion minion;
		Enum outState;

		public SingleStateChangeOutToEventRule(Minion m, Enum out, MCMEvent e) {
			super(e);
			minion = m;
			outState = out;
		}

		@Override
		public int matches(StateBundle out, StateBundle changes) {
			return (out.isInState(outState)) ? 2 : 0;
		}
	}

	private class CombinedStatesChangeToEventRule extends StateChangeToEventRule {
		Map<Minion, Enum> states = new EnumMap<Minion, Enum>(Minion.class);

		public CombinedStatesChangeToEventRule(Minion m, Enum state, MCMEvent e) {
			super(e);
			states.put(m, state);
		}

		public CombinedStatesChangeToEventRule addState(Minion m, Enum state) {
			states.put(m, state);
			return this;
		}

		public int matches(StateBundle out, StateBundle changes) {
			throw new RuntimeException("not implemented");
			// need to check the other states in the sba
		}

	}

	List<StateChangeToEventRule> eventRules = new ArrayList<StateChangeToEventRule>();

	public void initEventRules() {
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_U, MCMEvent.filterULoaded));
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_G, MCMEvent.filterGLoaded));
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_R, MCMEvent.filterRLoaded));
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_I, MCMEvent.filterILoaded));
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_Z, MCMEvent.filterZLoaded));
		eventRules.add(new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_Y, MCMEvent.filterYLoaded));
		eventRules.add(
				new SingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_NONE, MCMEvent.filterNoneLoaded));
		eventRules
				.add(new DefaultStateChangeToEventRule(Minion.FILTER, FilterState.class, MCMEvent.filterSystemMoving));

		eventRules.add(new SingleStateChangeToEventRule(Minion.RAFTS, RaftState.QUIESCENT, MCMEvent.ccdCleared));
		eventRules
				.add(new SingleStateChangeToEventRule(Minion.RAFTS, RaftState.INTEGRATING, MCMEvent.startIntegration));
		eventRules.add(new SingleStateChangeToEventRule(Minion.RAFTS, RaftState.READING_OUT, MCMEvent.startReadout));
		eventRules.add(new SingleStateChangeOutToEventRule(Minion.RAFTS, RaftState.READING_OUT, MCMEvent.endReadout));
		eventRules.add(new SingleStateChangeToEventRule(Minion.RAFTS, RaftState.NEEDS_CLEAR, MCMEvent.ccdNotReady));

		eventRules
				.add(new SingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.OPENING, MCMEvent.startShutterOpen));
		eventRules.add(new SingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.OPEN, MCMEvent.endShutterOpen));
		eventRules.add(
				new SingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.CLOSING, MCMEvent.startShutterClose));
		eventRules.add(new SingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.CLOSED, MCMEvent.endShutterClose));

	}

	public void checkEventRules(StateBundle out, StateBundle change) {
		MCMEvent bestEvent = null;
		int level = 0;
		for (StateChangeToEventRule rule : eventRules) {
			int l = rule.matches(out, change);
			if (l > level) {
				level = l;
				bestEvent = rule.getEvent();
			}
		}
		if (bestEvent != null) {
			log.info("sending MCM event " + bestEvent);
			StatusEnum<MCMEvent> message = new StatusEnum<MCMEvent>(bestEvent, mcm.getState());
			mcm.getMessagingAccess().sendStatusMessage(message);
		}
	}

	public void minionStateChange(String source, StateBundle out, StateBundle change) {

		log.info("from " + source + " state change " + change);

		if (change.getState(OperationalState.class) != null) {
			checkCamOpState();
		}
		checkEventRules(out, change);

	}

	private CCSOperationalState ccsState = CCSOperationalState.NORMAL;
	private CameraOperationalState camState = CameraOperationalState.NORMAL;
	private CCSSalState salState = CCSSalState.PUBLISH_ONLY;

	void checkPresence() {
		CCSOperationalState newS = ccsState;
		if (presentMinions.size() < minionNames.size()) {
			newS = CCSOperationalState.MISSING_SUBSYSTEM;
		} else {
			newS = CCSOperationalState.NORMAL;
		}
		if (ccsState != newS) {
			ccsState = newS;
			mcm.updateAgentState(ccsState);
			log.info("MCM state: " + ccsState);

		}
	}

	void checkCamOpState() {
		CameraOperationalState newS = camState;
		for (String mn : minionNames.values()) {
			StateBundle sb = sba.getState(mn);
			if (sb != null) {
				OperationalState s = (OperationalState) sb.getState(OperationalState.class);
				if (s == null) {
					newS = CameraOperationalState.ENGINEERING_FAULT;
					break;
				}
				if (s.equals(OperationalState.ENGINEERING_FAULT)) {
					newS = CameraOperationalState.ENGINEERING_FAULT;
					break;
				}
				if (s.equals(OperationalState.ENGINEERING_OK)) {
					newS = CameraOperationalState.ENGINEERING_OK;
				}

			} else {
				newS = CameraOperationalState.ENGINEERING_FAULT;
				break;
			}
		}
		if (!newS.equals(camState)) {
			log.info("camera operational state " + camState + " -> " + newS);
			camState = newS;
			mcm.updateAgentState(camState);
		}
	}

	void checkSalState() {
		CCSSalState newS = salState;

		// should we transition based on commands from OCS only, except for
		// FAULT ?

		if (camState == CameraOperationalState.ENGINEERING_FAULT)
			salState = CCSSalState.FAULT;

		if (!newS.equals(salState)) {
			log.info("camera SAL state " + salState + " -> " + newS);
			salState = newS;
			mcm.updateAgentState(salState);
		}

	}

}
