package org.lsst.ccs.subsystem.mcm;

import java.time.Duration;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.mcm.MCMUtilities.ExpectedStateCombination;
import org.lsst.ccs.subsystem.mcm.data.CameraStatus;
import org.lsst.ccs.subsystem.mcm.data.FilterReadinessState;
import org.lsst.ccs.subsystem.mcm.data.FilterState;
import org.lsst.ccs.subsystem.mcm.data.OperationTimeoutException;
import org.lsst.ccs.subsystem.mcm.data.RaftState;
import org.lsst.ccs.subsystem.mcm.data.ShutterReadinessState;
import org.lsst.ccs.subsystem.mcm.data.ShutterState;
import org.lsst.ccs.utilities.logging.Logger;

// TODO handle alarm while we are waiting for a task
// => abort? 

// TODO easy way to configure the aggregated status sent to OCS bridge

// TODO camera state = logic on individual states

public class MCM extends Module {

	protected MCMUtilities mu;

	public MCM(String name, int tickMillis) {
		super(name, tickMillis);
	}

	@Override
	public void initModule() {
		mu = new MCMUtilities(getSubsystem());
	}

	// todo abort, stop

	protected Random random = new Random(); // for testing purposes

	protected static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mcm");

	@Command
	public void initImages(long delay) {

		setAbortingOnAlarmMinions();

		if (delay < 70) {
			log.info("sending clear to rafts");
			sendAsync(Minion.RAFTS, "clear");
			log.info("sending prepare to shutter");
			sendAsync(Minion.SHUTTER, "prepare");
			log.info("waiting for RAFTS quiescent and shutter READY");
			try {
				int sel = random.nextInt() % 4;
				sel = 3;
				switch (sel) {
				case 0:
					waitForState(Minion.SHUTTER, ShutterReadinessState.READY, 200);
					waitForState(Minion.RAFTS, RaftState.QUIESCENT, 200);
					break;
				case 1:
					waitForState(Minion.RAFTS, RaftState.QUIESCENT, 200);
					waitForState(Minion.SHUTTER, ShutterReadinessState.READY, 200);
					break;
				case 2:
					expectingState(Minion.SHUTTER, ShutterReadinessState.READY)
							.expectingState(Minion.RAFTS, RaftState.QUIESCENT).waitForAllStatesHappening(200);
					break;
				case 3:
					expectingState(Minion.SHUTTER, ShutterReadinessState.READY)
							.expectingState(Minion.RAFTS, RaftState.QUIESCENT).waitForAllStates(200);
					break;

				}

			} catch (OperationTimeoutException e) {
				log.error("timeout while waiting", e);
				throw new RuntimeException(e);
			}
			log.info("Raft and shutter ready");
		} else {
			schedule(() -> sendAsync(Minion.RAFTS, "clear"), Duration.ofMillis(delay - 70));
			if (delay < 150) {
				sendAsync(Minion.SHUTTER, "prepare");
			} else {
				schedule(() -> sendAsync(Minion.SHUTTER, "prepare"), Duration.ofMillis(delay - 150));
			}

			// should we wait??
			try {
				waitForState(Minion.RAFTS, RaftState.QUIESCENT, delay + 200);
				waitForState(Minion.SHUTTER, ShutterReadinessState.READY, delay + 200);
			} catch (OperationTimeoutException e) {
				log.error("timeout while waiting", e);
				throw new RuntimeException(e);
			}
			log.info("Raft and shutter ready");

		}
	}

	@Command
	public void configureCamera(String configId) {
		Future<?> e1 = execute(() -> {
			log.info("e1 - start");
			waitMillis(1000);
			log.info("e1 - end");
		});
		Future<?> e2 = execute(() -> {
			log.info("e2 - start");
			waitMillis(1500);
			log.info("e2 - end");
		});

		try {
			e2.get();
			e1.get();
		} catch (InterruptedException | ExecutionException e) {
			log.error(e);
		}

		log.info("done");

	}

	@Command
	public void setFilter(String filterName) {

		// check other conditions?

		setAbortingOnAlarmMinions(Minion.FILTER);

		filterName = filterName.toUpperCase();
		FilterState goal = "NONE".equals(filterName) ? FilterState.UNLOADED
				: FilterState.valueOf("LOADED_" + filterName);

		if (isInState(Minion.FILTER, goal)) {
			log.info("MCM : filter already ok for " + filterName);
			return;
		}

		try {
			send(Minion.FILTER, "loadFilter", filterName);
			waitForState(Minion.FILTER, goal, 60000);
		} catch (OperationTimeoutException e) {
			log.error("timeout while waiting", e);
			throw e;
		} catch (AlarmException e) {
			log.error("alarm occurred ", e);
			// do not wrap AlarmException, local
			throw new RuntimeException("alarm occurred " + e.getMessage() + " while setFilter");
		} catch (Exception e) {
			log.error("error", e);
			throw new RuntimeException("error ", e);
		}

		log.info("filter " + filterName + " loaded");

	}

	@Command
	public void takeImages(int n, int exposureMillis, boolean openShutter, boolean scienceActive, boolean guidingActive,
			boolean WFSActive) {

		// NB raft integrate is sent sync to have invalid state exception
		// immediately
		// and not through the Future.

		if (openShutter && exposureMillis < 1000) {
			throw new RuntimeException("cannot handle opening the shutter for less that 1 second");
		}

		checkState(Minion.RAFTS, RaftState.QUIESCENT);
		checkState(Minion.FILTER, FilterReadinessState.READY);

		for (int i = 0; i < n; i++) {
			try {
				log.info("MCM: send raft integrate");
				send(Minion.RAFTS, "integrate");
				waitForState(Minion.RAFTS, RaftState.INTEGRATING, 200);
				log.info("MCM: raft is integrating");

				if (openShutter) {
					log.info("MCM: send shutter expose");
					sendAsync(Minion.SHUTTER, "expose", exposureMillis);
					waitForState(Minion.SHUTTER, ShutterState.OPEN, 1200);
					log.info("MCM: shutter is open");
					// schedule a stop integrating about .5 second before
					// shutter is closed.
					waitForState(Minion.SHUTTER, ShutterState.CLOSED, exposureMillis + 3000L);
					log.info("MCM: shutter is closed");
				} else {
					waitMillis(exposureMillis);
					log.info("MCM: exposure time elapsed, with closed shutter");
				}
				log.info("MCM: send raft readout");
				send(Minion.RAFTS, "readout");
				if (i == n - 1)
					break;
				waitForState(Minion.RAFTS, RaftState.READING_OUT, 200);
				log.info("MCM: raft is reading out");
				waitForState(Minion.RAFTS, RaftState.QUIESCENT, 3000);
				log.info("MCM: raft is quiescent");
			} catch (OperationTimeoutException e) {
				log.error("timeout while waiting", e);
				throw e;
			} catch (Exception e) {
				log.error("error", e);
				throw new RuntimeException("error ", e);
			}

		}
	}

	@Command
	public void initGuiders() {
		// To be done
	}

	@Override
	public void tick() {
		Map<String, Object> last = mu.getStatusAggregator().getAllLast();
		for (Map.Entry<String, Object> ke : last.entrySet()) {
			log.debug("   status " + ke.getKey() + " last " + ke.getValue() + " avg " + mu.getStatusAggregator().getAverage(ke.getKey())
					+ " hist " + mu.getStatusAggregator().getHistory(ke.getKey()).size());

			CameraStatus cs = new CameraStatus();
			cs.setRaftTemperature(mu.getStatusAggregator().getAverage("raftsim/temperature"));
			log.info("publish raft avg temp " + cs.getRaftTemperature());
			KeyValueData d = new KeyValueData("cameraStatus", cs);
			getSubsystem().publishSubsystemDataOnStatusBus(d);

		}
	}

	// delegated methods for convenience

	public Object send(Minion dst, String command, Object... parms) throws Exception {
		return mu.send(dst, command, parms);
	}

	public Future<Object> sendAsync(Minion dst, String command, Object... parms) {
		return mu.sendAsync(dst, command, parms);
	}

	// raw type propagated from ConcurrentMessagingUtils
	// and cannot cast T<X> to T<X<?>>
	@SuppressWarnings("rawtypes")
	public Future<StatusMessage> watchForState(Minion sys, Enum<?> state) {
		return mu.watchForState(sys, state);
	}

	public <T extends Enum<T>> void waitForState(Minion sys, T state, long timeout) {
		mu.waitForState(sys, state, timeout);
	}

	public void waitMillis(long millis) {
		mu.waitMillis(millis);
	}

	public <T extends Enum<T>> void checkState(Minion sys, T state) {
		mu.checkState(sys, state);
	}

	@SafeVarargs
	public final <T extends Enum<T>> void checkState(Minion sys, T... state) {
		mu.checkState(sys, state);
	}

	public <T extends Enum<T>> boolean isInState(Minion sys, T state) {
		return mu.isInState(sys, state);
	}

	public void setAbortingOnAlarmMinions(Minion... m) {
		mu.setAbortingOnAlarmMinions(m);
	}

	public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
		return mu.schedule(r, delay);
	}

	public Future<?> execute(Runnable r) {
		return mu.execute(r);
	}

	public <T extends Enum<T>> ExpectedStateCombination expectingState(Minion m, T state) {
		return mu.expectingState(m, state);
	}

}
