package org.lsst.ccs.subsystem.lockmanager;

import java.time.ZoneId;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Wrapper for ConcurrentMap that logs removed and added locks.
 *
 * @author onoprien
 */
class LockMap extends ConcurrentHashMap<String, AgentLockInfo.AgentLockInfoString> {
    
    private final Logger log;
    private final Level level;
    
    LockMap(Logger log, Level level) {
        this.log = log;
        this.level = level;
    }
    
// -- Overriding Map : ---------------------------------------------------------

    @Override
    public AgentLockInfo.AgentLockInfoString putIfAbsent(String agent, AgentLockInfo.AgentLockInfoString value) {
        AgentLockInfo.AgentLockInfoString out = super.putIfAbsent(agent, value);
        if (out == null) {
            log(agent, value, "ADD");
        }
        return out;
    }

    @Override
    public AgentLockInfo.AgentLockInfoString remove(Object agent) {
        AgentLockInfo.AgentLockInfoString out = super.remove(agent);
        if (out != null) {
            log((String)agent, out, "REMOVE");
        }
        return out;
    }

    @Override
    public AgentLockInfo.AgentLockInfoString put(String agent, AgentLockInfo.AgentLockInfoString value) {
        log(agent, value, "ADD");
        return super.put(agent, value);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void log(String agent, AgentLockInfo.AgentLockInfoString value, String action) {
        if (log.isLoggable(level)) {
            AgentLockInfo lock = value.getLockInfo();
            StringBuilder sb = new StringBuilder();
            sb.append("LOCK ").append(action).append(" ").append(agent).append(" ");
            sb.append(lock.getOwner()).append(":").append(lock.getMaxLevel()).append("[").append(toString(lock.getTimeStamp())).append("]").append(lock.getStatus()).append(". ");
            List<String> methods = new ArrayList<>();
            for (StackTraceElement stackTraceElement : Thread.currentThread().getStackTrace()) {
                String className = stackTraceElement.getClassName();
                if (className.contains("LockManager")) {
                    methods.add(stackTraceElement.getMethodName());
                }
            }
            sb.append(String.join("-", methods));
            String message = sb.toString();
            switch (level.getName()) {
                case "FINEST" -> log.finest(message);
                case "FINER" -> log.debug(message);
                case "FINE" -> log.fine(message);
                case "INFO" -> log.info(message);
                case "WARNING" -> log.warn(message);
                case "SEVERE" -> log.error(message);
                default -> {}
            }
        }
    }
    
    private String toString(CCSTimeStamp time) {
        return time.getUTCInstant().atZone(ZoneId.systemDefault()).toLocalTime().toString();
    }
    
}
