package org.lsst.ccs.subsystem.lockmanager;

import static org.lsst.ccs.bus.data.AgentInfo.AgentType.CONSOLE;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.LOCK_MANAGER;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.MCM;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.OCS_BRIDGE;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ACKNOWLEDGED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ATTACH;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REJECTED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.RELEASED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REQUESTED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ADVANCED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.SYSTEM;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.USER;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.lsst.ccs.Agent;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.data.AgentLockInfo.AgentLockInfoString;
import org.lsst.ccs.bus.messages.StatusLock;
import org.lsst.ccs.bus.messages.StatusLockAggregate;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.utilities.logging.Logger;

public class LockManager extends Subsystem implements HasLifecycle {

    public LockManager(String name) {
        super(name, LOCK_MANAGER);
    }

    public LockManager() {
        super(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME, LOCK_MANAGER);
    }

    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.lockmanager");

    @LookupField(strategy = CHILDREN)
    AgentLockService agentLockService;

    @LookupField(strategy = TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = TOP)
    private Agent agent;
    private AgentMessagingLayer aml;

    @LookupField(strategy = TREE)
    private PersistencyService localPersistenceService;

    @Persist
//    volatile ConcurrentHashMap<String, AgentLockInfoString> locksByAgentName = new ConcurrentHashMap<>();
    volatile ConcurrentHashMap<String, AgentLockInfoString> locksByAgentName = new LockMap(log, Level.FINE);

    @ConfigurationParameter(description = "Map of max levels", maxLength = 150, units = "unitless")
    volatile private Map<String,Map<String,String>> maxLevelMap = new TreeMap<>();
                
    ConcurrentHashMap<String, AgentLockInfoString> locksByToken = new ConcurrentHashMap<>();

    // to synchronize requests, key = agentName
    ConcurrentHashMap<String, CountDownLatch> latches = new ConcurrentHashMap<>();

    Set<String> lockUnawareWorkers = new HashSet<>();

    private final ThreadLocal<Boolean> executingLockOrAttach = new ThreadLocal<>(); // flag set while executing LockOrAttach command

    protected void addLock(AgentLockInfo lock) {
        AgentLockInfoString lis = new AgentLockInfoString(lock);
        locksByAgentName.put(lock.getAgentName(), lis);
        locksByToken.put(lock.getToken(), lis);
    }

    protected void removeLock(AgentLockInfo lock) {
        // works even if only agentName or token are there
        if (lock != null && lock.getToken() == null) {
            lock = locksByAgentName.get(lock.getAgentName()).getLockInfo();
        }
        if (lock != null) {
            locksByAgentName.remove(lock.getAgentName());
            locksByToken.remove(lock.getToken());
        }
    }

    StatusMessageListener sml = new StatusMessageListener() {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            Object o = msg.getObject();
            if (!(o instanceof AgentLockInfo))
                return;
            AgentLockInfo lock = (AgentLockInfo) o;
            String source = msg.getOriginAgentInfo().getName();
            if ("lockmanager".equals(source))
                return;

            log.info("LM got message from " + source + " : " + lock);

            // are we waiting for news on this?
            CountDownLatch latch = latches.get(lock.getAgentName());
            if (latch == null)
                return;

            // is this interesting news?
            if (lock.getStatus() != ACKNOWLEDGED && lock.getStatus() != REJECTED)
                return;

            if (lock.getStatus() == REJECTED) {
                log.warning("Rejected lock " + lock);
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            if (!locksByAgentName.containsKey(lock.getAgentName())
                    && locksByAgentName.get(lock.getAgentName()).getLockInfo().getStatus() != REQUESTED) {
                // this is inconsistent
                log.error("Inconsistent state, received " + lock + " with latch " + latch + " and request "
                        + locksByAgentName.get(lock.getAgentName()));
                // let's clear all that
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            log.info("received acknowledged lock " + lock);

            // let's first store the new lock
            addLock(lock);

            // write the locks to disk
            localPersistenceService.persistNow();

            // and wake up waiters
            latch.countDown();
        }
    };

    AgentPresenceListener apl = new AgentPresenceListener() {
        @Override
        public void connected(AgentInfo... agents) {

            Set<String> sent = new HashSet<>();
            // A locker is a worker capable of getting locks (console, MCM, OCS Bridge)
            boolean lockerHasConnected = false;
            // Do those agents need refresh?
            for (AgentInfo ai : agents) {
                AgentType tp = ai.getType();
                if (!lockerHasConnected && (tp == CONSOLE || tp == MCM || tp == OCS_BRIDGE))
                    lockerHasConnected = true;
                AgentLockInfoString lis = locksByAgentName.get(ai.getName());
                AgentLockInfo lockInfo = lis == null ? null : lis.getLockInfo();
                if (lockInfo != null) {
                    log.info("refreshing lock, agent just connected " + lockInfo.getAgentName() + " by "
                            + lockInfo.getOwner());
                    AgentLockInfo refresh = AgentLockInfo.createReminder(lockInfo);
                    StatusLock msg = new StatusLock(refresh);
                    sendStatusMessage(msg);
                    sent.add(lockInfo.getToken());
                }
                String lockableInfo = ai.getAgentProperty("lockable");
                if (!"true".equals(lockableInfo))
                    lockUnawareWorkers.add(ai.getName());
            }

            // we will also broadcast the other locks, for new consoles, as info messages
            // if any lock-capable worker showed up
            if (lockerHasConnected) {
                ArrayList<AgentLockInfo> ll = new ArrayList<>();
                for (AgentLockInfoString lis : locksByAgentName.values()) {
                    AgentLockInfo lock = lis.getLockInfo();
                    if (sent.contains(lock.getToken()))
                        continue;
                    AgentLockInfo refresh = AgentLockInfo.createInfo(lock);
                    ll.add((refresh));
                }
                AgentLockInfo[] larr = ll.toArray(new AgentLockInfo[0]);
                StatusLockAggregate msg = new StatusLockAggregate(larr);
                sendStatusMessage(msg);
            }
        }

        @Override
        public void disconnected(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                lockUnawareWorkers.remove(ai.getName());
                if (ai.getType() == AgentType.CONSOLE) {
                    AgentLockInfo lockInfo = getLockInfo(ai.getName());
                    if (lockInfo != null) {
                        locksByAgentName.remove(lockInfo.getAgentName());
                        lockInfo = AgentLockInfo.createRelease(lockInfo.getAgentName(), lockInfo.getOwner());
                        StatusLock msg = new StatusLock(lockInfo);
                        msg.setState(agentStateService.getState());
                        aml.sendStatusMessage(msg);
                    }
                }
            }
        }

    };

    @Override
    public void init() {
        boolean loadAtStartup = true;
        boolean persistAtShutdown = true;
        localPersistenceService.setAutomatic(loadAtStartup, persistAtShutdown);
    }

    @Override
    public void postInit() {
        maxLevelMap = normalizeLevelsConfig(maxLevelMap);
    }

//    @ConfigurationParameterChanger
//    public void setMaxLevelMap(Map<String, Map<String, String>> maxLevelMap) {
//        this.maxLevelMap = normalizeLevelsConfig(maxLevelMap);
//    }

    @Override
    public void postStart() {
        aml = agent.getMessagingAccess();
        aml.addStatusMessageListener(sml, BusMessageFilterFactory.embeddedObjectClass(AgentLockInfo.class));
        aml.getAgentPresenceManager().addAgentPresenceListener(apl);

        // locksByAgentName is persisted, let's update locksByToken.
        for (AgentLockInfoString lis : locksByAgentName.values()) {
            AgentLockInfo lock = lis.getLockInfo();
            String agentName = lock.getAgentName();
            boolean deadConsole = agentName.startsWith("ccs-shell_") || agentName.startsWith("CCS-Graphical-Console_");
            if (deadConsole) {
                deadConsole = !aml.getAgentPresenceManager().agentExists(agentName);
            }
            String token = lock.getToken();
            if (token == null || deadConsole) {
                // token == null: We persisted a lock request and crashed before confirmation, probably.
                // TODO resync with the subsystem? It was never ack to the locking user.
                locksByAgentName.remove(agentName);
            } else {
                locksByToken.put(token, lis);
            }
        }
    }

    @Command(type = QUERY, category = SYSTEM, description = "destroys a lock", level = NORMAL, autoAck = false)
    public void destroyLock(@Argument(name = "agentName", description = "agent name") String agentName, String userId) {
        log.debug("got destroy command for " + agentName + " from " + userId);

        AgentLockInfo lockInfo = getLockInfo(agentName);
        if (lockInfo == null) {
            sendNack("The subsystem is not locked.");
            return;
        }

        if (agentLockService.getMaxLevel(userId, lockInfo.getAgentName()) < ENGINEERING_ROUTINE) {
            sendNack("User "+ userId +" is not authorized to destroy locks.");
            return;
        }

        sendAck(null);

        locksByAgentName.remove(lockInfo.getAgentName());

        lockInfo = AgentLockInfo.createRelease(lockInfo.getAgentName(), lockInfo.getOwner());

        // broadcast
        StatusLock msg = new StatusLock(lockInfo);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

    }

    // this is a query because we don't want to have to lock the lock manager...
    // Create another command type just for the lock manager?

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo lockAgent(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        log.debug("got lock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getOwner() == null) {
            sendNack("Cannot lock with null user.");
            return null;
        }

        if (lock.getStatus() != REQUESTED) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo != null) {
            if (!lock.getOwner().equals(lockInfo.getOwner())) {
                sendNack("Lock is owned by another user: " + lockInfo.getOwner());
                return null;
            } else {
                if (executingLockOrAttach.get() == null) { // executing a plain lockAgent command
                    sendNack("You already own the lock, please use \"attachLock " + lockInfo.getAgentName() +"\"");
                    return null;
                } else { // executing lockOrAttach, locking failed due to existing lock with the same owner, will try attachLock instead
                    throw new IllegalStateException();
                }
            }
        }

        sendAck(null);

        // we need to broadcast the request
        // we take ownership of that request through atomic store
        AgentLockInfoString old = locksByAgentName.putIfAbsent(lock.getAgentName(), new AgentLockInfoString(lock));
        if (old != null) {
            // there was a lock attempt in process or a successful lock
            return null;// or exception? Or existing lock? TODO
        }

        AgentLockInfo updatedLock = null;
        // Is this a legacy, non lock-aware worker?
        if (lockUnawareWorkers.contains(lock.getAgentName())) {
            updatedLock = AgentLockInfo.createAcknowledgeLegacy(lock,  agentLockService.getMaxLevel(lock.getOwner(), lock.getAgentName()));
            // pretend to have received an ack to get consistent behavior with a mix of legacy/new subsystems
            // broadcast the LEGACY for consistent behavior, notify everybody of the lock
            StatusLock msg = new StatusLock(updatedLock);
            sendStatusMessage(msg);
            addLock(updatedLock);
        } else {
            CountDownLatch latch = new CountDownLatch(1);

            // we know the max level, let's create a request with it
            AgentLockInfo lockRequest = AgentLockInfo.createLockRequest(lock,
                    agentLockService.getMaxLevel(lock.getOwner(), lock.getAgentName()));

            latches.put(lockRequest.getAgentName(), latch);

            StatusLock msg = new StatusLock(lockRequest);
            sendStatusMessage(msg);
            log.info("sent message, waiting. Level: "+ lockRequest.getMaxLevel());

            // the subsystem should acknowledge the lock

            try {
                // TODO return false if timeout, do we need to handle?
                latch.await(500, TimeUnit.MILLISECONDS);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

            updatedLock = getLockInfo(lock.getAgentName());

            // if it's not ACK then we need to remove it and status is not locked.

            if (updatedLock != null && updatedLock.getStatus() != ACKNOWLEDGED) {
                log.info("did not receive ack for" + updatedLock + " status " + updatedLock.getStatus());
                locksByAgentName.remove(updatedLock.getAgentName());
                // too late to send a nack, ack was sent. We return null.
                // we could also send an exception instead of returning null or do that
                // in the RemoteLockService
                updatedLock = null;
            }

            if (updatedLock != null)
                log.info("got lock ack " + updatedLock + " token " + updatedLock.getToken());

            // either we failed or we got an akcnowledged lock
            assert (updatedLock == null || updatedLock.getStatus() == ACKNOWLEDGED);

            latches.remove(lock.getAgentName());
        }

        // we return the acknowledged lock
        return updatedLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "unlock a subsystem", level = NORMAL, autoAck = false)
    public void unlockAgent(@Argument(name = "lock", description = "Unlock request") AgentLockInfo lock) {
        log.info("got unlock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != RELEASED) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return;
        }

        // check there is indeed an associated lock
        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo == null) {
            sendNack("The subsystem is not locked.");
            return;
        }

        if (!lockInfo.getOwner().equals(lock.getOwner())) {
            sendNack("Lock is owned by another user: " + lockInfo.getOwner());
            return;
        }

        sendAck(null);

        removeLock(lock);

        // broadcast
        StatusLock msg = new StatusLock(lock);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);
    }

    @Command(type = QUERY, category = SYSTEM, description = "attach a locked subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo attachLock(@Argument(name = "lock", description = "Attach request") AgentLockInfo lock) {
        // We can recover the lock through the agentName or the token
        // The owner must match
        // If provided (that could become mandatory) the token should match
        log.info("got attach command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != ATTACH) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo knownLock = null;

        if (lock.getToken() != null) {
            knownLock = locksByToken.get(lock.getToken()).getLockInfo();
            if (knownLock == null) {
                sendNack("invalid lock token " + lock.getToken());
                return null;
            }
            if (lock.getAgentName() != null && !lock.getAgentName().equals(knownLock.getAgentName())) {
                sendNack("inconsistent agent name for lock: " + lock.getToken() + " " + lock.getAgentName() + " "
                        + knownLock.getAgentName());
                return null;
            }
        } else if (lock.getAgentName() != null) {
            AgentLockInfoString ls = locksByAgentName.get(lock.getAgentName());
            if (ls != null)
                knownLock = ls.getLockInfo();
        } else {
            sendNack("Invalid lock info, no token, no agentName");
            return null;
        }
        if (knownLock == null) {
            sendNack("The subsystem is not locked.");
            return null;
        }

        if (!knownLock.getOwner().equals(lock.getOwner())) {
            sendNack("Lock is owned by another user: " + knownLock.getOwner());
            return null;
        }

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);

        sendAck(null);

        return knownLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "detach a locked subsystem", level = NORMAL)
    public void detachLock(@Argument(name = "lock", description = "Detach request") AgentLockInfo lock) {
        log.info("got detach command for " + lock.getAgentName() + " from " + lock.getOwner());
        // This implementation is void.
        // At the moment, detaching is a purely local operation, the console forgets
        // about the lock

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);
    }

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem, or attach a lock if it already exists", level = NORMAL, autoAck = false)
    public AgentLockInfo lockOrAttach(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        log.info("got lockOrAttach command for " + lock.getAgentName() + " from " + lock.getOwner());
        executingLockOrAttach.set(true);
        try {
            return lockAgent(lock);
        } catch (IllegalStateException x) {
            AgentLockInfo request = AgentLockInfo.createAttachRequest(lock, lock);
            return attachLock(request);
        } finally {
            executingLockOrAttach.set(null);
        }
    }

    /**
     * Accesor for the locks stored by the lock manager
     *
     * Such layer is needed to avoid a null-pointer exception when invoking
     * getMyInfo() if the lock does not exist in the ConcurentHashMap
     *
     * @return the lock for agentName if existing or null
     *
     * @author Alexandre Boucaud
     */
    private AgentLockInfo getLockInfo(String agentName) {
        AgentLockInfoString lockInfoString = locksByAgentName.get(agentName);

        return lockInfoString == null ? null : lockInfoString.getLockInfo();
    }

    @Command(type = QUERY, category = SYSTEM, description = "for the given user, returns a map of name or name wildcard to maximum allowed level", level = NORMAL, autoAck = false)
    public Map<String,Integer> getMaxLevelUser(@Argument(name = "user", description = "User ID") String user) {
        Map<String,Integer> out = AgentLockService.getMaxLevelsUser(user, maxLevelMap);
        log.fine("Responding to getMaxLevelUser with "+ out);
        return out;
    }

    @Command(type = QUERY, category = SYSTEM, description = "for the given agent, returns a map of users to maximum allowed levels", level = NORMAL, autoAck = false)
    public Map<String,Integer> getMaxLevelSubsystem(@Argument(name = "subsystem", description = "Subsystem name") String subsystem) {
        Map<String,Integer> out = AgentLockService.getMaxLevelsSubsystem(subsystem, maxLevelMap);
        log.fine("Responding to getMaxLevelSubsystem with "+ out);
        return out;
    }
    
    @Command(type = QUERY, category = USER, description = "Prints maximum authorized lock levels for the specified user", level = NORMAL, autoAck = false)
    public String printMaxLevels(
            @Argument(name = "user", description = "User ID or group name. Leave blank to print max levels for all users.", defaultValue = "") String user, 
            @Argument(name = "subsystem", description = "Subsystem name. Leave blank to print max levels for all subsystems.", defaultValue = "") String subsystem, 
            @Argument(name = "expand", description = "If true, any groups the user belongs to are expanded, and actual max levels for subsystems are printed.", defaultValue = "false") boolean expand) {
        if (maxLevelMap.isEmpty()) return "";
        if (subsystem == null || subsystem.isBlank()) {
            return printMaxLevels(user, expand);
        } else {
            if (user == null || user.isBlank()) {
                Map<String,Integer> m = getMaxLevelSubsystem(subsystem);
                if (expand) {
                    return String.join(System.lineSeparator(), m.entrySet().stream().map((e) -> e.getKey() +" : "+ e.getValue()).toList());
                } else {
                    return m.toString();
                }
            } else {
                user = user.trim();
                int level = agentLockService.getMaxLevel(user, subsystem);
                if (expand) {
                    return Integer.toString(level);
                } else {
                    return new Command.Level().with(Command.Level.OnUnnamed.ON_UNNAMED_NUMERIC).getName(level);
                }
            }
        }
    }
    
    @Deprecated(forRemoval=true)
//    @Command(type = QUERY, category = USER, description = "Prints maximum authorized lock levels for the specified user", level = NORMAL, autoAck = false)
    public String printMaxLevels(
            @Argument(name = "user", description = "User ID or group name. Leave empty to print max levels for all users.", defaultValue = "") String user, 
            @Argument(name = "expand", description = "If true, any groups the user belongs to are expanded, and actual max levels for subsystems are printed.", defaultValue = "false") boolean expand) {
        if (maxLevelMap.isEmpty()) return "";
        user = user == null ? "" : user.strip();
        if (expand) {
            if (user.isEmpty()) {
                LinkedList<String> users = new LinkedList<>();
                for (String u : maxLevelMap.keySet()) {
                    Map<String,Integer> userMap = getMaxLevelUser(u);
                    StringBuilder sb = new StringBuilder();
                    sb.append(u).append(" : [");
                    sb.append(String.join(", ", userMap.entrySet().stream().map(e -> e.getKey()+" : "+e.getValue()).toList()));
                    sb.append("]");
                    users.add(sb.toString());
                }
                return String.join(System.lineSeparator(), users);
            } else {
                Map<String,Integer> userMap = getMaxLevelUser(user);
                return String.join(System.lineSeparator(), userMap.entrySet().stream().map(e -> e.getKey()+" : "+e.getValue()).toList());
            }
        } else {
            if (user.isEmpty()) {
                Map<String,Map<String,String>> groups = new TreeMap<>();
                Map<String,Map<String,String>> users = new TreeMap<>();
                maxLevelMap.forEach((k,v) -> {
                    if (k.startsWith("@")) {
                        groups.put(k.substring(1), v);
                    } else {
                        users.put(k, v);
                    }
                });
                StringBuilder sb = new StringBuilder();
                sb.append("Groups:").append(System.lineSeparator());
                groups.forEach((k,v) -> {
                    sb.append("  ").append(k).append(" : ").append(AgentLockService.User.levelsToString(v,true)).append(System.lineSeparator());
                });
                sb.append("Users:").append(System.lineSeparator());
                users.forEach((k,v) -> {
                    sb.append("  ").append(k).append(" : ").append(AgentLockService.User.levelsToString(v,true)).append(System.lineSeparator());
                });
                return sb.toString();
            } else {
                Map<String,String> v = maxLevelMap.get(user);
                if (v == null) {
                    v = maxLevelMap.get("@"+ user);
                }
                if (v == null) {
                    throw new IllegalArgumentException("No such user or group: "+ user);
                } else {
                    return AgentLockService.User.levelsToString(v,false);
                }
            }
        }
    }
    
    @Command(type = QUERY, category = USER, description = "Adds, removes, or modifies an authorized user.", level = ENGINEERING_ADVANCED, autoAck = false)
    public String setUser(
            @Argument(name = "user", description = "User ID") String user, 
            @Argument(name = "maxLevels", description = "Group name, or a map of subsystem names to maximum allowed levels.") String maxLevels) {
        user = user.strip();
        TreeMap<String,Map<String,String>> config = new TreeMap<>(maxLevelMap);
        String out = "";
        if (maxLevels == null || maxLevels.isBlank()) {
            Map<String,String> v = config.remove(user);
            if (v == null) {
                throw new IllegalArgumentException("No such user: "+ user);
            } else {
                out = "Removed user "+ user +".";
            }
        } else {
            Map<String,String> v = AgentLockService.User.levelsFromString(maxLevels);
            StringBuilder sb = new StringBuilder(config.put(user, v) == null ? "Added" : "Modified");
            out = sb.append(" user ").append(user).append(" : ").append(AgentLockService.User.levelsToString(v, true)).toString();
        }
        getAgentService(ConfigurationService.class).change("/", "maxLevelMap", config);
        return out;
    }
    
    @Command(type = QUERY, category = USER, description = "Adds, removes, or modifies a group of authorized users.", level = ENGINEERING_ADVANCED, autoAck = false)
    public String setGroup(
            @Argument(name = "group", description = "Group name") String group, 
            @Argument(name = "maxLevels", description = "String representation of a map of subsystem names to maximum allowed levels. Leave empty to delete the group.") String maxLevels) {
        group = group.strip();
        TreeMap<String,Map<String,String>> config = new TreeMap<>(maxLevelMap);
        String out = "";
        if (maxLevels == null || maxLevels.isBlank()) {
            Map<String,String> v = config.remove("@"+ group);
            if (v == null) {
                throw new IllegalArgumentException("No such group: "+ group);
            } else {
                out = "Removed group "+ group +".";
            }
        } else {
            Map<String,String> v = AgentLockService.User.levelsFromString(maxLevels);
            StringBuilder sb = new StringBuilder(config.put("@"+group, v) == null ? "Added" : "Modified");
            out = sb.append(" group ").append(group).append(" : ").append(AgentLockService.User.levelsToString(v, true)).toString();
        }
        getAgentService(ConfigurationService.class).change("/", "maxLevelMap", config);
        return out;
    }
    
    @Command(type = QUERY, category = USER, description = "Modifies maximum level settings of an authorized user or group by setting explicit level for the specified subsystem.", level = ENGINEERING_ADVANCED, autoAck = false)
    public String setMaxLevel(
            @Argument(name = "user", description = "Existing user ID or group name") String user, 
            @Argument(name = "subsystem", description = "Subsystem name or a wildcard") String subsystem, 
            @Argument(name = "maxLevel", description = "Maximum allowed level.") String maxLevel) {
        user = user.strip();
        subsystem = subsystem.strip();
        maxLevel = maxLevel.strip();
        TreeMap<String,Map<String,String>> config = new TreeMap<>(maxLevelMap);
        String k = user;
        Map<String,String> v = config.get(k);
        if (v == null) {
            k = "@"+ k;
            v = config.get(k);
        }
        if (v == null) {
            throw new IllegalArgumentException("No such user or group: "+ user);
        }
        v = new LinkedHashMap<>(v);
        v.remove(subsystem);
        String s = AgentLockService.User.levelsToString(v, false);
        v = AgentLockService.User.levelsFromString(s +","+ subsystem +":"+ maxLevel);
        config.put(k, v);
        if (config.containsKey("@"+ k)) config.put("@"+ k, v);
        getAgentService(ConfigurationService.class).change("/", "maxLevelMap", config);
        return (k.startsWith("@") ? "Group" : "User") +" : "+ AgentLockService.User.levelsToString(v, true);
    }
    
    @Command(type = QUERY, category = USER, description = "Adds new or existing user to the specified group.", level = ENGINEERING_ADVANCED, autoAck = false)
    public String addUserToGroup(String user, String group) {
        user = user.strip();
        group = group.strip();
        if (!maxLevelMap.containsKey("@"+group)) return "No such group.";
        if (maxLevelMap.containsKey("@"+user)) return "A user cannot have the same name as a group.";
        TreeMap<String,Map<String,String>> config = new TreeMap<>(maxLevelMap);
        Map<String,String> v = config.get(user);
        if (v == null) {
            v = new LinkedHashMap<>();
            v.put(group, "");
        } else {
            String s = AgentLockService.User.levelsToString(v, false);
            v = AgentLockService.User.levelsFromString(s +","+ group +":");
        }
        config.put(user, v);
        getAgentService(ConfigurationService.class).change("/", "maxLevelMap", config);
        return user +" : "+ AgentLockService.User.levelsToString(v, true);
    }
    
    @Command(type = QUERY, category = USER, description = "Removes a user from the specified group.", level = ENGINEERING_ADVANCED, autoAck = false)
    public String removeUserFromGroup(String user, String group) {
        user = user.strip();
        group = group.strip();
        if (!maxLevelMap.containsKey("@"+group)) return "No such group.";
        Map<String,String> v = maxLevelMap.get(user);
        if (v == null) return "No such user.";
        v = new LinkedHashMap<>(v);
        if (v.remove(group, "")) {
            TreeMap<String,Map<String,String>> config = new TreeMap<>(maxLevelMap);
            String out;
            if (v.isEmpty()) {
                config.remove(user);
                out = "Removed user "+ user +".";
            } else {
                config.put(user, v);
                out = user +" : "+ AgentLockService.User.levelsToString(v, true);
            }
            getAgentService(ConfigurationService.class).change("/", "maxLevelMap", config);
            return out;
        } else {
            return "User "+ user +" is not in group "+ group;
        }
    }
    
    @Command(type = QUERY, category = USER, description = "Lists groups of authorized users.", level = NORMAL, autoAck = false)
    public String groups() {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String,Map<String,String>> e : maxLevelMap.entrySet()) {
            String group = e.getKey();
            if (group.startsWith("@")) {
                sb.append(group.substring(1)).append(" : ").append(AgentLockService.User.levelsToString(e.getValue(), true)).append(System.lineSeparator());
            }
        }
        return sb.toString();
    }
    
    @Command(type = QUERY, category = USER, description = "Lists users in the specified group.", level = NORMAL, autoAck = false)
    public String group(@Argument(name = "group", description = "Group name") String group) {
        group = group.strip();
        String key = group.startsWith("@") ? group : "@"+group;
        if (maxLevelMap.get(key) == null) {
            throw new IllegalArgumentException("No such group: "+ group);
        }
        Set<String> users = new TreeSet<>();
        for (Map.Entry<String,Map<String,String>> e : maxLevelMap.entrySet()) {
            String user = e.getKey();
            if (!user.startsWith("@")) {
                List<Map.Entry<String,String>> entries = new LinkedList<>();
                entries.addAll(e.getValue().entrySet());
                HashSet<String> groups = new HashSet<>();
                ListIterator<Map.Entry<String,String>> it = entries.listIterator(entries.size());
                while (it.hasPrevious()) {
                    Map.Entry<String,String> ee = it.previous();
                    String g = ee.getKey();
                    if (ee.getValue().isBlank() && !groups.contains(g)) {
                        if (g.equals(group)) {
                            users.add(user);
                            break;
                        } else {
                            Map<String,String> settingsForGroup = maxLevelMap.get("@"+ g);
                            if (settingsForGroup != null) {
                                settingsForGroup.entrySet().forEach(eee -> it.add(eee));
                                groups.add(g);
                            }
                        }
                        
                    }
                }
            }
        }
        return String.join(", ", users) +".";
    }
    
    @Command(type = QUERY, category = USER, description = "Prints maximum level of an authorized user or group for the specified subsystem.", level = NORMAL, autoAck = false)
    public String maxLevel(
            @Argument(name = "user", description = "User ID or group name") String user, 
            @Argument(name = "subsystem", description = "Subsystem name. If empty, print the entire map.") String subsystem) {
        if (user == null || user.isBlank()) throw new IllegalArgumentException("User not specified");
        if (subsystem == null || subsystem.isBlank()) throw new IllegalArgumentException("Subsystem not specified");
        return Integer.toString(agentLockService.getMaxLevel(user, subsystem));
    }
    
    static TreeMap<String,Map<String,String>> normalizeLevelsConfig(Map<String,Map<String,String>> config) {
        TreeMap<String,Map<String,String>> out = new TreeMap<>();
        config.forEach((user,v) -> {
            user = user.strip();
            try {
                v = AgentLockService.User.levelsFromString(AgentLockService.User.levelsToString(v, false));
                out.put(user, v);
            } catch (IllegalArgumentException x) {
            }
        });
        return out;
    }
    
}
