package org.lsst.ccs.subsystem.lockmanager;

import static org.lsst.ccs.bus.data.AgentInfo.AgentType.CONSOLE;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.LOCK_MANAGER;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.MCM;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.OCS_BRIDGE;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ACKNOWLEDGED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ATTACH;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REJECTED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.RELEASED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REQUESTED;
import static org.lsst.ccs.command.annotations.Command.ADMIN;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.SYSTEM;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.USER;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.Agent;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.data.AgentLockInfo.AgentLockInfoString;
import org.lsst.ccs.bus.messages.StatusLock;
import org.lsst.ccs.bus.messages.StatusLockAggregate;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.utilities.logging.Logger;

public class LockManager extends Subsystem implements HasLifecycle {

    public LockManager(String name) {
        super(name, LOCK_MANAGER);
    }

    public LockManager() {
        super("lockmanager", LOCK_MANAGER);
    }

    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.lockmanager");

    @LookupField(strategy = CHILDREN)
    AgentLockService agentLockService;

    @LookupField(strategy = TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = TOP)
    private Agent agent;
    private AgentMessagingLayer aml;

    @LookupField(strategy = TREE)
    private PersistencyService localPersistenceService;

    @Persist
    volatile ConcurrentHashMap<String, AgentLockInfoString> locksByAgentName = new ConcurrentHashMap<>();

    @ConfigurationParameter(description = "Map of max levels", maxLength = 150, units = "unitless")
    volatile private Map<String,Map<String,String>> maxLevelMap = new ConcurrentHashMap<>();
                
    ConcurrentHashMap<String, AgentLockInfoString> locksByToken = new ConcurrentHashMap<>();

    // to synchronize requests, key = agentName
    ConcurrentHashMap<String, CountDownLatch> latches = new ConcurrentHashMap<>();

    Set<String> lockUnawareWorkers = new HashSet<>();

    private final ThreadLocal<Boolean> executingLockOrAttach = new ThreadLocal<>(); // flag set while executing LockOrAttach command

    protected void addLock(AgentLockInfo lock) {
        AgentLockInfoString lis = new AgentLockInfoString(lock);
        locksByAgentName.put(lock.getAgentName(), lis);
        locksByToken.put(lock.getToken(), lis);
    }

    protected void removeLock(AgentLockInfo lock) {
        // works even if only agentName or token are there
        if (lock != null && lock.getToken() == null) {
            lock = locksByAgentName.get(lock.getAgentName()).getLockInfo();
        }
        if (lock != null) {
            locksByAgentName.remove(lock.getAgentName());
            locksByToken.remove(lock.getToken());
        }
    }

    StatusMessageListener sml = new StatusMessageListener() {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            Object o = msg.getObject();
            if (!(o instanceof AgentLockInfo))
                return;
            AgentLockInfo lock = (AgentLockInfo) o;
            String source = msg.getOriginAgentInfo().getName();
            if ("lockmanager".equals(source))
                return;

            log.info("LM got message from " + source + " : " + lock);

            // are we waiting for news on this?
            CountDownLatch latch = latches.get(lock.getAgentName());
            if (latch == null)
                return;

            // is this interesting news?
            if (lock.getStatus() != ACKNOWLEDGED && lock.getStatus() != REJECTED)
                return;

            if (lock.getStatus() == REJECTED) {
                log.warning("Rejected lock " + lock);
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            if (!locksByAgentName.containsKey(lock.getAgentName())
                    && locksByAgentName.get(lock.getAgentName()).getLockInfo().getStatus() != REQUESTED) {
                // this is inconsistent
                log.error("Inconsistent state, received " + lock + " with latch " + latch + " and request "
                        + locksByAgentName.get(lock.getAgentName()));
                // let's clear all that
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            log.info("received acknowledged lock " + lock);

            // let's first store the new lock
            addLock(lock);

            // write the locks to disk
            localPersistenceService.persistNow();

            // and wake up waiters
            latch.countDown();
        }
    };

    AgentPresenceListener apl = new AgentPresenceListener() {
        @Override
        public void connected(AgentInfo... agents) {

            Set<String> sent = new HashSet<>();
            // A locker is a worker capable of getting locks (console, MCM, OCS Bridge)
            boolean lockerHasConnected = false;
            // Do those agents need refresh?
            for (AgentInfo ai : agents) {
                AgentType tp = ai.getType();
                if (!lockerHasConnected && (tp == CONSOLE || tp == MCM || tp == OCS_BRIDGE))
                    lockerHasConnected = true;
                AgentLockInfoString lis = locksByAgentName.get(ai.getName());
                AgentLockInfo lockInfo = lis == null ? null : lis.getLockInfo();
                if (lockInfo != null) {
                    log.info("refreshing lock, agent just connected " + lockInfo.getAgentName() + " by "
                            + lockInfo.getOwner());
                    AgentLockInfo refresh = AgentLockInfo.createReminder(lockInfo);
                    StatusLock msg = new StatusLock(refresh);
                    sendStatusMessage(msg);
                    sent.add(lockInfo.getToken());
                }
                String lockableInfo = ai.getAgentProperty("lockable");
                if (!"true".equals(lockableInfo))
                    lockUnawareWorkers.add(ai.getName());
            }

            // we will also broadcast the other locks, for new consoles, as info messages
            // if any lock-capable worker showed up
            if (lockerHasConnected) {
                ArrayList<AgentLockInfo> ll = new ArrayList<>();
                for (AgentLockInfoString lis : locksByAgentName.values()) {
                    AgentLockInfo lock = lis.getLockInfo();
                    if (sent.contains(lock.getToken()))
                        continue;
                    AgentLockInfo refresh = AgentLockInfo.createInfo(lock);
                    ll.add((refresh));
                }
                AgentLockInfo[] larr = ll.toArray(new AgentLockInfo[0]);
                StatusLockAggregate msg = new StatusLockAggregate(larr);
                sendStatusMessage(msg);
            }
        }

        @Override
        public void disconnected(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                lockUnawareWorkers.remove(ai.getName());
                if (ai.getType() == AgentType.CONSOLE) {
                    AgentLockInfo lockInfo = getLockInfo(ai.getName());
                    if (lockInfo != null) {
                        locksByAgentName.remove(lockInfo.getAgentName());
                        lockInfo = AgentLockInfo.createRelease(lockInfo.getAgentName(), lockInfo.getOwner());
                        StatusLock msg = new StatusLock(lockInfo);
                        msg.setState(agentStateService.getState());
                        aml.sendStatusMessage(msg);
                    }
                }
            }
        }

    };

    @Override
    public void init() {
        boolean loadAtStartup = true;
        boolean persistAtShutdown = true;
        localPersistenceService.setAutomatic(loadAtStartup, persistAtShutdown);
    }

    @Override
    public void postInit() {
    }

    @Override
    public void postStart() {
        aml = agent.getMessagingAccess();
        aml.addStatusMessageListener(sml, BusMessageFilterFactory.embeddedObjectClass(AgentLockInfo.class));
        aml.getAgentPresenceManager().addAgentPresenceListener(apl);

        // locksByAgentName is persisted, let's update locksByToken.
        for (AgentLockInfoString lis : locksByAgentName.values()) {
            AgentLockInfo lock = lis.getLockInfo();
            String agentName = lock.getAgentName();
            boolean deadConsole = agentName.startsWith("ccs-shell_") || agentName.startsWith("CCS-Graphical-Console_");
            if (deadConsole) {
                deadConsole = !aml.getAgentPresenceManager().agentExists(agentName);
            }
            String token = lock.getToken();
            if (token == null || deadConsole) {
                // token == null: We persisted a lock request and crashed before confirmation, probably.
                // TODO resync with the subsystem? It was never ack to the locking user.
                locksByAgentName.remove(agentName);
            } else {
                locksByToken.put(token, lis);
            }
        }
    }

    @Command(type = QUERY, category = SYSTEM, description = "destroys a lock", level = NORMAL, autoAck = false)
    public void destroyLock(@Argument(name = "agentName", description = "agent name") String agentName, String userId) {
        AgentLockInfo lockInfo = getLockInfo(agentName);

        if (lockInfo == null) {
            sendNack("The subsystem is not locked.");
            return;
        }

        if (agentLockService.getMaxLevel(userId, lockInfo.getAgentName()) < ENGINEERING_ROUTINE) {
            sendNack("User "+ userId +" is not authorized to destroy locks.");
            return;
        }

        sendAck(null);

        locksByAgentName.remove(lockInfo.getAgentName());

        lockInfo = AgentLockInfo.createRelease(lockInfo.getAgentName(), lockInfo.getOwner());

        // broadcast
        StatusLock msg = new StatusLock(lockInfo);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

    }

    // this is a query because we don't want to have to lock the lock manager...
    // Create another command type just for the lock manager?

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo lockAgent(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        log.info("got lock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getOwner() == null) {
            sendNack("Cannot lock with null user.");
            return null;
        }

        if (lock.getStatus() != REQUESTED) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo != null) {
            if (!lock.getOwner().equals(lockInfo.getOwner())) {
                sendNack("Lock is owned by another user: " + lockInfo.getOwner());
                return null;
            } else {
                if (executingLockOrAttach.get() == null) { // executing a plain lockAgent command
                    sendNack("You already own the lock, please use \"attachLock " + lockInfo.getAgentName() +"\"");
                    return null;
                } else { // executing lockOrAttach, locking failed due to existing lock with the same owner, will try attachLock instead
                    throw new IllegalStateException();
                }
            }
        }

        sendAck(null);

        // we need to broadcast the request
        // we take ownership of that request through atomic store
        AgentLockInfoString old = locksByAgentName.putIfAbsent(lock.getAgentName(), new AgentLockInfoString(lock));
        if (old != null) {
            // there was a lock attempt in process or a successful lock
            return null;// or exception? Or existing lock? TODO
        }

        AgentLockInfo updatedLock = null;
        // Is this a legacy, non lock-aware worker?
        if (lockUnawareWorkers.contains(lock.getAgentName())) {
            updatedLock = AgentLockInfo.createAcknowledgeLegacy(lock,  agentLockService.getMaxLevel(lock.getOwner(), lock.getAgentName()));
            // pretend to have received an ack to get consistent behavior with a mix of legacy/new subsystems
            // broadcast the LEGACY for consistent behavior, notify everybody of the lock
            StatusLock msg = new StatusLock(updatedLock);
            sendStatusMessage(msg);
            addLock(updatedLock);
        } else {
            CountDownLatch latch = new CountDownLatch(1);

            // we know the max level, let's create a request with it
            AgentLockInfo lockRequest = AgentLockInfo.createLockRequest(lock,
                    agentLockService.getMaxLevel(lock.getOwner(), lock.getAgentName()));

            latches.put(lockRequest.getAgentName(), latch);

            StatusLock msg = new StatusLock(lockRequest);
            sendStatusMessage(msg);
            log.info("sent message, waiting. Level: "+ lockRequest.getMaxLevel());

            // the subsystem should acknowledge the lock

            try {
                // TODO return false if timeout, do we need to handle?
                latch.await(500, TimeUnit.MILLISECONDS);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

            updatedLock = getLockInfo(lock.getAgentName());

            // if it's not ACK then we need to remove it and status is not locked.

            if (updatedLock != null && updatedLock.getStatus() != ACKNOWLEDGED) {
                log.info("did not receive ack for" + updatedLock + " status " + updatedLock.getStatus());
                locksByAgentName.remove(updatedLock.getAgentName());
                // too late to send a nack, ack was sent. We return null.
                // we could also send an exception instead of returning null or do that
                // in the RemoteLockService
                updatedLock = null;
            }

            if (updatedLock != null)
                log.info("got lock ack " + updatedLock + " token " + updatedLock.getToken());

            // either we failed or we got an akcnowledged lock
            assert (updatedLock == null || updatedLock.getStatus() == ACKNOWLEDGED);

            latches.remove(lock.getAgentName());
        }

        // we return the acknowledged lock
        return updatedLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "unlock a subsystem", level = NORMAL, autoAck = false)
    public void unlockAgent(@Argument(name = "lock", description = "Unlock request") AgentLockInfo lock) {
        log.debug("got unlock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != RELEASED) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return;
        }

        // check there is indeed an associated lock
        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo == null) {
            sendNack("The subsystem is not locked.");
            return;
        }

        if (!lockInfo.getOwner().equals(lock.getOwner())) {
            sendNack("Lock is owned by another user: " + lockInfo.getOwner());
            return;
        }

        sendAck(null);

        removeLock(lock);

        // broadcast
        StatusLock msg = new StatusLock(lock);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

        // return, there is no race condition there
        return;
    }

    @Command(type = QUERY, category = SYSTEM, description = "attach a locked subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo attachLock(@Argument(name = "lock", description = "Attach request") AgentLockInfo lock) {
        // We can recover the lock through the agentName or the token
        // The owner must match
        // If provided (that could become mandatory) the token should match
        log.debug("got attachLock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != ATTACH) {
            sendNack("Lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo knownLock = null;

        if (lock.getToken() != null) {
            knownLock = locksByToken.get(lock.getToken()).getLockInfo();
            if (knownLock == null) {
                sendNack("invalid lock token " + lock.getToken());
                return null;
            }
            if (lock.getAgentName() != null && !lock.getAgentName().equals(knownLock.getAgentName())) {
                sendNack("inconsistent agent name for lock: " + lock.getToken() + " " + lock.getAgentName() + " "
                        + knownLock.getAgentName());
                return null;
            }
        } else if (lock.getAgentName() != null) {
            AgentLockInfoString ls = locksByAgentName.get(lock.getAgentName());
            if (ls != null)
                knownLock = ls.getLockInfo();
        } else {
            sendNack("Invalid lock info, no token, no agentName");
            return null;
        }
        if (knownLock == null) {
            sendNack("The subsystem is not locked.");
            return null;
        }

        if (!knownLock.getOwner().equals(lock.getOwner())) {
            sendNack("Lock is owned by another user: " + knownLock.getOwner());
            return null;
        }

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);

        sendAck(null);

        return knownLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "detach a locked subsystem", level = NORMAL)
    public void detachLock(@Argument(name = "lock", description = "Detach request") AgentLockInfo lock) {
        log.debug("got detachLock command for " + lock.getAgentName() + " from " + lock.getOwner());
        // This implementation is void.
        // At the moment, detaching is a purely local operation, the console forgets
        // about the lock

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);
    }

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem, or attach a lock if it already exists", level = NORMAL, autoAck = false)
    public AgentLockInfo lockOrAttach(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        executingLockOrAttach.set(true);
        try {
            return lockAgent(lock);
        } catch (IllegalStateException x) {
            AgentLockInfo request = AgentLockInfo.createAttachRequest(lock, lock);
            return attachLock(request);
        } finally {
            executingLockOrAttach.set(null);
        }
    }

    /**
     * Accesor for the locks stored by the lock manager
     *
     * Such layer is needed to avoid a null-pointer exception when invoking
     * getMyInfo() if the lock does not exist in the ConcurentHashMap
     *
     * @return the lock for agentName if existing or null
     *
     * @author Alexandre Boucaud
     */
    private AgentLockInfo getLockInfo(String agentName) {
        AgentLockInfoString lockInfoString = locksByAgentName.get(agentName);

        return lockInfoString == null ? null : lockInfoString.getLockInfo();
    }

    @Command(type = QUERY, category = SYSTEM, description = "for the given user, returns a map of name or name wildcard to maximum allowed level", level = NORMAL, autoAck = false)
    public Map<String,Integer> getMaxLevelUser(@Argument(name = "user", description = "User ID") String user) {
        Map<String,Integer> out = AgentLockService.getMaxLevelsUser(user, maxLevelMap);
        log.fine("Responding to getMaxLevelUser with "+ out);
        return out;
    }

    @Command(type = QUERY, category = SYSTEM, description = "for the given agent, returns a map of users to maximum allowed levels", level = NORMAL, autoAck = false)
    public Map<String,Integer> getMaxLevelSubsystem(@Argument(name = "subsystem", description = "Subsystem name") String subsystem) {
        Map<String,Integer> out = AgentLockService.getMaxLevelsSubsystem(subsystem, maxLevelMap);
        log.fine("Responding to getMaxLevelSubsystem with "+ out);
        return out;
    }
    
    @Command(type = QUERY, category = USER, description = "Adds, removes, or modifies an authorized user.", level = ADMIN, autoAck = false)
    public String addUser(
            @Argument(name = "user", description = "User ID") String user, 
            @Argument(name = "maxLevels", description = "Map of subsystem names to maximum allowed levels.") Map<String,String> maxLevels) {
        Map<String,Map<String,String>> old = AgentLockService.trimMaxLevelsConfig(maxLevelMap);
        boolean remove = maxLevels == null || maxLevels.isEmpty();
        if (remove) {
            old.remove(user);
        } else {
            old.put(user, maxLevels);
        }
        ConfigurationService serv = getAgentService(ConfigurationService.class);
        serv.change("/", "maxLevelMap", new ConcurrentHashMap<>(old));
//        serv.publishConfigurationInfo();
        if (remove) {
            return "Removed user "+ user;
        } else {
            StringBuilder sb = new StringBuilder("Max levels for ").append(user).append(":").append(System.lineSeparator());
            Map<String,Integer> levels = new TreeMap<>(getMaxLevelUser(user));
            levels.forEach((sub,level) -> {
                sb.append("  ").append(sub).append(" = ").append(level).append(System.lineSeparator());
            });
            return sb.toString();
        }
    }
    
    @Command(type = QUERY, category = USER, description = "Prints maximum authorized lock levels for the specified user", level = ENGINEERING_ROUTINE, autoAck = false)
    public String printMaxLevels(
            @Argument(name = "user", description = "User ID. Leave empty to print max levels for all users.", defaultValue = "") String user, 
            @Argument(name = "expand", description = "If true, any groups the user belongs to are expanded, and actual max levels for subsystems are printed.", defaultValue = "false") boolean expand) {
        if (maxLevelMap.isEmpty()) return "";
        if (expand) {
            if (user == null || user.isEmpty()) {
                LinkedList<String> users = new LinkedList<>();
                for (String u : maxLevelMap.keySet()) {
                    u = u.trim();
                    Map<String,Integer> userMap = getMaxLevelUser(u);
                    StringBuilder sb = new StringBuilder();
                    sb.append(u).append(" : [");
                    sb.append(String.join(", ", userMap.entrySet().stream().map(e -> e.getKey()+" : "+e.getValue()).toList()));
                    sb.append("]");
                    users.add(sb.toString());
                }
                return String.join(System.lineSeparator(), users);
            } else {
                Map<String,Integer> userMap = getMaxLevelUser(user);
                return String.join(System.lineSeparator(), userMap.entrySet().stream().map(e -> e.getKey()+" : "+e.getValue()).toList());
            }
        } else {
            Map<String,Map<String,String>> config = AgentLockService.trimMaxLevelsConfig(maxLevelMap);
            if (user == null || user.isEmpty()) {
                MaxLevelOrder comp = new MaxLevelOrder();
                StringBuilder sb = new StringBuilder();
                config.forEach((u,m) -> {
                    sb.append(u).append(" : [");
                    sb.append(String.join(", ", comp.entryList(m).stream().map(e -> e.getKey()+" : "+e.getValue()).toList()));
                    sb.append("]").append(System.lineSeparator());
                });
                return sb.toString();
            } else {
                Map<String,String> userMap = config.get(user);
                if (userMap == null || userMap.isEmpty()) {
                    return "";
                } else {
                    StringBuilder sb = new StringBuilder();
                    for (Map.Entry<String,String> e : new MaxLevelOrder().entryList(userMap)) {
                        sb.append(e.getKey()).append(" : ").append(e.getValue()).append(System.lineSeparator());
                    }
                    return sb.toString();
                }
            }
        }
    }
    
    private class MaxLevelOrder implements Comparator<Map.Entry<String,String>> {
        
        @Override
        public int compare(Map.Entry<String, String> e1, Map.Entry<String, String> e2) {
            return e2.getValue().compareTo(e1.getValue());
        }
        @Override
        public Comparator<Map.Entry<String, String>> thenComparing(Comparator<? super Map.Entry<String, String>> other) {
            return (e1,e2) -> e1.getKey().compareTo(e2.getKey());
        }
        
        List<Map.Entry<String,String>> entryList(Map<String, String> map) {
            ArrayList<Map.Entry<String,String>> out = new ArrayList<>(map.entrySet());
            out.sort(this);
            return out;
        }
    }

}
