package org.lsst.ccs.subsystem.lockmanager;

import static org.lsst.ccs.bus.data.AgentInfo.AgentType.CONSOLE;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.LOCK_MANAGER;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.MCM;
import static org.lsst.ccs.bus.data.AgentInfo.AgentType.OCS_BRIDGE;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ACKNOWLEDGED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.ATTACH;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REJECTED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.RELEASED;
import static org.lsst.ccs.bus.data.AgentLockInfo.Status.REQUESTED;
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.CommandCategory.SYSTEM;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.Agent;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.data.AgentLockInfo.AgentLockInfoString;
import org.lsst.ccs.bus.messages.StatusLock;
import org.lsst.ccs.bus.messages.StatusLockAggregate;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.utilities.logging.Logger;

public class LockManager extends Subsystem implements HasLifecycle {

    public LockManager(String name) {
        super(name, LOCK_MANAGER);
    }

    public LockManager() {
        super("lockmanager", LOCK_MANAGER);
    }

    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.lockmanager");

    private static final int destroyLockRequiredLevel = 99;

    @LookupField(strategy = CHILDREN)
    AgentLockService agentLockService;

    @LookupField(strategy = TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = TOP)
    private Agent agent;
    private AgentMessagingLayer aml;

    @LookupField(strategy = TREE)
    private PersistencyService localPersistenceService;

    @Persist
    volatile ConcurrentHashMap<String, AgentLockInfoString> locksByAgentName = new ConcurrentHashMap<>();

    ConcurrentHashMap<String, AgentLockInfoString> locksByToken = new ConcurrentHashMap<>();

    // to synchronize requests, key = agentName
    ConcurrentHashMap<String, CountDownLatch> latches = new ConcurrentHashMap<>();

    Set<String> lockUnawareWorkers = new HashSet<>();

    private final ThreadLocal<Boolean> executingLockOrAttach = new ThreadLocal<>(); // flag set while executing LockOrAttach command

    protected void addLock(AgentLockInfo lock) {
        AgentLockInfoString lis = new AgentLockInfoString(lock);
        locksByAgentName.put(lock.getAgentName(), lis);
        locksByToken.put(lock.getToken(), lis);
    }

    protected void removeLock(AgentLockInfo lock) {
        // works even if only agentName or token are there
        if (lock != null && lock.getToken() == null) {
            lock = locksByAgentName.get(lock.getAgentName()).getLockInfo();
        }
        if (lock != null) {
            locksByAgentName.remove(lock.getAgentName());
            locksByToken.remove(lock.getToken());
        }
    }

    StatusMessageListener sml = new StatusMessageListener() {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            Object o = msg.getObject();
            if (!(o instanceof AgentLockInfo))
                return;
            AgentLockInfo lock = (AgentLockInfo) o;
            String source = msg.getOriginAgentInfo().getName();
            if ("lockmanager".equals(source))
                return;

            log.info("LM got message from " + source + " : " + lock);

            // are we waiting for news on this?
            CountDownLatch latch = latches.get(lock.getAgentName());
            if (latch == null)
                return;

            // is this interesting news?
            if (lock.getStatus() != ACKNOWLEDGED && lock.getStatus() != REJECTED)
                return;

            if (lock.getStatus() == REJECTED) {
                log.warning("Rejected lock " + lock);
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            if (!locksByAgentName.containsKey(lock.getAgentName())
                    && locksByAgentName.get(lock.getAgentName()).getLockInfo().getStatus() != REQUESTED) {
                // this is inconsistent
                log.error("Inconsistent state, received " + lock + " with latch " + latch + " and request "
                        + locksByAgentName.get(lock.getAgentName()));
                // let's clear all that
                latches.remove(lock.getAgentName());
                removeLock(lock);
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            log.info("received acknowledged lock " + lock);

            // let's first store the new lock
            addLock(lock);

            // write the locks to disk
            localPersistenceService.persistNow();

            // and wake up waiters
            latch.countDown();
        }
    };

    AgentPresenceListener apl = new AgentPresenceListener() {
        @Override
        public void connected(AgentInfo... agents) {

            Set<String> sent = new HashSet<>();
            // A locker is a worker capable of getting locks (console, MCM, OCS Bridge)
            boolean lockerHasConnected = false;
            // Do those agents need refresh?
            for (AgentInfo ai : agents) {
                AgentType tp = ai.getType();
                if (!lockerHasConnected && (tp == CONSOLE || tp == MCM || tp == OCS_BRIDGE))
                    lockerHasConnected = true;
                AgentLockInfoString lis = locksByAgentName.get(ai.getName());
                AgentLockInfo lockInfo = lis == null ? null : lis.getLockInfo();
                if (lockInfo != null) {
                    log.info("refreshing lock, agent just connected " + lockInfo.getAgentName() + " by "
                            + lockInfo.getOwner());
                    AgentLockInfo refresh = AgentLockInfo.createReminder(lockInfo);
                    StatusLock msg = new StatusLock(refresh);
                    sendStatusMessage(msg);
                    sent.add(lockInfo.getToken());
                }
                String lockableInfo = ai.getAgentProperty("lockable");
                if (!"true".equals(lockableInfo))
                    lockUnawareWorkers.add(ai.getName());
            }

            // we will also broadcast the other locks, for new consoles, as info messages
            // if any lock-capable worker showed up
            if (lockerHasConnected) {
                ArrayList<AgentLockInfo> ll = new ArrayList<>();
                for (AgentLockInfoString lis : locksByAgentName.values()) {
                    AgentLockInfo lock = lis.getLockInfo();
                    if (sent.contains(lock.getToken()))
                        continue;
                    AgentLockInfo refresh = AgentLockInfo.createInfo(lock);
                    ll.add((refresh));
                }
                AgentLockInfo[] larr = ll.toArray(new AgentLockInfo[0]);
                StatusLockAggregate msg = new StatusLockAggregate(larr);
                sendStatusMessage(msg);
            }
        }

        @Override
        public void disconnected(AgentInfo... agents) {
            for (AgentInfo ai : agents) {
                lockUnawareWorkers.remove(ai.getName());
            }
        }

    };

    @Override
    public void init() {
        boolean loadAtStartup = true;
        boolean persistAtShutdown = true;
        localPersistenceService.setAutomatic(loadAtStartup, persistAtShutdown);
    }

    @Override
    public void postInit() {
    }

    @Override
    public void postStart() {
        aml = agent.getMessagingAccess();
        aml.addStatusMessageListener(sml, BusMessageFilterFactory.embeddedObjectClass(AgentLockInfo.class));
        aml.getAgentPresenceManager().addAgentPresenceListener(apl);

        // locksByAgentName is persisted, let's update locksByToken.
        for (AgentLockInfoString lis : locksByAgentName.values()) {
            String token = lis.getLockInfo().getToken();
            if (token == null) {
                // We persisted a lock request and crashed before confirmation, probably.
                // TODO resync with the subsystem? It was never ack to the locking user.
                locksByAgentName.remove(lis.getLockInfo().getAgentName());
            } else {
                locksByToken.put(token, lis);
            }
        }
    }

    @Command(type = QUERY, category = SYSTEM, description = "destroys a lock", level = NORMAL, autoAck = false)
    public void destroyLock(@Argument(name = "agentName", description = "agent name") String agentName, String userId) {
        // can only be called from users with a maxlevel > xxx
        AgentLockInfo lockInfo = getLockInfo(agentName);

        if (lockInfo == null) {
            sendNack("no existing lock");
            return;
        }

        if (getMaxLevel(userId, lockInfo.getAgentName()) < destroyLockRequiredLevel) {
            sendNack("insufficient user max level to be allowed to destroy lock");
            return;
        }

        sendAck(null);

        locksByAgentName.remove(lockInfo.getAgentName());

        lockInfo = AgentLockInfo.createRelease(lockInfo.getAgentName(), lockInfo.getOwner());

        // broadcast
        StatusLock msg = new StatusLock(lockInfo);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

        return;
    }

    // this is a query because we don't want to have to lock the lock manager...
    // Create another command type just for the lock manager?

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo lockAgent(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        log.info("got lock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getOwner() == null) {
            sendNack("cannot lock with null user");
            return null;
        }

        if (lock.getStatus() != REQUESTED) {
            sendNack("lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo != null) {
            if (!lock.getOwner().equals(lockInfo.getOwner())) {
                sendNack("lock owned by another user/agent: " + lockInfo.getOwner());
                return null;
            } else {
                if (executingLockOrAttach.get() == null) { // executing a plain lockAgent command
                    sendNack("lock already owned by user -- please use attachLock " + lockInfo.getAgentName());
                    return null;
                } else { // executing lockOrAttach, locking failed due to existing lock with the same owner, will try attachLock instead
                    throw new IllegalStateException();
                }
            }
        }

        sendAck(null);

        // we need to broadcast the request
        log.info("proceeding");
        // we take ownership of that request through atomic store
        AgentLockInfoString old = locksByAgentName.putIfAbsent(lock.getAgentName(), new AgentLockInfoString(lock));
        if (old != null) {
            // there was a lock attempt in process or a successful lock
            return null;// or exception? Or existing lock? TODO
        }

        AgentLockInfo updatedLock = null;
        // Is this a legacy, non lock-aware worker?
        if (lockUnawareWorkers.contains(lock.getAgentName())) {
            updatedLock = AgentLockInfo.createAcknowledgeLegacy(lock,  getMaxLevel(lock.getOwner(), lock.getAgentName()));
            // pretend to have received an ack to get consistent behavior with a mix of legacy/new subsystems
            // broadcast the LEGACY for consistent behavior, notify everybody of the lock
            StatusLock msg = new StatusLock(updatedLock);
            sendStatusMessage(msg);
            addLock(updatedLock);
        } else {
            CountDownLatch latch = new CountDownLatch(1);

            // we know the max level, let's create a request with it
            AgentLockInfo lockRequest = AgentLockInfo.createLockRequest(lock,
                    getMaxLevel(lock.getOwner(), lock.getAgentName()));

            latches.put(lockRequest.getAgentName(), latch);

            StatusLock msg = new StatusLock(lockRequest);
            sendStatusMessage(msg);
            log.info("sent message, waiting");

            // the subsystem should acknowledge the lock

            try {
                // TODO return false if timeout, do we need to handle?
                latch.await(500, TimeUnit.MILLISECONDS);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

            updatedLock = getLockInfo(lock.getAgentName());

            // if it's not ACK then we need to remove it and status is not locked.

            if (updatedLock != null && updatedLock.getStatus() != ACKNOWLEDGED) {
                log.info("did not receive ack for" + updatedLock + " status " + updatedLock.getStatus());
                locksByAgentName.remove(updatedLock.getAgentName());
                // too late to send a nack, ack was sent. We return null.
                // we could also send an exception instead of returning null or do that
                // in the RemoteLockService
                updatedLock = null;
            }

            if (updatedLock != null)
                log.info("got lock ack " + updatedLock + " token " + updatedLock.getToken());

            // either we failed or we got an akcnowledged lock
            assert (updatedLock == null || updatedLock.getStatus() == ACKNOWLEDGED);

            latches.remove(lock.getAgentName());
        }

        // we return the acknowledged lock
        return updatedLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "unlock a subsystem", level = NORMAL, autoAck = false)
    public void unlockAgent(@Argument(name = "lock", description = "Unlock request") AgentLockInfo lock) {
        log.debug("got unlock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != RELEASED) {
            sendNack("lock request with bad status " + lock.getStatus());
            return;
        }

        // check there is indeed an associated lock
        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo == null) {
            sendNack("no existing lock");
            return;
        }

        if (!lockInfo.getOwner().equals(lock.getOwner())) {
            sendNack("lock owned by another user/agent: " + lockInfo.getOwner());
            return;
        }

        sendAck(null);

        removeLock(lock);

        // broadcast
        StatusLock msg = new StatusLock(lock);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

        // return, there is no race condition there
        return;
    }

    @Command(type = QUERY, category = SYSTEM, description = "attach a locked subsystem", level = NORMAL, autoAck = false)
    public AgentLockInfo attachLock(@Argument(name = "lock", description = "Attach request") AgentLockInfo lock) {
        // We can recover the lock through the agentName or the token
        // The owner must match
        // If provided (that could become mandatory) the token should match
        log.debug("got attachLock command for " + lock.getAgentName() + " from " + lock.getOwner());

        if (lock.getStatus() != ATTACH) {
            sendNack("lock request with bad status " + lock.getStatus());
            return null;
        }

        AgentLockInfo knownLock = null;

        if (lock.getToken() != null) {
            knownLock = locksByToken.get(lock.getToken()).getLockInfo();
            if (knownLock == null) {
                sendNack("invalid lock token " + lock.getToken());
                return null;
            }
            if (lock.getAgentName() != null && !lock.getAgentName().equals(knownLock.getAgentName())) {
                sendNack("inconsistent agent name for lock: " + lock.getToken() + " " + lock.getAgentName() + " "
                        + knownLock.getAgentName());
                return null;
            }
        } else if (lock.getAgentName() != null) {
            AgentLockInfoString ls = locksByAgentName.get(lock.getAgentName());
            if (ls != null)
                knownLock = ls.getLockInfo();
        } else {
            sendNack("invalid lock info, no token, no agentName");
            return null;
        }
        if (knownLock == null) {
            sendNack("no lock found");
            return null;
        }

        if (!knownLock.getOwner().equals(lock.getOwner())) {
            sendNack("lock not owned by caller " + knownLock.getOwner() + " " + lock.getOwner());
            return null;
        }

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);

        sendAck(null);

        return knownLock;
    }

    @Command(type = QUERY, category = SYSTEM, description = "detach a locked subsystem", level = NORMAL)
    public void detachLock(@Argument(name = "lock", description = "Detach request") AgentLockInfo lock) {
        log.debug("got detachLock command for " + lock.getAgentName() + " from " + lock.getOwner());
        // This implementation is void.
        // At the moment, detaching is a purely local operation, the console forgets
        // about the lock

        StatusLock msg = new StatusLock(lock);
        sendStatusMessage(msg);
    }

    @Command(type = QUERY, category = SYSTEM, description = "lock a subsystem, or attach a lock if it already exists", level = NORMAL, autoAck = false)
    public AgentLockInfo lockOrAttach(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        executingLockOrAttach.set(true);
        try {
            return lockAgent(lock);
        } catch (IllegalStateException x) {
            AgentLockInfo request = AgentLockInfo.createAttachRequest(lock, lock);
            return attachLock(request);
        } finally {
            executingLockOrAttach.set(null);
        }
    }

    /**
     * Accesor for the locks stored by the lock manager
     *
     * Such layer is needed to avoid a null-pointer exception when invoking
     * getMyInfo() if the lock does not exist in the ConcurentHashMap
     *
     * @return the lock for agentName if existing or null
     *
     * @author Alexandre Boucaud
     */
    private AgentLockInfo getLockInfo(String agentName) {
        AgentLockInfoString lockInfoString = locksByAgentName.get(agentName);

        return lockInfoString == null ? null : lockInfoString.getLockInfo();
    }

    public int getMaxLevel(String userid, String targetAgentName) {
        if (userid == null)
            return 0;
        if ("ccs".equals(userid))
            return 0;
        return 99;
    }

}
