package org.lsst.ccs.subsystem.lockmanager;

import java.time.Instant;
import java.util.StringJoiner;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.Agent;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.data.AgentLockInfo.Status;
import org.lsst.ccs.bus.messages.StatusLock;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandCategory;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

public class LockManager extends Subsystem implements HasLifecycle {

    public LockManager(String name) {
        super(name, AgentInfo.AgentType.LOCK_MANAGER);
    }

    public LockManager() {
        super("lockmanager", AgentInfo.AgentType.LOCK_MANAGER);
    }

    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.lockmanager");

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    AgentLockService agentLockService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    private AgentMessagingLayer aml;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private PersistencyService localPersistenceService;

    @Persist
    ConcurrentHashMap<String, AgentLockInfoString> locks = new ConcurrentHashMap<>();

    ConcurrentHashMap<String, CountDownLatch> latches = new ConcurrentHashMap<>();

    StatusMessageListener sml = new StatusMessageListener() {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            Object o = msg.getObject();
            if (!(o instanceof AgentLockInfo))
                return;
            AgentLockInfo lock = (AgentLockInfo) o;

            log.finest(msg);

            // are we waiting for news on this?
            CountDownLatch latch = latches.get(lock.getAgentName());
            if (latch == null)
                return;

            // is this interesting news?
            if (lock.getStatus() != AgentLockInfo.Status.ACKNOWLEDGED
                    && lock.getStatus() != AgentLockInfo.Status.REJECTED)
                return;

            if (lock.getStatus() == AgentLockInfo.Status.REJECTED) {
                log.warning("Rejected lock " + lock);
                latches.remove(lock.getAgentName());
                locks.remove(lock.getAgentName());
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            if (!locks.containsKey(lock.getAgentName())
                    && locks.get(lock.getAgentName()).getMyInfo().getStatus() != AgentLockInfo.Status.REQUESTED) {
                // this is inconsistent
                log.error("Inconsistent state, received " + lock + " with latch " + latch + " and request "
                        + locks.get(lock.getAgentName()));
                // let's clear all that
                latches.remove(lock.getAgentName());
                locks.remove(lock.getAgentName());
                // and wake-up whoever is waiting on this
                latch.countDown();
                return;
            }

            log.debug("received acknowledged lock");

            // let's first store the new lock
            locks.put(lock.getAgentName(), new AgentLockInfoString(lock));

            // write the locks to disk
            localPersistenceService.persistNow();

            // and wake up waiters
            latch.countDown();
        }
    };

    @Override
    public void init() {
        boolean loadAtStartup = true;
        boolean persistAtShutdown = true;
        localPersistenceService.setAutomatic(loadAtStartup, persistAtShutdown);
    }

    @Override
    public void postInit() {
    }

    @Override
    public void postStart() {
        aml = agent.getMessagingAccess();

        aml.addStatusMessageListener(sml, BusMessageFilterFactory.embeddedObjectClass(AgentLockInfo.class));
    }

    @Command(type = CommandType.QUERY, category = CommandCategory.SYSTEM, description = "locks a subsystem", level = 0)
    public AgentLockInfo lockAgent(@Argument(name = "lock", description = "Lock request") AgentLockInfo lock) {
        log.debug("got lock command for " + lock.getAgentName() + " from " + lock.getOwnerName());

        // already locked by someone else?
        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        log.debug("current lock " + lockInfo);

        if (lockInfo != null && !lock.getOwnerName().equals(lockInfo.getOwnerName()))
            throw new RuntimeException("lock owned by another user/agent: " + lockInfo.getOwnerName());

        // we need to broadcast the request

        AgentLockInfoString old = locks.putIfAbsent(lock.getAgentName(), new AgentLockInfoString(lock));
        if (old != null) {
            // there was a lock attempt in process or a successful lock
            return null;// or exception? Or existing lock?
        }

        CountDownLatch latch = new CountDownLatch(1);

        latches.put(lock.getAgentName(), latch);

        StatusLock msg = new StatusLock(lock);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);
        log.debug("sent message, waiting");

        // the subsystem should acknowledge the lock

        try {
            latch.await(5, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        AgentLockInfo updatedLock = getLockInfo(lock.getAgentName());
        log.debug("got lock ack " + updatedLock);

        assert (updatedLock == null || updatedLock.getStatus() == Status.ACKNOWLEDGED);

        latches.remove(lock.getAgentName());
        // we return the acknowledged lock
        return updatedLock;

    }

    @Command(type = CommandType.QUERY, category = CommandCategory.SYSTEM, description = "unlocks a subsystem", level = 0)
    public void unlockAgent(@Argument(name = "lock", description = "Unlock request") AgentLockInfo lock) {
        log.debug("got unlock command for " + lock.getAgentName() + " from " + lock.getOwnerName());

        // check there is indeed an associated lock
        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (lockInfo == null)
            throw new RuntimeException("no existing lock");

        if (!lockInfo.getOwnerName().equals(lock.getOwnerName()))
            throw new RuntimeException(
                    "lock owned by another user/agent: " + lockInfo.getOwnerName());

        locks.remove(lock.getAgentName());

        // broadcast
        StatusLock msg = new StatusLock(lock);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);

        // return, there is no race condition there
        return;
    }

    @Command(type = CommandType.QUERY, category = CommandCategory.SYSTEM, description = "updates lock level", level = 0)
    public void updateLevel(@Argument(name = "lock", description = "lock info") AgentLockInfo lock,
            @Argument(name = "level", description = "new level") int level) {
        log.debug("got updateLevel command");

        AgentLockInfo lockInfo = getLockInfo(lock.getAgentName());

        if (!lockInfo.getOwnerName().equals(lock.getOwnerName()))
            throw new RuntimeException("lock owned by another user/agent: " + lockInfo.getOwnerName());

        // TODO check levels

        StatusLock msg = new StatusLock(lock, level);
        msg.setState(agentStateService.getState());
        aml.sendStatusMessage(msg);
    }

    /**
     * Class that allows the serialization of the AgentLockInfo
     *
     * We use the PersistenceService to make sure the locks
     * are stored at any given time. The PersistenceService
     * needs the persisted object to be serializable, in the
     * form a String, which the AgentLockInfo implementation
     * is not.
     *
     * This class alleviates that issue.
     *
     * @author Alexandre Boucaud
     */
    public static class AgentLockInfoString {

        private AgentLockInfo myInfo;

        static String delimiter = ";";

        public AgentLockInfoString(AgentLockInfo info) {
            this.myInfo = info;
        }

        public AgentLockInfoString(String stringArgs) {
            String[] args = stringArgs.split(delimiter);

            String agentName = args[0];
            String ownerAgent = args[1];
            String userId = args[2];
            int maxLevel = Integer.parseInt(args[3]);
            Status status = Status.valueOf(args[4]);
            CCSTimeStamp timeStamp = CCSTimeStamp.currentTimeFromMillis(Instant.parse(args[5]).toEpochMilli());

            this.myInfo = new AgentLockInfo(agentName, ownerAgent, userId, maxLevel, status, timeStamp);
        }

        public AgentLockInfo getMyInfo() {
            return myInfo;
        }

        @Override
        public String toString() {
            StringJoiner lockInfo = new StringJoiner(delimiter);

            lockInfo.add(myInfo.getAgentName());
            lockInfo.add(myInfo.getOwnerAgentName());
            lockInfo.add(myInfo.getUserID());
            lockInfo.add(String.valueOf(myInfo.getMaxLevel()));
            lockInfo.add(myInfo.getStatus().name());
            // Convert time stamp to string using the UTC time since
            // CCSTimestamps are constructed from UTC before the TAI conversion
            lockInfo.add(myInfo.getTimeStamp().getUTCInstant().toString());

            return lockInfo.toString();
        }
    }


    /**
     * Accesor for the locks stored by the lock manager
     *
     * Such layer is needed to avoid a null-pointer exception
     * when invoking getMyInfo() if the lock does not exist
     * in the ConcurentHashMap
     *
     * @return the lock for agentName if existing or null
     *
     * @author Alexandre Boucaud
     */
    private AgentLockInfo getLockInfo(String agentName) {
        AgentLockInfoString lockInfoString = locks.get(agentName);

        return lockInfoString == null ? null : lockInfoString.getMyInfo();
    }

}
