package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.util.AsciiFuncs;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.imagehandling.config.WaitForHeaderService;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.utilities.image.FitsCheckSum;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;

/**
 * Handles post image processing of FITS files, including adding header data
 * from the header service, and executing commands on generated FITS files. This
 * class is used for science images and guider images.
 *
 * @author tonyj
 */
class PostImageFileHandling {

    private static final Logger LOG = Logger.getLogger(PostImageFileHandling.class.getName());

    private final ImageHandlingConfig config;
    private final AtomicBoolean isHeaderServiceEnabled;
    /**
     * These map an image name to a future which will complete when the specified data arrives.
     */
    private final Map<ImageName, CompletableFuture<List<ImageHeaderData.Header>>> futureHeaderDataForImage = new ConcurrentHashMap<>();
    private final Map<ImageName, CompletableFuture<Double>> futureMeasuredShutterTimeForImage = new ConcurrentHashMap<>();
    private final CommandExecutor commandExecutor;
    private final AlertService alertService;

    PostImageFileHandling(ImageHandlingConfig config, CommandExecutor commandExecutor, AlertService alertService) {
        this.config = config;
        this.isHeaderServiceEnabled = new AtomicBoolean(true);
        this.commandExecutor = commandExecutor;
        this.alertService = alertService;
    }

    /**
     * Given a list of files, this method will return a future which will
     * complete after the header service or shutter time data data has been received and merged. This
     * method is called from GuiderHandling and from (Science) ImageHandling.
     *
     * @param filesToProcess The list of files to process
     * @param imageName The image name
     * @return The Future.
     */
    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName) {
        // We only need to wait for the header service data if configured to do so
        boolean needToWaitForHeaderService = config.getWaitForHeaderService() == WaitForHeaderService.ALWAYS
                || (config.getWaitForHeaderService() == WaitForHeaderService.AUTO && isHeaderServiceEnabled.get());
        // Never wait for header service data if image is triggered by CCS(LSSTCCSRAFTS-780)
        needToWaitForHeaderService = needToWaitForHeaderService && imageName.getController() != Controller.CCS;
        boolean needToWaitForMeasuredShutterTime = imageName.getSource() == Source.MainCamera; // Only if lsstcam
        return handleAsynchronousData(filesToProcess, imageName, needToWaitForHeaderService, needToWaitForMeasuredShutterTime);
        
    }
    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName, boolean needToWaitForHeaderService, boolean needToWaitForMeasuredShutterTime)  {
        CompletableFuture<FileList> resultingFileList = filesToProcess;

        // If we need to wait for header service data, then this code returns a future which will complete when 
        // * the original filesToProcess future is complete AND
        // * header service has arrived, and the resulting data base been merged into the FITS file, or the data arrival timed out
        if (needToWaitForHeaderService) {
            CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = getHeaderDataFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureHeaderData, (fl, headerData) -> {
                return mergeHeaderServiceData(imageName, fl, headerData);
            });
        }
        // If we need to wait for measured shutter time, then this code returns a future which will complete when
        // * The original filesToProcess future is complete AND
        // * The measured shutter time has arrived, and the resulting data has been merged into the header service, or the data arrival timed out
        if (needToWaitForMeasuredShutterTime) {
            CompletableFuture<Double> futureMeasuredShutterTimeData = getMeasuredShutterTimeFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureMeasuredShutterTimeData, (fl, measuredShutterTime) -> {
                return mergeShutterTimeData(imageName, fl, measuredShutterTime);
            });
        }
        return resultingFileList;
    }

    FileList mergeHeaderServiceData(ImageName imageName, FileList fl, List<ImageHeaderData.Header> headerData) {
        String message = null;
        if (headerData == null) {
            message = String.format("Header service data for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);

        } else {
            try {
                fixupFitsFiles(fl, headerData);
            } catch (FitsException | IOException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && alertService != null && config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert(ImageHandlingClient.MISSING_HEADERSERVICEDATA_ALERTID, "Missing or malformed header service data");
            alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;

    }

    FileList mergeShutterTimeData(ImageName imageName, FileList fl, Double measuredShutterTime) {
        String message = null;
        if (measuredShutterTime == null) {
            message = String.format("The measured shutter time for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                ImageHeaderData.Header header = new ImageHeaderData.Header("SHUTTIME", String.valueOf(measuredShutterTime), "Shutter Exposure Time");
                fixupFitsFiles(fl, Collections.singletonList(header));
            } catch (FitsException | IOException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && alertService != null && config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            // TODO: Use a different alert id?
            Alert alert = new Alert(ImageHandlingClient.MISSING_HEADERSERVICEDATA_ALERTID, "Missing measured shutter data");
            alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;

    }

    void handleFitsFileCommands(CompletableFuture<FileList> filesToProcess, Location location, ImageName imageName, String mode) {
        if (!config.getCommands().isEmpty()) {
            filesToProcess.thenAccept(fl -> {
                Map<String, String> env = new HashMap<>();
                env.put("BOARD", location.getBoardName());
                env.put("RAFT", location.getRaftName());
                env.put("IMAGENAME", imageName.toString());
                env.put("MODE", mode);
                env.put("SEQNO", imageName.getNumberString());
                env.put("DATE", imageName.getDateString());
                env.put("CONTROLLER", imageName.getController().getCode());
                env.put("SOURCE", imageName.getSource().getCode());
                commandExecutor.executeFitsFileList(fl, env, location, imageName);
            });
        }
    }

    private void fixupFitsFiles(FileList fileList, List<ImageHeaderData.Header> headerData) throws FitsException, IOException {
        for (File file : fileList) {
            Fits fits = new Fits(file);
            final BasicHDU<?> hdu = fits.getHDU(0);
            Header header = hdu.getHeader();
            long deltaCheckSum = 0;
            for (ImageHeaderData.Header headerServiceHeader : headerData) {
                String keyword = headerServiceHeader.getKeyword();
                if (keyword == null || "COMMENT".equals(keyword)) {
                    continue;
                }
                if (keyword.startsWith("HIERARCH")) {
                    keyword = keyword.replace(" ", ".");
                }
                HeaderCard card = header.findCard(keyword);
                String newValue = headerServiceHeader.getValue();

                if (card != null) {
                    String oldValue = card.getValue();

                    if ((oldValue == null || oldValue.isEmpty()) && !(newValue == null || newValue.isEmpty())) {
                        long oldCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        if (!isString(newValue)) {
                            card.setValue(headerServiceHeader.getValue());
                        } else {
                            card = new HeaderCard(card.getKey(), newValue, card.getComment());
                            header.updateLine(card.getKey(), card);
                        }
                        long newCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        deltaCheckSum += newCheckSum - oldCheckSum;
                    } else if (!Objects.equals(oldValue, newValue)) {
                        LOG.log(Level.WARNING, "For card {0} header service={1} but CCS={2}", new Object[]{card.getKey(), newValue, oldValue});
                    }
                } else {
                    LOG.log(Level.WARNING, "For card {0} header service={1} but CCS was not defined", new Object[]{keyword, newValue});
                }
            }
            FitsCheckSum.updateCheckSum(header, deltaCheckSum);
            header.rewrite();
        }
    }

    private boolean isString(String string) {
        if ("true".equalsIgnoreCase(string) || "false".equalsIgnoreCase(string)) {
            return false;
        }
        try {
            Double.valueOf(string);
            return false;
        } catch (NumberFormatException x) {
            return true;
        }
    }

    /**
     * Look for an existing, or create a new, future which is waiting for the
     * header data for this image, and complete it by providing the measured
     * data.
     *
     * @param imageName The corresponding image name
     * @param headers The newly arrived headers.
     */
    void headerDataArrived(ImageName imageName, List<ImageHeaderData.Header> headers) {
        // TODO: This map is going to grow without bounds
        CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = getHeaderDataFutureForImage(imageName);
        futureHeaderData.complete(headers);
    }

    /**
     * Look for an existing, or create a new, future which is waiting for the
     * measured shutter time for this image, and complete it by providing the measured
     * data.
     *
     * @param imageName The corresponding image name
     * @param headers The newly arrived headers.
     */
    
    void measuredShutterTimeArrived(MeasuredShutterTime shutterTime) {
        CompletableFuture<Double> futureMeasuredShutterTime = getMeasuredShutterTimeFutureForImage(shutterTime.getImageName());
        futureMeasuredShutterTime.complete(shutterTime.getMeasuredShutterOpenTime());
    }

    /**
     * Look up or create a Future which will complete when either the header
     * data arrives, or after a timeout.
     *
     * @param imageName The corresponding imageName
     * @return The future
     */
    private CompletableFuture<List<ImageHeaderData.Header>> getHeaderDataFutureForImage(ImageName imageName) {
        return futureHeaderDataForImage.computeIfAbsent(imageName, s -> new CompletableFuture<List<ImageHeaderData.Header>>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15), TimeUnit.SECONDS));
    }

    /**
     * Look up or create a Future which will complete when either the measured
     * shutter time arrives, or after a timeout.
     *
     * @param imageName The corresponding imageName
     * @return The future
     */
    private CompletableFuture<Double> getMeasuredShutterTimeFutureForImage(ImageName imageName) {
        return futureMeasuredShutterTimeForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Double>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15), TimeUnit.SECONDS));
    }

    void setHeaderServiceEnabled(boolean enabled) {
        isHeaderServiceEnabled.set(enabled);
    }

    CompletableFuture<File> handleAdditionalFile(AdditionalFile additionalFile) {

        if (!config.getAdditionalFileCommands().isEmpty()) {
            Map<String, String> props = new HashMap<>();
            props.put("FileName", additionalFile.getFileName());
            props.put("FileType", additionalFile.getFileType());
            ImageName obsId = additionalFile.getObsId();
            props.put("ImageName", obsId.toString());
            props.put("ImageDate", obsId.getDateString());
            props.put("ImageNumber", obsId.getNumberString());
            props.put("ImageController", obsId.getController().getCode());
            props.put("ImageSource", obsId.getSource().getCode());
            // For compatibility with FITSFILES
            props.put("SEQNO", obsId.getNumberString());
            props.put("DATE", obsId.getDateString());
            props.put("CONTROLLER", obsId.getController().getCode());
            props.put("SOURCE", obsId.getSource().getCode());
            props.put("MODE", "OTHER");
            File file = config.getAdditionalFileName(additionalFile, props);
            try {
                // TODO: Write file asynchronously?
                try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file))) {
                    additionalFile.writeFile(out);
                }
                Map<String, String> env = props.entrySet().stream().collect(Collectors.toMap(e -> e.getKey().toUpperCase(), e -> e.getValue()));
                return commandExecutor.executeAdditional(file, obsId, additionalFile.getFileType(), env);
            } catch (IOException x) {
                LOG.log(Level.SEVERE, "Error handling additional file: " + additionalFile.getFileName(), x);
                return CompletableFuture.failedFuture(x);
            }
        }
        return CompletableFuture.completedFuture(null);
    }

}
