package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.imagedb.ImageFileDatabase;
import org.lsst.ccs.subsystem.imagehandling.imagedb.ImageFileDatabase.Operation;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.SensorLocation;

/**
 * A class for executing commands on (FITS) files
 *
 * @author tonyj
 */
public class CommandExecutor {

    private static final Logger LOG = Logger.getLogger(CommandExecutor.class.getName());
    private static final Pattern TELEMETRY_PATTERN = Pattern.compile("Telemetry\\:\\s*(.*)\\:\\s*(.*)");
    private static final Pattern SLOT_PATTERN = Pattern.compile(".*_S(.*)\\.fits");

    private final ImageHandlingConfig config;
    private final AtomicInteger runningProcesses = new AtomicInteger();
    private final Agent agent;
    private final ImageFileDatabase ifd;

    public CommandExecutor(ImageHandlingConfig config, Agent agent, ImageFileDatabase ifd) {
        this.config = config;
        this.agent = agent;
        this.ifd = ifd;
    }

    void executeFitsFileList(FileList fileList, Map<String, String> env, Location location, ImageName imageName) {
        for (File file : fileList) {
            Matcher matcher = SLOT_PATTERN.matcher(file.getName());
            if (matcher.matches()) {
                String slot = "S" + matcher.group(1);
                SensorLocation sensorLocation = new SensorLocation(location, Integer.parseInt(slot.substring(2, 3)));
                executeFitsFile(file, env, sensorLocation, imageName, slot);
            } else {
                LOG.log(Level.WARNING, "Ignored malformed file name {0}", file.getName());
            }
        }
    }

    void executeFitsFile(File fitsFile, Map<String, String> env, SensorLocation sensorLocation, ImageName imageName, String slot) {
        List<String> commands = config.getCommands();
        Map<String, String> newEnv = new LinkedHashMap(env);
        newEnv.put("FITSFILE", fitsFile.getAbsolutePath());
        newEnv.put("SLOT", slot);
        newEnv.put("FILETYPE", "FITS");
        int fileId = registerFileInDatabase(imageName, sensorLocation, fitsFile, "FITS");
        // executes the set of commands to be issued for this command
        CompletableFuture<File> future = executeFile(fitsFile, newEnv, commands);
        // to be executed after we get the log dile 
        future.thenAccept(logFile -> {
            String raft = newEnv.getOrDefault("RAFT", "R??");
            KeyValueDataList keyValueDataList = scanFileForTelemetry(logFile, "scripts/" + raft + "/" + slot);
            if (!keyValueDataList.getListOfKeyValueData().isEmpty()) {
                agent.publishSubsystemDataOnStatusBus(keyValueDataList);
            }
            registerOperations(fileId, logFile, fitsFile, false);
        });
    }

    private int registerFileInDatabase(ImageName imageName, SensorLocation sl, File file, String fileType) {
        int fileId = 0;
        if (ifd != null) {
            try {
                // line below inserts the name of the fits file - and sensor location (downstream)
                fileId = ifd.insertFile(imageName, sl, fileType, file.getAbsolutePath());
            } catch (SQLException x) {
                LOG.log(Level.WARNING, "Could not update image database for file: " + file, x);
            }
        }
        return fileId;
    }

    private void registerOperations(int fileId, File logFile, File file, boolean ins_update) {
        if (ifd != null && fileId != 0) {
            try {
                List<Operation> operations = scanFileForOperations(logFile, fileId);
                ifd.insertFileOperations(operations, ins_update);
            } catch (IOException | SQLException ex) {
                LOG.log(Level.WARNING, "Could not update image operations database for file: " + file, ex);
            }
        }
    }

    private CompletableFuture<File> executeFile(File file, Map<String, String> env, List<String> commands) {
        // TODO: Chain multiple commands together
        List<String> command = List.of("/bin/bash", "-c", commands.stream().collect(Collectors.joining(";")));
        String date = env.getOrDefault("DATE", "UNKNOWN");
        final File logFileDir = new File(config.getLogDirectory(), date);
        File logFile = new File(logFileDir, file.getName() + ".log");
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.environment().putAll(config.getEnvironment());
        processBuilder.environment().putAll(env);
        processBuilder.environment().put("LOGFILE", logFile.getAbsolutePath());
        processBuilder.environment().put("FILE", file.getAbsolutePath());
        processBuilder.directory(config.getCurrentDirectory());
        processBuilder.redirectErrorStream(true);
        processBuilder.redirectOutput(logFile);
        try {
            CompletableFuture result = new CompletableFuture<>();
            Files.createDirectories(logFileDir.toPath());
            int running = runningProcesses.incrementAndGet();
            processBuilder.environment().put("RUNNING", String.valueOf(running));
            Process process = processBuilder.start();
            LOG.log(Level.INFO, () -> String.format("Process for file %s started (%d running) on thread %s", file, running, Thread.currentThread().getName()));
            process.onExit().thenAccept((p) -> {
                runningProcesses.decrementAndGet();
                int rc = p.exitValue();
                if (rc != 0) {
                    String message = String.format("Process for file %s terminated with non-zero return code %d", file, rc);
                    LOG.log(Level.WARNING, message);
                    result.completeExceptionally(new RuntimeException(message));
                } else {
                    LOG.log(Level.INFO, () -> String.format("Process for file %s complete", file));
                    result.complete(logFile);
                }
            });
            return result;
        } catch (IOException x) {
            LOG.log(Level.WARNING, "Unable to create process for file " + file, x);
            return CompletableFuture.failedFuture(x);
        }
    }

    private KeyValueDataList scanFileForTelemetry(File logFile, String root) {
        KeyValueDataList kvdl = new KeyValueDataList("");
        try {
            try (BufferedReader in = Files.newBufferedReader(logFile.toPath())) {
                for (;;) {
                    String line = in.readLine();
                    if (line == null) {
                        break;
                    }
                    Matcher matcher = TELEMETRY_PATTERN.matcher(line);
                    if (matcher.matches()) {
                        String key = matcher.group(1);
                        String valueStr = matcher.group(2);
                        double value = Double.NaN;
                        try {
                            value = Double.parseDouble(valueStr);
                        } catch (NumberFormatException nfe) {
                            LOG.log(Level.WARNING, "Error parsing telemetry from command output, {0}", valueStr);
                        }
                        if (key.startsWith("/")) {
                            kvdl.addData(key, value);
                        } else {
                            kvdl.addData(root + "/" + key, value);

                        }
                    }

                }
            }
        } catch (IOException x) {
            LOG.log(Level.WARNING, "IO Error scanning log file: " + logFile, x);
        }
        return kvdl;
    }

    /**
     * Called to execute commands on a non-FITS file. Such files are not sensor
     * specific.
     *
     * @param file The file to be processed
     * @param obsId The corresponding image name
     * @param fileType The type of the file
     * @param env Environment to be passed to the processing script
     * @return A future log file of the command output.
     */
    CompletableFuture<File> executeAdditional(File file, ImageName obsId, String fileType, Map<String, String> env) {
        int fileId = registerFileInDatabase(obsId, null, file, fileType);
        CompletableFuture<File> future = executeFile(file, env, config.getAdditionalFileCommands());
        return future.thenApply(logFile -> {
            registerOperations(fileId, logFile, file, false);
            return logFile;
        });
    }

    CompletableFuture<File> executeMissingFile(List<String> commands, File fitsFile, int fileId, Map<String, String> env) {
        CompletableFuture<File> future = executeFile(fitsFile, env, commands);
        return future.thenApply(logFile -> {
            registerOperations(fileId, logFile, fitsFile, true);
            return logFile;
        });
    }

    /**
     * Scan a log file for operations. Operations in the log file are delimited
     * with Start: End: lines
     *
     * @param logFile The log file to scan
     * @param fileId The database id or the entry for the file which was
     * processed.
     * @return A list of operations extracted from the log file
     * @throws IOException If the log file could not be processed.
     */
    public List<Operation> scanFileForOperations(File logFile, int fileId) throws IOException {

        String readLine;

        long timeStampStart = 0, timeStampStop;
        String currentOperation = null;

        StringBuilder logContent = new StringBuilder();
        List<Operation> result = new ArrayList<>();

        try (BufferedReader br = new BufferedReader(new FileReader(logFile))) {

            boolean inParseBlock = false;

            while ((readLine = br.readLine()) != null) {

                final String[] tokens = readLine.split(" ");

                boolean start = tokens[0].equals("Start:");
                boolean end = tokens[0].equals("End:");

                // we should not be encountering an error when we are already in the parse block 
                if (inParseBlock && start) {
                    LOG.log(Level.WARNING, "Error in log file - 2 consecutive Starts encountered \n");
                }

                if (start && !inParseBlock) { // we don't want to use a corrupted Start to begin parsing....
                    inParseBlock = true;
                    currentOperation = tokens[1];
                    timeStampStart = Long.parseLong(tokens[2]);
                    logContent.setLength(0);
                }

                if (!end && !start && inParseBlock) {
                    logContent.append(readLine).append('\n');// appending everything between start and end to a log file ...
                } else if (end && tokens[1].equals(currentOperation) && inParseBlock) {
                    inParseBlock = false;// encountering End means we are done parsing
                    timeStampStop = Long.parseLong(tokens[3]);

                    Float duration = (timeStampStop - timeStampStart) / 1000.0f;

                    int rc = Integer.parseInt(tokens[2]);
                    boolean success = rc == 0;
                    String reasons = null;

                    if (!success) {
                        // if we didn't succeed we want some reasons 
                        reasons = String.format("%." + 256 + "s", logContent);
                    } else if (end && !tokens[1].equals(currentOperation) && inParseBlock) {

                        LOG.log(Level.WARNING, "Error in log file - current operation at End doesn't match with Start \n");
                    }

                    result.add(new Operation(fileId, currentOperation, rc, duration, success, reasons));

                }

            }

        }
        return result;

    }

}
