package org.lsst.ccs.subsystem.imagehandling.imagedb;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.SensorLocation;

/**
 * Opens a connect to an image database, and updates the database at the
 * completion of each image.
 *
 * @author tonyj
 */
public class ImageFileDatabase implements AutoCloseable {

    private final static int MAX_RETRIES = 1;

    private Connection conn;
    private static final Logger LOG = Logger.getLogger(ImageFileDatabase.class.getName());
    private final ImageDatabaseService imageDatabaseService;
    private final String host;
    private PreparedStatement stmt1;
    private PreparedStatement stmt2;

    public ImageFileDatabase(ImageDatabaseService imageDatabaseService) {
        this.imageDatabaseService = imageDatabaseService;
        String localHost;
        try {
            localHost = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException ex) {
            localHost = "Unknown";
        }
        this.host = localHost;
    }

    private void openConnection() throws SQLException {
        this.conn = imageDatabaseService.openConnection();
        try (PreparedStatement stmt = conn.prepareStatement("""
                create table if not exists ccs_image_file (
                file_id int not null AUTO_INCREMENT, 
                telCode varchar(2) not null, 
                seqnum integer not null, 
                dayobs varchar(8) not null, 
                controller varchar(2) not null,
                fileType varchar(24) not null,
                raft varchar(3) not null,
                sensor varchar(3) not null,
                writeHost varchar(64) not null, 
                writeFile varchar(255) not null, 
                primary key (telCode, seqnum, dayobs, controller, fileType, raft, sensor), 
                index(file_id)
                )""")) {
            stmt.execute();
        }
        try (PreparedStatement stmt = conn.prepareStatement("""
                create table if not exists ccs_image_file_operation (
                file_id int not null, 
                operation varchar(20) not null, 
                operation_return_code int, 
                time float not null, 
                success boolean not null, 
                reason varchar(256), 
                primary key (file_id, operation)
                )""")) {
            stmt.execute();
        }
        stmt1 = conn.prepareStatement(
                "insert into ccs_image_file (telcode, seqnum, dayobs, controller, fileType, raft, sensor, writeHost, writeFile) "
                + "values (?, ?, ?, ?, ?,"
                + "        ?, ?, ?, ?) ", Statement.RETURN_GENERATED_KEYS);

        stmt2 = conn.prepareStatement(
                "insert into ccs_image_file_operation (file_id, operation, operation_return_code, time, success, reason) "
                + "values (?, ?, ?, ?, ?, ?)");
    }

    public synchronized int insertFile(ImageName obsId, SensorLocation location, String fileType, String file) throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }
            try {
                stmt1.setString(1, obsId.getSource().getCode());
                stmt1.setInt(2, obsId.getNumber());
                stmt1.setString(3, obsId.getDateString());
                stmt1.setString(4, obsId.getController().getCode());
                stmt1.setString(5, fileType);
                if (location != null) {
                    Location rebLocation = location.getRebLocation();
                    stmt1.setString(6, rebLocation.getRaftName());
                    stmt1.setString(7, rebLocation.getSensorName(location.getSensor()));
                } else {
                    stmt1.setString(6, "NR");
                    stmt1.setString(7, "NS");
                }
                stmt1.setString(8, host);
                stmt1.setString(9, file);
                stmt1.execute();
                int auto_id;
                try (ResultSet rs = stmt1.getGeneratedKeys()) {
                    rs.next();
                    auto_id = rs.getInt(1);
                }
                conn.commit();
                return auto_id;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }
    
    public record Operation (int fileId, String operation, int operation_return_code, float time, boolean success, String reason) {}

    public synchronized void insertFileOperations(List<Operation> operations) throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }
            try {
                for (Operation operation : operations) {
                    stmt2.setInt(1, operation.fileId);
                    stmt2.setString(2, operation.operation);
                    stmt2.setInt(3, operation.operation_return_code);
                    stmt2.setFloat(4, operation.time);
                    stmt2.setBoolean(5, operation.success);
                    stmt2.setString(6, operation.reason);
                    stmt2.addBatch();
                }
                stmt2.executeBatch();
                conn.commit();
                return;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }

    /**
     * Only used for testing
     *
     * @return
     */
    Connection getConnection() {
        return this.conn;
    }

    @Override
    public void close() throws SQLException {
        if (conn != null) {
            imageDatabaseService.closeConnection();
            this.conn = null;
        }
    }
}
