package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;

/**
 * A class for executing commands on (FITS) files
 *
 * @author tonyj
 */
class CommandExecutor {

    private static final Logger LOG = Logger.getLogger(CommandExecutor.class.getName());
    private static final Pattern TELEMETRY_PATTERN = Pattern.compile("Telemetry\\:\\s*(.*)\\:\\s*(.*)");
    private static final Pattern SLOT_PATTERN = Pattern.compile(".*_S(.*)\\.fits");

    private final ImageHandlingConfig config;
    private final AtomicInteger runningProcesses = new AtomicInteger();
    private final Agent agent;

    CommandExecutor(ImageHandlingConfig config, Agent agent) {
        this.config = config;
        this.agent = agent;
    }

    void executeFitsFileList(FileList fileList, Map<String, String> env) {
        for (File file : fileList) {
            executeFitsFile(file, env);
        }
    }

    void executeFitsFile(File fitsFile, Map<String, String> env) {
        List<String> commands = config.getCommands();
        Map<String, String> newEnv = new LinkedHashMap(env);
        Matcher matcher = SLOT_PATTERN.matcher(fitsFile.getName());
        String slot = matcher.matches() ? "S" + matcher.group(1) : "S??";
        newEnv.put("FITSFILE", fitsFile.getAbsolutePath());
        newEnv.put("SLOT", slot);
        newEnv.put("FILETYPE", "FITS");
        CompletableFuture<File> future = executeFile(fitsFile, newEnv, commands);
        future.thenAccept(logFile -> {
            String raft = newEnv.getOrDefault("RAFT", "R??");
            KeyValueDataList keyValueDataList = scanFileForTelemetry(logFile, "scripts/" + raft + "/" + slot);
            if (!keyValueDataList.getListOfKeyValueData().isEmpty()) {
                agent.publishSubsystemDataOnStatusBus(keyValueDataList);
            }           
        });
    }

    CompletableFuture<File> executeFile(File file, Map<String, String> env, List<String> commands) {
        // TODO: Chain multiple commands together
        List<String> command = List.of("/bin/bash", "-c", commands.stream().collect(Collectors.joining(";")));
        String date = env.getOrDefault("DATE", "UNKNOWN");
        final File logFileDir = new File(config.getLogDirectory(), date);
        File logFile = new File(logFileDir, file.getName() + ".log");
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.environment().putAll(config.getEnvironment());
        processBuilder.environment().putAll(env);
        processBuilder.environment().put("LOGFILE", logFile.getAbsolutePath());
        processBuilder.environment().put("FILE", file.getAbsolutePath());
        processBuilder.directory(config.getCurrentDirectory());
        processBuilder.redirectErrorStream(true);
        processBuilder.redirectOutput(logFile);
        try {
            CompletableFuture result = new CompletableFuture<>();
            Files.createDirectories(logFileDir.toPath());
            int running = runningProcesses.incrementAndGet();
            processBuilder.environment().put("RUNNING", String.valueOf(running));
            Process process = processBuilder.start();
            LOG.log(Level.INFO, () -> String.format("Process for file %s started (%d running) on thread %s", file, running, Thread.currentThread().getName()));
            process.onExit().thenAccept((p) -> {
                runningProcesses.decrementAndGet();
                int rc = p.exitValue();
                if (rc != 0) {
                    String message = String.format("Process for file %s terminated with non-zero return code %d", file, rc);
                    LOG.log(Level.WARNING, message);
                    result.completeExceptionally(new RuntimeException(message));
                } else {
                    LOG.log(Level.INFO, () -> String.format("Process for file %s complete", file));
                    result.complete(logFile);
                }
            });
            return result;
        } catch (IOException x) {
            LOG.log(Level.WARNING, "Unable to create process for file " + file, x);
            return CompletableFuture.failedFuture(x);
        }
    }

    private KeyValueDataList scanFileForTelemetry(File logFile, String root) {
        KeyValueDataList kvdl = new KeyValueDataList("");
        try {
            try (BufferedReader in = Files.newBufferedReader(logFile.toPath())) {
                for (;;) {
                    String line = in.readLine();
                    if (line == null) {
                        break;
                    }
                    Matcher matcher = TELEMETRY_PATTERN.matcher(line);
                    if (matcher.matches()) {
                        String key = matcher.group(1);
                        String valueStr = matcher.group(2);
                        double value = Double.NaN;
                        try {
                            value = Double.parseDouble(valueStr);
                        } catch (NumberFormatException nfe) {
                            LOG.log(Level.WARNING, "Error parsing telemetry from command output, {0}", valueStr);
                        }
                        if (key.startsWith("/")) {
                            kvdl.addData(key, value);
                        } else {
                            kvdl.addData(root + "/" + key, value);

                        }
                    }

                }
            }
        } catch (IOException x) {
            LOG.log(Level.WARNING, "IO Error scanning log file: " + logFile, x);
        }
        return kvdl;
    }

    CompletableFuture<File> executeAdditional(File file, Map<String, String> env) {
        return executeFile(file, env, config.getAdditionalFileCommands());
    }
}
