/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.imagehandling;

import com.fasterxml.jackson.core.JsonProcessingException;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.BinaryTableHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.util.AsciiFuncs;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.imagehandling.CommandExecutor;
import org.lsst.ccs.subsystem.imagehandling.ImageHandlingConfig;
import org.lsst.ccs.subsystem.imagehandling.ShutterMotionProfileHandler;
import org.lsst.ccs.subsystem.imagehandling.config.WaitForHeaderService;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import org.lsst.ccs.utilities.image.FitsCheckSum;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

class PostImageFileHandling {
    private static final Logger LOG = Logger.getLogger(PostImageFileHandling.class.getName());
    private final ImageHandlingConfig config;
    private final AtomicBoolean isHeaderServiceEnabled;
    private final LocationSet locationSet;
    private final Map<ImageName, CompletableFuture<List<ImageHeaderData.Header>>> futureHeaderDataForImage = new ConcurrentHashMap<ImageName, CompletableFuture<List<ImageHeaderData.Header>>>();
    private final Map<ImageName, CompletableFuture<Double>> futureMeasuredShutterTimeForImage = new ConcurrentHashMap<ImageName, CompletableFuture<Double>>();
    private final Map<ImageName, FutureShutterProfiles> futureShutterMotionProfileForImage = new ConcurrentHashMap<ImageName, FutureShutterProfiles>();
    private final Map<ImageName, ImageProblematicHeaders> problematicHeadersForImage = new ConcurrentHashMap<ImageName, ImageProblematicHeaders>();
    private final CommandExecutor commandExecutor;
    private final AlertService alertService;

    PostImageFileHandling(ImageHandlingConfig config, CommandExecutor commandExecutor, AlertService alertService) {
        this.config = config;
        this.isHeaderServiceEnabled = new AtomicBoolean(true);
        this.commandExecutor = commandExecutor;
        this.alertService = alertService;
        this.locationSet = config.getLocationsToProcess();
    }

    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName) {
        boolean needToWaitForHeaderService = this.config.getWaitForHeaderService() == WaitForHeaderService.ALWAYS || this.config.getWaitForHeaderService() == WaitForHeaderService.AUTO && this.isHeaderServiceEnabled.get();
        needToWaitForHeaderService = needToWaitForHeaderService && imageName.getController() != Controller.CCS;
        boolean needToWaitForMeasuredShutterTime = imageName.getSource() == Source.MainCamera;
        return this.handleAsynchronousData(filesToProcess, imageName, needToWaitForHeaderService, needToWaitForMeasuredShutterTime);
    }

    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName, boolean needToWaitForHeaderService, boolean needToWaitForMeasuredShutterTime) {
        CompletionStage<Object> resultingFileList = filesToProcess;
        if (needToWaitForHeaderService) {
            CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = this.getHeaderDataFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureHeaderData, (fl, headerData) -> this.mergeHeaderServiceData(imageName, (FileList)fl, (List<ImageHeaderData.Header>)headerData));
        }
        if (needToWaitForMeasuredShutterTime) {
            CompletableFuture<Double> futureMeasuredShutterTimeData = this.getMeasuredShutterTimeFutureForImage(imageName);
            CompletionStage combined = resultingFileList.thenCombine(futureMeasuredShutterTimeData, (fl, measuredShutterTime) -> new FileListAndShutterTime(this.mergeShutterTimeData(imageName, (FileList)fl, (Double)measuredShutterTime), (Double)measuredShutterTime));
            FutureShutterProfiles futureShutterMotionProfile = this.getShutterMotionProfilesForImage(imageName).completeOnTimeout(null, 15L, TimeUnit.SECONDS);
            resultingFileList = ((CompletableFuture)combined).thenCombine((CompletionStage)futureShutterMotionProfile, (fileListAndShutterTime, jsonFiles) -> {
                Double shutterTime = fileListAndShutterTime.getShutterTime();
                if (shutterTime != null && shutterTime > 0.0) {
                    return this.mergeShutterMotionProfiles(imageName, fileListAndShutterTime.getFilelist(), (List<JsonFile>)jsonFiles);
                }
                return fileListAndShutterTime.getFilelist();
            });
        }
        return resultingFileList;
    }

    FileList mergeHeaderServiceData(ImageName imageName, FileList fl, List<ImageHeaderData.Header> headerData) {
        String message = null;
        if (headerData == null) {
            message = String.format("Header service data for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                this.fixupFitsFiles(fl, headerData, false, imageName);
            }
            catch (IOException | FitsException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && this.alertService != null && this.config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert("missingHeaders", "Missing or malformed header service data");
            this.alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;
    }

    FileList mergeShutterTimeData(ImageName imageName, FileList fl, Double measuredShutterTime) {
        String message = null;
        if (measuredShutterTime == null) {
            message = String.format("The measured shutter time for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                ArrayList<ImageHeaderData.Header> changedHeaders = new ArrayList<ImageHeaderData.Header>();
                ImageHeaderData.Header header = new ImageHeaderData.Header("SHUTTIME", (Object)measuredShutterTime, "Shutter Exposure Time");
                changedHeaders.add(header);
                if (measuredShutterTime > 0.0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header("XPOSURE", (Object)measuredShutterTime, "Effective exposure duration");
                    changedHeaders.add(header2);
                }
                this.fixupFitsFiles(fl, changedHeaders, true, imageName);
            }
            catch (IOException | FitsException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && this.alertService != null && this.config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert("missingHeaders", "Missing measured shutter data");
            this.alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;
    }

    FileList mergeShutterMotionProfiles(ImageName imageName, FileList fl, List<JsonFile> jsonFiles) {
        if (jsonFiles == null || jsonFiles.size() < 2) {
            String message = String.format("The shutter motion profiles for %s did not arrive in a timely manner jsonFiles=%s", imageName, jsonFiles);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                JsonFile openJson = jsonFiles.get(0);
                JsonFile closeJson = jsonFiles.get(1);
                if (openJson.getFileName().endsWith("Close")) {
                    openJson = jsonFiles.get(1);
                    closeJson = jsonFiles.get(0);
                }
                this.addShutterMotionProfilesToFitsFile(fl, openJson, closeJson, imageName);
            }
            catch (IOException | FitsException x) {
                String message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        return fl;
    }

    void handleFitsFileCommands(CompletableFuture<FileList> filesToProcess, Location location, ImageName imageName, String mode) {
        if (!this.config.getCommands().isEmpty()) {
            filesToProcess.thenAccept(fl -> {
                HashMap<String, String> env = new HashMap<String, String>();
                env.put("BOARD", location.getBoardName());
                env.put("RAFT", location.getRaftName());
                env.put("IMAGENAME", imageName.toString());
                env.put("MODE", mode);
                env.put("SEQNO", imageName.getNumberString());
                env.put("DATE", imageName.getDateString());
                env.put("CONTROLLER", imageName.getController().getCode());
                env.put("SOURCE", imageName.getSource().getCode());
                this.commandExecutor.executeFitsFileList((FileList)fl, (Map<String, String>)env, location, imageName);
            });
        }
    }

    void completePostImageFileHandlingForImage(ImageName imageName) {
        ImageProblematicHeaders imageProblematicHeaders = this.getImageProblematicHeaders(imageName);
        imageProblematicHeaders.logIfNeeded();
        this.futureHeaderDataForImage.remove(imageName);
        this.futureMeasuredShutterTimeForImage.remove(imageName);
        this.futureShutterMotionProfileForImage.remove(imageName);
        this.problematicHeadersForImage.remove(imageName);
    }

    void fixupFitsFiles(FileList fileList, List<ImageHeaderData.Header> headerData, boolean overwrite, ImageName imageName) throws FitsException, IOException {
        ImageProblematicHeaders imageProblematicHeaders = this.getImageProblematicHeaders(imageName);
        for (File file : fileList) {
            Fits fits = new Fits(file);
            BasicHDU hdu = fits.getHDU(0);
            Header header = hdu.getHeader();
            long deltaCheckSum = 0L;
            for (ImageHeaderData.Header headerServiceHeader : headerData) {
                String newValueAsString;
                String keyword = headerServiceHeader.getKeyword();
                if (keyword == null || "COMMENT".equals(keyword) || "FILENAME".equals(keyword)) continue;
                if (keyword.startsWith("HIERARCH")) {
                    keyword = keyword.replace(" ", ".");
                }
                HeaderCard card = header.findCard(keyword);
                Object newValue = headerServiceHeader.getValue();
                String string = newValueAsString = newValue == null ? null : newValue.toString();
                if (card != null) {
                    String oldValue = card.getValue();
                    if ((oldValue == null || oldValue.isEmpty() || overwrite) && newValue != null) {
                        long oldCheckSum = FitsCheckSum.checksum((byte[])AsciiFuncs.getBytes((String)card.toString()));
                        if (newValue instanceof String) {
                            String newValueString = (String)newValue;
                            if (!newValueString.isEmpty()) {
                                if (card.isStringValue()) {
                                    card.setValue(newValueString);
                                } else {
                                    card = new HeaderCard(card.getKey(), newValueString, card.getComment());
                                    header.updateLine(card.getKey(), card);
                                }
                            }
                        } else {
                            if (card.isStringValue()) {
                                LOG.log(Level.WARNING, "Header service keyword {0} of type {1} but isString is true", new Object[]{keyword, newValue.getClass()});
                            }
                            if (newValue instanceof Double) {
                                Double newValueDouble = (Double)newValue;
                                if (!Double.isNaN(newValueDouble)) {
                                    card.setValue(newValueDouble.doubleValue());
                                }
                            } else if (newValue instanceof Boolean) {
                                Boolean newValueBoolean = (Boolean)newValue;
                                card.setValue(newValueBoolean.booleanValue());
                            } else if (newValue instanceof Integer) {
                                Integer newValueInteger = (Integer)newValue;
                                card.setValue(newValueInteger.intValue());
                            } else {
                                LOG.log(Level.WARNING, "Header service keyword {0} of unexpected type {1} ignored", new Object[]{keyword, newValue.getClass()});
                            }
                        }
                        long newCheckSum = FitsCheckSum.checksum((byte[])AsciiFuncs.getBytes((String)card.toString()));
                        deltaCheckSum += newCheckSum - oldCheckSum;
                        continue;
                    }
                    Class cardType = card.valueType();
                    boolean differ = true;
                    if (oldValue != null && newValue != null) {
                        oldValue = oldValue.trim();
                        if (cardType == Boolean.class) {
                            differ = oldValue.toLowerCase().charAt(0) != newValueAsString.toLowerCase().charAt(0);
                        } else {
                            boolean bl = differ = !Objects.equals(oldValue, newValueAsString);
                        }
                    }
                    if (!differ) continue;
                    CardValues cardValues = new CardValues(keyword, newValueAsString, oldValue);
                    imageProblematicHeaders.addCardValue(cardValues);
                    continue;
                }
                CardValues cardValues = new CardValues(keyword, newValueAsString, null);
                imageProblematicHeaders.addCardValue(cardValues);
            }
            FitsCheckSum.updateCheckSum((Header)header, (long)deltaCheckSum);
            header.rewrite();
        }
    }

    private void addShutterMotionProfilesToFitsFile(FileList fileList, JsonFile openFile, JsonFile closeFile, ImageName imageName) throws FitsException, JsonProcessingException, IOException {
        BinaryTableHDU openHDU = ShutterMotionProfileHandler.createHDU(openFile.toString());
        BinaryTableHDU closeHDU = ShutterMotionProfileHandler.createHDU(closeFile.toString());
        for (File file : fileList) {
            ShutterMotionProfileHandler.appendHDUsToFitsFile(file, openHDU, closeHDU);
        }
        ArrayList<String> primaryHeaderEntries = new ArrayList<String>();
        primaryHeaderEntries.add("HIERARCH.SHUTTER.STARTTIME.TAI.ISOT");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.STARTTIME.TAI.MJD");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.SIDE");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.MODEL");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.MODELSTARTTIME");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.PIVOTPOINT1");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.PIVOTPOINT2");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK0");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK1");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK2");
        ArrayList<ImageHeaderData.Header> headers = new ArrayList<ImageHeaderData.Header>();
        for (String primaryHeaderEntry : primaryHeaderEntries) {
            String openSide = openHDU.getHeader().getStringValue(primaryHeaderEntry);
            if (openSide != null) {
                ImageHeaderData.Header header1 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.OPEN."), (Object)openSide, "");
                headers.add(header1);
            } else {
                double openDouble = openHDU.getHeader().getDoubleValue(primaryHeaderEntry, -999.0);
                if (openDouble != -999.0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.OPEN."), (Object)openDouble, "");
                    headers.add(header2);
                }
            }
            String closeSide = closeHDU.getHeader().getStringValue(primaryHeaderEntry);
            if (closeSide != null) {
                ImageHeaderData.Header header1 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.CLOSE."), (Object)closeSide, "");
                headers.add(header1);
                continue;
            }
            double closeDouble = closeHDU.getHeader().getDoubleValue(primaryHeaderEntry, -999.0);
            if (closeDouble == -999.0) continue;
            ImageHeaderData.Header header2 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.CLOSE."), (Object)closeDouble, "");
            headers.add(header2);
        }
        this.fixupFitsFiles(fileList, headers, true, imageName);
    }

    void headerDataArrived(ImageName imageName, List<ImageHeaderData.Header> headers) {
        CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = this.getHeaderDataFutureForImage(imageName);
        futureHeaderData.complete(headers);
    }

    void measuredShutterTimeArrived(MeasuredShutterTime shutterTime) {
        CompletableFuture<Double> futureMeasuredShutterTime = this.getMeasuredShutterTimeFutureForImage(shutterTime.getImageName());
        futureMeasuredShutterTime.complete(shutterTime.getMeasuredShutterOpenTime());
    }

    void shutterMotionProfileArrived(JsonFile jsonFile) {
        FutureShutterProfiles futureShutterMotionProfile = this.getShutterMotionProfilesForImage(jsonFile.getObsId());
        futureShutterMotionProfile.add(jsonFile);
    }

    private CompletableFuture<List<ImageHeaderData.Header>> getHeaderDataFutureForImage(ImageName imageName) {
        return this.futureHeaderDataForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Object>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15).intValue(), TimeUnit.SECONDS));
    }

    private CompletableFuture<Double> getMeasuredShutterTimeFutureForImage(ImageName imageName) {
        return this.futureMeasuredShutterTimeForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Object>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15).intValue(), TimeUnit.SECONDS));
    }

    private FutureShutterProfiles getShutterMotionProfilesForImage(ImageName imageName) {
        return this.futureShutterMotionProfileForImage.computeIfAbsent(imageName, s -> new FutureShutterProfiles());
    }

    private ImageProblematicHeaders getImageProblematicHeaders(ImageName imageName) {
        return this.problematicHeadersForImage.computeIfAbsent(imageName, s -> new ImageProblematicHeaders((ImageName)s));
    }

    void setHeaderServiceEnabled(boolean enabled) {
        this.isHeaderServiceEnabled.set(enabled);
    }

    CompletableFuture<File> handleAdditionalFile(AdditionalFile additionalFile) {
        if (!this.config.getAdditionalFileCommands().isEmpty()) {
            HashMap<String, String> props = new HashMap<String, String>();
            props.put("FileName", additionalFile.getFileName());
            props.put("FileType", additionalFile.getFileType());
            ImageName obsId = additionalFile.getObsId();
            props.put("ImageName", obsId.toString());
            props.put("ImageDate", obsId.getDateString());
            props.put("ImageNumber", obsId.getNumberString());
            props.put("ImageController", obsId.getController().getCode());
            props.put("ImageSource", obsId.getSource().getCode());
            props.put("SEQNO", obsId.getNumberString());
            props.put("DATE", obsId.getDateString());
            props.put("CONTROLLER", obsId.getController().getCode());
            props.put("SOURCE", obsId.getSource().getCode());
            props.put("MODE", "OTHER");
            File file = this.config.getAdditionalFileName(additionalFile, props);
            try {
                try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));){
                    additionalFile.writeFile((OutputStream)out);
                }
                Map<String, String> env = props.entrySet().stream().collect(Collectors.toMap(e -> ((String)e.getKey()).toUpperCase(), e -> (String)e.getValue()));
                return this.commandExecutor.executeAdditional(file, obsId, additionalFile.getFileType(), env);
            }
            catch (IOException x) {
                LOG.log(Level.SEVERE, "Error handling additional file: " + additionalFile.getFileName(), x);
                return CompletableFuture.failedFuture(x);
            }
        }
        return CompletableFuture.completedFuture(null);
    }

    private static class FutureShutterProfiles
    extends CompletableFuture<List<JsonFile>> {
        private final List<JsonFile> files = new ArrayList<JsonFile>(2);

        private FutureShutterProfiles() {
        }

        void add(JsonFile file) {
            this.files.add(file);
            if (this.files.size() == 2) {
                this.complete(this.files);
            }
        }

        public FutureShutterProfiles completeOnTimeout(List<JsonFile> value, long timeout, TimeUnit unit) {
            super.completeOnTimeout(value, timeout, unit);
            return this;
        }
    }

    private static class ImageProblematicHeaders {
        private final Map<String, List<CardValues>> warningsDifferMap = new ConcurrentHashMap<String, List<CardValues>>();
        private final Map<String, List<CardValues>> warningsNullCardMap = new ConcurrentHashMap<String, List<CardValues>>();
        private final ImageName imageName;

        ImageProblematicHeaders(ImageName imageName) {
            this.imageName = imageName;
        }

        void addCardValue(CardValues cardValues) {
            String keyword;
            Map<String, List<CardValues>> map = cardValues.ccsValue != null ? this.warningsDifferMap : this.warningsNullCardMap;
            List list = map.computeIfAbsent(keyword = cardValues.headerKeyword, k -> new CopyOnWriteArrayList());
            if (!list.contains(cardValues)) {
                list.add(cardValues);
            }
        }

        void logIfNeeded() {
            if (!this.warningsDifferMap.isEmpty() || !this.warningsNullCardMap.isEmpty()) {
                StringBuilder sb = new StringBuilder("The following header cards inconsistencies have been detected between the CCS value and the header service provided value for image " + this.imageName + ":\n");
                if (!this.warningsDifferMap.isEmpty()) {
                    sb.append("\nThe following header keywords have different values:\n");
                    for (List<CardValues> list : this.warningsDifferMap.values()) {
                        for (CardValues cv : list) {
                            sb.append("   ").append(cv.toString()).append("\n");
                        }
                    }
                }
                if (!this.warningsNullCardMap.isEmpty()) {
                    sb.append("\nThe following header keywords are null in CCS:\n");
                    for (List<CardValues> list : this.warningsNullCardMap.values()) {
                        for (CardValues cv : list) {
                            sb.append("   ").append(cv.toString()).append("\n");
                        }
                    }
                }
                sb.append("\n");
                LOG.log(Level.INFO, sb.toString());
            }
        }
    }

    private static class CardValues {
        private final String serviceValue;
        private final String ccsValue;
        private final String headerKeyword;

        CardValues(String headerKeyword, String serviceValue, String ccsValue) {
            this.serviceValue = serviceValue;
            this.ccsValue = ccsValue;
            this.headerKeyword = headerKeyword;
        }

        public boolean equals(Object obj) {
            CardValues in = (CardValues)obj;
            if (!Objects.equals(in.serviceValue, this.serviceValue)) {
                return false;
            }
            if (!Objects.equals(in.headerKeyword, this.headerKeyword)) {
                return false;
            }
            return Objects.equals(in.ccsValue, this.ccsValue);
        }

        public int hashCode() {
            int hash = 7;
            hash = 73 * hash + Objects.hashCode(this.serviceValue);
            hash = 73 * hash + Objects.hashCode(this.ccsValue);
            hash = 173 * hash + Objects.hashCode(this.headerKeyword);
            return hash;
        }

        public String toString() {
            return this.headerKeyword + "\t header_service: " + this.serviceValue + "\t ccs: " + this.ccsValue;
        }
    }

    private static class FileListAndShutterTime {
        private final FileList filelist;
        private final Double shutterTime;

        public FileListAndShutterTime(FileList filelist, Double shutterTime) {
            this.filelist = filelist;
            this.shutterTime = shutterTime;
        }

        public FileList getFilelist() {
            return this.filelist;
        }

        public Double getShutterTime() {
            return this.shutterTime;
        }
    }
}

