package org.lsst.ccs.subsystem.imagehandling.imagedb;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.SensorLocation;

/**
 * Opens a connect to an image database, and updates the database at the
 * completion of each image.
 *
 * @author tonyj
 */
public class ImageFileDatabase implements AutoCloseable {

    private final static int MAX_RETRIES = 1;

    private Connection conn;
    private static final Logger LOG = Logger.getLogger(ImageFileDatabase.class.getName());
    private final ImageDatabaseService imageDatabaseService;
    private final String host;
    private PreparedStatement stmt1;
    private PreparedStatement stmt2;
    private PreparedStatement stmt3;

    public ImageFileDatabase(ImageDatabaseService imageDatabaseService) {
        this.imageDatabaseService = imageDatabaseService;
        String localHost;
        try {
            localHost = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException ex) {
            localHost = "Unknown";
        }
        this.host = localHost;
    }

    public void openConnection() throws SQLException {
        this.conn = imageDatabaseService.openConnection();
        try (PreparedStatement stmt = conn.prepareStatement("""
                create table if not exists ccs_image_file (
                file_id int not null AUTO_INCREMENT, 
                telCode varchar(2) not null, 
                seqnum integer not null, 
                dayobs varchar(8) not null, 
                controller varchar(2) not null,
                fileType varchar(24) not null,
                raft varchar(3) not null,
                sensor varchar(3) not null,
                writeHost varchar(64) not null, 
                writeFile varchar(255) not null, 
                primary key (telCode, seqnum, dayobs, controller, fileType, raft, sensor), 
                index(file_id)
                )""")) {
            stmt.execute();
        }
        try (PreparedStatement stmt = conn.prepareStatement("""
                create table if not exists ccs_image_file_operation (
                file_id int not null, 
                operation varchar(20) not null, 
                operation_return_code int, 
                time float not null, 
                success boolean not null, 
                reason varchar(256), 
                primary key (file_id, operation)
                )""")) {
            stmt.execute();
        }
        stmt1 = conn.prepareStatement(
                "insert into ccs_image_file (telcode, seqnum, dayobs, controller, fileType, raft, sensor, writeHost, writeFile) "
                + "values (?, ?, ?, ?, ?,"
                + "        ?, ?, ?, ?) ", Statement.RETURN_GENERATED_KEYS);

        stmt2 = conn.prepareStatement(
                "insert into ccs_image_file_operation (file_id, operation, operation_return_code, time, success, reason) "
                + "values (?, ?, ?, ?, ?, ?)");

        stmt3 = conn.prepareStatement(
                "insert into ccs_image_file_operation (file_id, operation, operation_return_code, time, success, reason) "
                + "values (?, ?, ?, ?, ?, ?) on duplicate key update operation=values(operation), operation_return_code=values(operation_return_code), time=values(time), success=values(success), reason=values(reason)");
    }

    public synchronized int insertFile(ImageName obsId, SensorLocation location, String fileType, String file) throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }
            try {
                stmt1.setString(1, obsId.getSource().getCode());
                stmt1.setInt(2, obsId.getNumber());
                stmt1.setString(3, obsId.getDateString());
                stmt1.setString(4, obsId.getController().getCode());
                stmt1.setString(5, fileType);
                if (location != null) {
                    Location rebLocation = location.getRebLocation();
                    stmt1.setString(6, rebLocation.getRaftName());
                    stmt1.setString(7, rebLocation.getSensorName(location.getSensor()));
                } else {
                    stmt1.setString(6, "NR");
                    stmt1.setString(7, "NS");
                }
                stmt1.setString(8, host);
                stmt1.setString(9, file);
                stmt1.execute();
                int auto_id;
                try (ResultSet rs = stmt1.getGeneratedKeys()) {
                    rs.next();
                    auto_id = rs.getInt(1);
                }
                conn.commit();
                return auto_id;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }

    public record Operation(int fileId, String operation, int operation_return_code, float time, boolean success, String reason) {

    }

    /**
     * Only used for testing
     *
     * @return
     */
    Connection getConnection() {
        return this.conn;
    }

    @Override
    public void close() throws SQLException {
        if (conn != null) {
            imageDatabaseService.closeConnection();
            this.conn = null;
        }
    }

    public synchronized void insertFileOperations(List<Operation> operations, boolean ins_update) throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }

            PreparedStatement stmt_exec = stmt2;
            if (ins_update) {
                stmt_exec = stmt3;
            }
            try {

                for (Operation operation : operations) {

                    stmt_exec.setInt(1, operation.fileId);
                    stmt_exec.setString(2, operation.operation);
                    stmt_exec.setInt(3, operation.operation_return_code);
                    stmt_exec.setFloat(4, operation.time);
                    stmt_exec.setBoolean(5, operation.success);
                    stmt_exec.setString(6, operation.reason);
                    
                    stmt_exec.addBatch();

                }
                stmt_exec.executeBatch();
                conn.commit();
                return;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }

    public synchronized List<String> missingFileNames(Date dateBefore) throws SQLException {
        List<String> fileNames = new ArrayList<>();
        try {

            PreparedStatement stmt = this.conn.prepareStatement("select ccs_image_file.file_id, ccs_image_file.writeFile  from ccs_image_file left join ccs_image_file_operation on ccs_image_file.file_id = ccs_image_file_operation.file_id where ccs_image_file_operation.file_id is null and ccs_image_file.dayobs < ? and ccs_image_file.writeHost = ?");
            stmt.setString(1, dateBefore.toString().replace("-", ""));
            stmt.setString(2, this.host);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {
                fileNames.add(rs.getString("writeFile"));
                //   System.out.println(" File ID Missing " + rs.getString("writeFile"));
            }

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return fileNames;
    }

    public synchronized List<String> failedOperationFileNames(Date dateBefore) throws SQLException {
        List<String> failedOperationFileNames = new ArrayList<>();
        try {

            PreparedStatement stmt = this.conn.prepareStatement("select ccs_image_file.writeFile, ccs_image_file_operation.file_id from ccs_image_file join ccs_image_file_operation on ccs_image_file.file_id = ccs_image_file_operation.file_id where ccs_image_file_operation.operation_return_code > 0 and ccs_image_file.dayobs < ? and ccs_image_file.writeHost = ?");
            stmt.setString(1, dateBefore.toString().replace("-", ""));
            stmt.setString(2, this.host);

            ResultSet rs = stmt.executeQuery();

            while (rs.next()) {
                failedOperationFileNames.add(rs.getString("writeFile"));
                //    System.out.println(" File ID Missing " + rs.getString("writeFile") + " " + rs.getInt("file_id"));
            }

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return failedOperationFileNames;

    }

    public record MissingFile(int fileId, String file, String fileType, String dayobs) {

    }

    public synchronized List<MissingFile> missingFailedUSDFFileNames(String dayobs, int limit) throws SQLException {
        List<MissingFile> missingFiles = new ArrayList<>();

        //  SQL Query thanks to Tony 
        PreparedStatement stmt = this.conn.prepareStatement("""
            select distinct f.file_id, f.writeFile, f.fileType, f.dayobs from ccs_image_file f 
            left join ccs_image_file_operation o on (f.file_id=o.file_id and operation like 'usdf_%_copy')  
            where dayobs=? and writeHost=? and (success is null or success is false)
            order by f.writeFile limit ?
        """);
        stmt.setString(1, dayobs);
        stmt.setString(2, this.host);
        stmt.setInt(3, limit);

        ResultSet rs = stmt.executeQuery();

        while (rs.next()) {
            MissingFile missingFile = new MissingFile(rs.getInt(1), rs.getString(2), rs.getString(3), rs.getString(4));
            missingFiles.add(missingFile);
        }

        return missingFiles;
    }

}
