package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.imagehandling.imagedb.ImageDatabaseService;
import org.lsst.ccs.subsystem.imagehandling.imagedb.ImageFileDatabase;

/**
 *
 * @author Farrukh Azfar
 */
public class Resender {

    private static final Logger LOG = Logger.getLogger(Resender.class.getName());

    public static void main(String[] args) throws SQLException, ExecutionException, InterruptedException {
        // Print out arguments 
        for (String arg : args) {
            System.out.println(arg);
        }

        String dbURL = args[0]; // Database URL
        String dayobs = args[1]; // Date of files we want to resend
        String command = args[2]; // The command to use to resend files
        int limit = args.length > 3  ? Integer.parseInt(args[3]) : 1; 
        int threads= args.length > 4 ? Integer.parseInt(args[4]) : 1;
        
        // Create the database connection.
        ImageDatabaseService imageDatabaseService = ImageDatabaseService.testInstance(dbURL);
        ImageFileDatabase ifd = new ImageFileDatabase(imageDatabaseService);
        ifd.openConnection();
        Queue<ImageFileDatabase.MissingFile> missingFiles = new ConcurrentLinkedQueue(ifd.missingFailedUSDFFileNames(dayobs, limit));
        LOG.log(Level.INFO, "Found {0} files to resend for {1}", new Object[]{missingFiles.size(), dayobs});

        // Create the command executor
        ImageHandlingConfig config = new ImageHandlingConfig();
        Map<String, String> env = new HashMap<>();
        
        // This will come from config
        env.put("USDF_BUCKET", "rubin-summit");
        env.put("INSTRUMENT", "LSSTCam");
        env.put("USDF_MC_HOST", "usdf");
        env.put("USDF_USE_S3DAEMON", "true");

        CommandExecutor commandExecutor = new CommandExecutor(config, null, ifd);

        Runnable task = () -> {
            try {
                for (;;) {
                    ImageFileDatabase.MissingFile missingFile = missingFiles.poll();
                    if (missingFile == null) {
                        break;  
                    }
                    LOG.log(Level.INFO, "Resending {0}", missingFile.file());
                    List<String> commands = Collections.singletonList(command);
                    File file = new File(missingFile.file());
                    String fileType = missingFile.fileType();
                    Map<String, String> newEnv = new LinkedHashMap(env);
                    newEnv.put("FILETYPE", fileType);
                    newEnv.put("FILE", file.getAbsolutePath());
                    newEnv.put("DATE", missingFile.dayobs());
                    if ("FITS".equals(missingFile.fileType())) {
                        newEnv.put("FITSFILE", file.getAbsolutePath());
                    }
                    CompletableFuture<File> futureFile = commandExecutor.executeMissingFile(commands, file, missingFile.fileId(), newEnv);
                    File logFile = futureFile.get();
                    LOG.log(Level.INFO, "Log {0}", logFile);
                }
            } catch (Throwable t) {
                LOG.log(Level.SEVERE, "Error handling missing files", t);
            }
        };
        ExecutorService threadPool = Executors.newFixedThreadPool(threads);
        for (int i=0; i<threads; i++) {
            threadPool.submit(task);
        }
        threadPool.shutdown();
        threadPool.awaitTermination(1, TimeUnit.DAYS);
    }
}
