package org.lsst.ccs.subsystem.imagehandling;

import java.io.Serializable;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.daq.utilities.FitsServiceInterface;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class ScienceHandling {

    private static final Logger LOG = Logger.getLogger(ScienceHandling.class.getName());

    private final ImageHandlingConfig imageHandlingConfig;
    private final FitsService fitsService;
    private final PostImageFileHandling postImageFileHandling;
    private final FocalPlane geometry;
    private final Agent agent;
    private final AgentStatusAggregatorService statusAggregator;
    private final Map<String, Map<String, Serializable>> metaDataSetsForImage = new ConcurrentHashMap<>();

    private ThreadPoolExecutor executor;
    private Store scienceStore;
    private volatile ImageHandler imageHandlerPendingDarktime = null;
    private volatile boolean darkTimeArrived;
    private volatile String runNumber;
    private String previousRunNumber;
    private List<RebNode> rebNodes;

    ScienceHandling(ImageHandlingConfig imageHandlingConfig, FocalPlane geometry, FitsService fitsService, PostImageFileHandling postImageFileHandling, Agent agent, AgentStatusAggregatorService statusAggregator) {
        this.imageHandlingConfig = imageHandlingConfig;
        this.geometry = geometry;
        this.fitsService = fitsService;
        this.postImageFileHandling = postImageFileHandling;
        this.agent = agent;
        this.statusAggregator = statusAggregator;
    }

    void start() throws DAQException {
        Semaphore semaphore = new Semaphore(imageHandlingConfig.getDaqThreads());
        int nLocations = imageHandlingConfig.getLocationsToProcess().size();
        final int nThreads = Math.max(20, nLocations + 2); // Temporary auxtel+comcam fix
        // pre-create all the stores we will need
        ConcurrentLinkedDeque<Store> stores = new ConcurrentLinkedDeque<>();
        for (int i = 0; i < nThreads; i++) {
            stores.add(new Store(imageHandlingConfig.getDaqPartition()));
        }
        ThreadFactory readThreadFactory = (Runnable r) -> new ReadThread(r, stores, semaphore);

        executor = new ThreadPoolExecutor(nThreads, nThreads, 60L,
                TimeUnit.SECONDS, new SynchronousQueue<>(), readThreadFactory);

        // RebNodes are no longer created in the groovy, so we make them here
        rebNodes = new ArrayList<>();
        for (Location location : imageHandlingConfig.getLocationsToProcess()) {
            final Reb reb = geometry.getRebAtLocation(location);
            fitsService.addReb(reb, "science");
            FitsServiceInterface fitsServiceForReb = fitsService.getFitsServiceForReb(reb);
            RebNode rebNode = new RebNode(reb, fitsServiceForReb);
            rebNodes.add(rebNode);
        }

        scienceStore = new Store(imageHandlingConfig.getDaqPartition());
        scienceStore.addImageListener(new ImageListener() {

            @Override
            public void imageCreated(Image image) {
                if (imageHandlingConfig.isUseStreaming()) {
                    if (!checkFolder(image)) {
                        return;
                    }
                    handleImage(image, true);
                }
            }

            @Override
            public void imageComplete(Image image) {
                KeyValueData kvd = new KeyValueData(ImageReceivedEvent.EVENT_KEY, new ImageReceivedEvent(image.getMetaData()));
                agent.publishSubsystemDataOnStatusBus(kvd);
                if (!imageHandlingConfig.isUseStreaming()) {
                    if (!checkFolder(image)) {
                        return;
                    }
                    handleImage(image, false);
                }
            }

            private void handleImage(Image image, boolean isStreaming) {
                LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
                locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
                ImageHandler imageHandler;
                synchronized (ScienceHandling.this) {
                    if (!Objects.equals(runNumber, previousRunNumber)) {
                        LOG.log(Level.INFO, () -> String.format("Run number changed from %s to %s, clearing status aggregators", previousRunNumber, runNumber));
                        previousRunNumber = runNumber;
                        Instant cutoff = Instant.now().minus(Duration.ofMinutes(1));
                        for (String subsystemToClear : imageHandlingConfig.getSubsystemsToClear()) {
                            long start = System.nanoTime();
                            statusAggregator.clearAgentDataOlderThan(subsystemToClear, cutoff);
                            LOG.log(Level.INFO, () -> String.format("Clearing status aggregator for %s took %,dns", subsystemToClear, System.nanoTime() - start));
                        }
                        runNumber = null;
                    }

                    imageHandler = new ImageHandler(getMetaDataSet(image.getMetaData().getName()), image, executor, imageHandlingConfig, rebNodes, isStreaming, postImageFileHandling);
                    synchronized (ScienceHandling.this) {
                        if (darkTimeArrived) {
                            imageHandler.darkTimeArrived();
                            darkTimeArrived = false;
                        } else {
                            imageHandlerPendingDarktime = imageHandler;
                        }
                    }
                }
                Future<FileList> future = executor.submit(imageHandler);
                executor.submit(() -> {
                    try {
                        FileList filelist = future.get(Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.FitsTimeoutSeconds", 30), TimeUnit.SECONDS);
                        ImageEventSender sender = new ImageEventSender(filelist, null, agent, locationsWritten, image.getMetaData().getName());
                        sender.run();
                    } catch (InterruptedException | ExecutionException | TimeoutException x) {
                        ImageEventSender sender = new ImageEventSender(null, x, agent, locationsWritten, image.getMetaData().getName());
                        sender.run();
                    }
                });
            }

            private boolean checkFolder(Image image) {
                // Check if image is in expected folder
                final String daqFolder = imageHandlingConfig.getDaqFolder();
                if (daqFolder != null && !"".equals(daqFolder)) {
                    if (!imageHandlingConfig.getDaqFolder().equals(image.getMetaData().getCreationFolderName())) {
                        return false;
                    }
                }
                return true;
            }
        });

    }

    void stop() throws DAQException {
        // If we were already started, shutdown existing thread pool
        if (executor != null) {
            executor.shutdown();
            try {
                LOG.log(Level.INFO, "Shutting down science handling executor with {0} active tasks", executor.getActiveCount());
                boolean success = executor.awaitTermination(30, TimeUnit.SECONDS);
                if (!success) {
                    LOG.log(Level.WARNING, "Timed out waiting for science handling executor to shutdown");
                }
            } catch (InterruptedException x) {
                LOG.log(Level.WARNING, "Unexpected interrupt exception while waiting for restart", x);
            }
            executor = null;
        }
        if (scienceStore != null) {
            scienceStore.close();
            scienceStore = null;
        }
    }

    /**
     * Called when darktime arrives. The message does not include an image name,
     * so for now we will have to make some assumptions. In future we may need
     * to include the image name in the event.
     */
    void darkTimeArrived(String imageName) {
        synchronized (this) {
            // If ExposureTime was not explicitly set, then we attempt to set it here
            Map<String, Serializable> existingMap = getMetaDataSet(imageName);
            if (!existingMap.containsKey("ExposureTime")) {
                Object start = existingMap.get("startIntegrationTime");
                Object end =  existingMap.get("endIntegrationTime");
                // Note instanceof fails if null
                if (start instanceof CCSTimeStamp && end instanceof CCSTimeStamp) {
                    CCSTimeStamp tsStart = (CCSTimeStamp) start;
                    CCSTimeStamp tsEnd = (CCSTimeStamp) end;
                    double exposureTime = Duration.between(tsStart.getTAIInstant(), tsEnd.getTAIInstant()).toMillis()/1000.0;
                    LOG.log(Level.INFO, "Setting missing ExposureTime to {0}", exposureTime);
                    existingMap.put("ExposureTime", exposureTime);
                }
            }
            if (imageHandlerPendingDarktime != null) {
                imageHandlerPendingDarktime.darkTimeArrived();
                imageHandlerPendingDarktime = null;
            } else {
                darkTimeArrived = true;
            }
        }
    }

    void runNumberArrived(String runNumber) {
        synchronized (this) {
            this.runNumber = runNumber;
        }
    }

    // LSSTCCSRAFTS-678 Store the metadata for a give imageName.
    void addMetaData(String imageName, Map<String, Serializable> headerMap) {
        LOG.log(Level.INFO, "Adding metadata keys for image {0}: {1}", new Object[]{imageName, headerMap.keySet()});
        Map<String, Serializable> existingMap = getMetaDataSet(imageName);
        existingMap.putAll(headerMap);
    }

    // LSSTCCSRAFTS-678 Get the metadata for a given imageName and remove it.
    Map<String, Serializable> getMetaDataSet(String imageName) {
        return metaDataSetsForImage.computeIfAbsent(imageName, (in) -> new ConcurrentHashMap<>());
    }

    void simulateTrigger(Location location, ImageMetaData meta, int[] registerList, Path rawData) throws DAQException {
        scienceStore.simulateTrigger(location, meta, registerList, rawData);
    }

    FileList fetchImage(String imageName) throws DAQException, InterruptedException, ExecutionException {
        Store store = new Store(imageHandlingConfig.getDaqPartition());
        final String daqFolder = imageHandlingConfig.getDaqFolder();
        Folder imageFolder = store.getCatalog().find(daqFolder);
        if (imageFolder == null) {
            throw new RuntimeException("Folder " + daqFolder + " not found");
        }
        Image image = imageFolder.find(imageName);
        if (image == null) {
            throw new RuntimeException("Image " + imageName + " not found");
        }
        LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
        locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
        Future<FileList> future = executor.submit(new ImageHandler(getMetaDataSet(imageName), image, executor, imageHandlingConfig, rebNodes, false, postImageFileHandling));
        return future.get();
    }

}
