package org.lsst.ccs.subsystem.imagehandling;

import java.io.Serializable;
import java.nio.file.Path;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.ConfigurationListener;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Singleton initiated from groovy file
 *
 * @author tonyj
 */
public class ImageHandlingClient implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageHandlingClient.class.getName());

    @LookupPath
    private String path;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageHandlingConfig imageHandlingConfig;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService statusAggregator;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configurationService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private FitsService fitsService;

    private PostImageFileHandling postImageFileHandling;
    private final FocalPlane geometry;
    private GuiderHandling guiderHandling;
    private ScienceHandling scienceHandling;

    public static final String MISSING_HEADERSERVICEDATA_ALERTID = "missingHeaders";
    public static final Alert missingHeaderServiceDataAlert = new Alert(MISSING_HEADERSERVICEDATA_ALERTID, "Alert raised when header service data fails to arrive when expected.");

    public ImageHandlingClient(FocalPlane geometry) {
        this.geometry = geometry;
    }

    @Override
    public void init() {
        registerAlerts(alertService);
    }    

    @Override
    public void start() {
        LOG.log(Level.FINE, "Starting ImageHandling client {0})", path);
        CommandExecutor commandExecutor = new CommandExecutor(imageHandlingConfig, agent);
        postImageFileHandling = new PostImageFileHandling(imageHandlingConfig, commandExecutor, alertService);

    }

    @Override
    public void postStart() {
        try {
            scienceHandling = new ScienceHandling(imageHandlingConfig, geometry, fitsService, postImageFileHandling, agent, statusAggregator);
            startScience(scienceHandling);
            guiderHandling = new GuiderHandling(imageHandlingConfig, ForkJoinPool.commonPool(), geometry, fitsService, postImageFileHandling);
            startGuiding(guiderHandling);
            configurationService.addConfigurationListener(new ConfigurationListener() {
                @Override
                public void configurationChanged(ConfigurationInfo newConfigurationInfo, ConfigurationInfo oldConfigurationInfo, ConfigurationListener.ConfigurationOperation configurationOperation) {
                    List<ConfigurationParameterInfo> latestChanges = newConfigurationInfo.diff(oldConfigurationInfo);
                    if (!latestChanges.isEmpty()) {
                        boolean guiderRestartNeeded = false;
                        boolean scienceRestartNeeded = false;

                        for (ConfigurationParameterInfo change : latestChanges) {
                            switch (change.getParameterName()) {
                                case "guiderPartition":
                                    guiderRestartNeeded = true;
                                    break;
                                case "guiderLocations":
                                    guiderRestartNeeded = true;
                                    break;
                                case "daqPartition":
                                    scienceRestartNeeded = true;
                                    break;
                                case "locations":
                                    scienceRestartNeeded = true;
                                    break;
                            }
                        }
                        if (scienceRestartNeeded) {
                            LOG.info("Reloading image handling for science rafts due to configuration changes...");
                            try {
                                startScience(scienceHandling);
                            } catch (DAQException x) {
                                LOG.log(Level.SEVERE, "Failed to restart image handling for science rafts after config change");
                                // TODO: Go into fault state?
                            }
                        }
                        if (guiderRestartNeeded) {
                            LOG.info("Reloading guider due to configuration changes...");
                            try {
                                startGuiding(guiderHandling);
                            } catch (DAQException x) {
                                LOG.log(Level.SEVERE, "Failed to restart guiding after config change");
                                // TODO: Go into fault state?
                            }
                        }
                    }
                }
            });
        } catch (DAQException x) {
            // TODO: Is this the right thing to do? unfortunately HasLifecycle 
            // javadoc does not say what happens if an exception is thrown.
            throw new RuntimeException("Failed to connect to DAQ", x);
        }
    }

    private void startScience(ScienceHandling sh) throws DAQException {
        sh.stop();
        if (imageHandlingConfig.getDaqPartition() != null && !imageHandlingConfig.getDaqPartition().trim().isEmpty()) {
            sh.start();
        }
    }

    private void startGuiding(GuiderHandling gh) throws DAQException {
        gh.stop();
        String guiderPartition = imageHandlingConfig.getGuiderPartition();
        if (guiderPartition != null && !guiderPartition.trim().isEmpty()) {
            gh.start();
        }
    }

    FileList fetchImage(String imageName) throws DAQException, InterruptedException, ExecutionException {
        return scienceHandling.fetchImage(imageName);
    }

    /**
     * Called when darktime arrives. The message dose not include an image name,
     * so for now we will have to make some assumptions. In future we may need
     * to include the image name in the event.
     */
    void darkTimeArrived(String imageName) {
        scienceHandling.darkTimeArrived(imageName);
    }

    void runNumberArrived(String runNumber) {
        scienceHandling.runNumberArrived(runNumber);
    }

    /**
     * Called if and when header service data arrives
     *
     * @param imageName
     * @param headers
     */
    void headerDataArrived(String imageName, List<ImageHeaderData.Header> headers) {
        postImageFileHandling.headerDataArrived(imageName, headers);
    }

    void setHeaderServiceEnabled(boolean enabled) {
        postImageFileHandling.setHeaderServiceEnabled(enabled);
    }

    // LSSTCCSRAFTS-678 Store the DAQ metadata for a give imageName.
    // The method below is used to retrieve this metadata when the fits file is
    // written
    void addMetaData(String imageName, Map<String, Serializable> headerMap) {
        scienceHandling.addMetaData(imageName, headerMap);
    }

    void simulateTrigger(Location location, ImageMetaData meta, int[] registerList, Path rawData) throws DAQException {
        scienceHandling.simulateTrigger(location, meta, registerList, rawData);
    }

    boolean isPrimary() {
        return imageHandlingConfig.isPrimary();
    }

    void handleAdditionFile(AdditionalFile additionalFile) {
        postImageFileHandling.handleAdditionalFile(additionalFile);
    }
    
    void registerAlerts(AlertService alertService) {
        alertService.registerAlert(missingHeaderServiceDataAlert, new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        });        
    }
}
