package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ByteChannel;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Source;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter.PerCCDMetaDataProvider;
import org.lsst.ccs.subsystem.imagehandling.states.ImageHandlingState;
import org.lsst.ccs.utilities.ccd.CCD;

/**
 * A Callable which when called reads a single-source worth of data from the DAQ
 * and writes a set of FITS files.
 * @author tonyj
 */
class SourceHandler implements Callable<FileList> {

    private static final String TMP_POSTFIX = ".$tmp$";
    private final Source source;
    private final static int BUFFER_SIZE = 1024 * 1024;
    private final ImageHandlingConfig config;
    private final RebNode rebNode;

    /**
     * Create a new SourceHandler
     * @param source The source to read data from
     * @param config The configuration to use
     * @param rebNode  The REB corresponding to the source.
     */
    SourceHandler(Source source, ImageHandlingConfig config, RebNode rebNode) {
        this.source = source;
        this.config = config;
        this.rebNode = rebNode;
    }

    @Override
    public FileList call() throws IOException {

        rebNode.updateState(ImageHandlingState.READING);
        try {
            FileList result = new FileList();
            FitsIntWriter.FileNamer namer = (Map<String, Object> props) -> {
                File file = config.getFitsFile(props);
                // Note, this is the name the file will have (after it is closed and renamed)
                // rather than the temporary naem it has initially.
                props.put("OriginalFileName", file.getName());
                // We assign a temporary name for the initial write, which will be deleted in case of error
                File tmpFile = new File(file.getParent(), file.getName() + TMP_POSTFIX);
                return tmpFile;
            };
            PerCCDMetaDataProvider metaDataProvider = (CCD ccd) -> Collections.singletonList(rebNode.getFitsService().getFitsHeaderMetadataProvider(ccd.getUniqueId()));
            // Set up channels to write per amplifier data
            List<File> filesToBeWritten = null;
            IOException io = null;
            try (FitsIntWriter decompress = new FitsIntWriter(source, rebNode.getReb(), rebNode.getFitsService().getHeaderSpecificationMap(), namer, metaDataProvider);
                    ByteChannel readChannel = source.openChannel(Source.ChannelMode.READ)) {
                filesToBeWritten = decompress.getFiles();
                ByteBuffer bb = ByteBuffer.allocateDirect(BUFFER_SIZE);
                bb.order(ByteOrder.LITTLE_ENDIAN);
                for (;;) {
                    int l = readChannel.read(bb);
                    if (l < 0) {
                        break;
                    }
                    bb.flip();
                    decompress.write(bb.asIntBuffer());
                    bb.clear();
                }
            } catch (IOException x) {
                io = x;
                throw io;
            } finally {
                if (filesToBeWritten != null) {
                    if (io != null) {
                        for (File file : filesToBeWritten) {
                           // Note: This does not throw an exception on failure
                           file.delete();
                        }
                    } else {
                        for (File file : filesToBeWritten) {
                            String oldName = file.getName();
                            if (oldName.endsWith(TMP_POSTFIX)) {
                                final String newName = oldName.substring(0,oldName.length()-TMP_POSTFIX.length());
                                final File newFile = new File(file.getParent(), newName);
                                boolean success = file.renameTo(newFile);
                                if (!success) {
                                    /// This should be exceedingly rare, so we will not worry about cleanup in this case
                                    throw new IOException("Unable to rename "+oldName+" to "+newName);
                                }
                                result.add(newFile);
                            }
                            
                        }
                    }
                }
            }
            return result;
        } catch (IOException | DAQException ioe ) {
            throw new IOException("Exception for " + rebNode.getReb().getFullName(), ioe);
        } finally {
            rebNode.updateState(ImageHandlingState.IDLE);
        }
    }
}
