package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import java.util.Map;
import java.util.stream.Collectors;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.pattern.FileNamePatternUtils;

/**
 *
 * @author tonyj
 */
@SuppressWarnings("FieldMayBeFinal")
public class ImageHandlingConfig implements ConfigurationBulkChangeHandler {

    @ConfigurationParameter(category = "FitsHandling")
    private String FITSRootDirectory = System.getProperty("java.io.tmpdir");

    @ConfigurationParameter(category = "FitsHandling")
    private String FITSDirectoryPattern = "${ImageDate}/${ImageName}";

    @ConfigurationParameter(category = "FitsHandling")
    private String FITSFilePattern = "${ImageName}_${RaftBay}_${CCDSlot}.fits";

    @ConfigurationParameter(category = "FitsHandling")
    private boolean FITSAutoSave = true;
    
    @ConfigurationParameter(category = "DAQ")
    private String daqPartition = "dev";

    @ConfigurationParameter(category = "DAQ")
    private String daqFolder = "raw";
    
    @ConfigurationParameter(category = "DAQ")
    private String locations = "R22/Reb0,R22/Reb1,R22/Reb2";

    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        String dir = parametersView.get("FITSRootDirectory").toString();
        File dirFile = new File(dir);
        dirFile.mkdirs();
        if (!dirFile.isDirectory()) {
            throw new IllegalArgumentException("FITSRootDirectory: Not a directory");
        }
        if (!dirFile.canWrite()) {
            throw new IllegalArgumentException("FITSRootDirectory: Not writable");
        }
        // Validate that the locations syntax is valid.
        LocationSet.of(parametersView.get("locations").toString());
        // TODO: Also verify that the locations are a subset of the locations 
        // specified in the geometry/fits handlers.
    }

    public String getFITSRootDirectory() {
        return FITSRootDirectory;
    }

    public String getFITSDirectoryPattern() {
        return FITSDirectoryPattern;
    }

    public String getFITSFilePattern() {
        return FITSFilePattern;
    }

    public boolean isFITSAutoSave() {
        return FITSAutoSave;
    }

    public String getDaqPartition() {
        return daqPartition;
    }

    LocationSet getLocationsToProcess() {
        return LocationSet.of(locations);
    }

    public String getDaqFolder() {
        return daqFolder;
    }
    
    File getFitsFile(Map<String, Object> data) {
        // Convert input data to a map of String, String TODO: Move this to FileNamePatternUtils
        Map<String, String> props = data.entrySet().stream().collect(Collectors.toMap(e->e.getKey(), e->e.getValue().toString()));
        File root = new File(FITSRootDirectory);
        String fileDir = FileNamePatternUtils.resolveFileName(FITSDirectoryPattern, props);
        File dir = new File(root, fileDir);
        String fileName = FileNamePatternUtils.resolveFileName(FITSFilePattern, props);  
        File result = new File(dir, fileName);
        result.getParentFile().mkdirs();
        return result;
    }  
}
