package org.lsst.ccs.subsystem.imagehandling;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.imagehandling.data.HeaderServiceEnabled;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData.Header;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderKeywords;

/**
 * A class that listens to messages coming from the focal-plane subsystem.
 *
 * @author The LSST CCS Team
 */
class BusMessageHandler implements HasLifecycle {

    @SuppressWarnings("VolatileArrayField")
    private volatile String[] metaDataRegisters;
    private FocalPlaneConfigurationAndDataListener configListener;
    private OCSBridgeDataListener ocsBridgeDataListener;
    private static final Logger LOG = Logger.getLogger(BusMessageHandler.class.getName());

    private final Predicate<BusMessage<? extends Serializable, ?>> focalPlanePredicate = (bm) -> isFocalPlaneAgent(bm.getOriginAgentInfo());
    private final Predicate<BusMessage<? extends Serializable, ?>> ocsBridgePredicate = (bm) -> isOCSBridgeAgent(bm.getOriginAgentInfo());

    private final Predicate<BusMessage<? extends Serializable, ?>> headerKeywordsPredicate
            = BusMessageFilterFactory.messageClass(StatusSubsystemData.class).and((bm) -> ImageHeaderKeywords.IMAGE_HEADER_KEYWORDS.equals(((StatusSubsystemData) bm).getDataKey()));

    private final Predicate<BusMessage<? extends Serializable, ?>> headerDataPredicate
            = BusMessageFilterFactory.messageClass(StatusSubsystemData.class).and((bm) -> ImageHeaderData.EVENT_KEY.equals(((StatusSubsystemData) bm).getDataKey()));

        private final Predicate<BusMessage<? extends Serializable, ?>> headerServiceEnabledPredicate
            = BusMessageFilterFactory.messageClass(StatusSubsystemData.class).and((bm) -> HeaderServiceEnabled.EVENT_KEY.equals(((StatusSubsystemData) bm).getDataKey()));
    
    private final Predicate<BusMessage<? extends Serializable, ?>> configurationDataPredicate
            = BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class);

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageHandlingClient imageHandlingClient;

    @Override
    public void postStart() {
        // Called in postStart so the listener is registered (and thus called) after the FitsService (hopefully)
        configListener = new FocalPlaneConfigurationAndDataListener();
        agent.getMessagingAccess().addStatusMessageListener(configListener, focalPlanePredicate.and(configurationDataPredicate.or(headerKeywordsPredicate)));
        ocsBridgeDataListener = new OCSBridgeDataListener();
        agent.getMessagingAccess().addStatusMessageListener(ocsBridgeDataListener, ocsBridgePredicate.and(headerDataPredicate.or(headerServiceEnabledPredicate)));

    }

    @Override
    public void shutdown() {
        agent.getMessagingAccess().removeStatusMessageListener(configListener);
        agent.getMessagingAccess().removeStatusMessageListener(ocsBridgeDataListener);
    }

    public String[] getMetaDataRegisters() {
        return metaDataRegisters;
    }

    private static boolean isFocalPlaneAgent(AgentInfo agentInfo) {
        return AgentCategory.FOCAL_PLANE.name().equals(agentInfo.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY));
    }

    private static boolean isOCSBridgeAgent(AgentInfo agentInfo) {
        return AgentType.OCS_BRIDGE.equals(agentInfo.getType());
    }

    private class FocalPlaneConfigurationAndDataListener implements StatusMessageListener {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            if (msg instanceof StatusConfigurationInfo) {
                ConfigurationInfo ci = ((StatusConfigurationInfo) msg).getConfigurationInfo();
                for (ConfigurationParameterInfo cpi : ci.getAllParameterInfo()) {
                    if (cpi.getComponentName().equals("focalPlaneConfig")) {
                        if (cpi.getParameterName().equals("metaDataRegisters")) {
                            metaDataRegisters = (String[]) cpi.getCurrentValueObject();
                            LOG.log(Level.FINEST, "Updating metaDataRegister to {0}", new Object[]{Arrays.asList(metaDataRegisters)});
                        }
                    }
                }
            }

            if (msg instanceof StatusSubsystemData) {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                ImageHeaderKeywords keywords = (ImageHeaderKeywords)ssd.getObject().getValue();
                Map<String, Serializable> headerMap = keywords.getKeywords();
                LOG.log(Level.INFO, () -> String.format("Received image header keywords %s", keywords));
                if (headerMap.containsKey("darkTime")) {
                    imageHandlingClient.darkTimeArrived();
                } else if (headerMap.containsKey("RunNumber")) {
                    Object runNumber = headerMap.get("RunNumber");
                    imageHandlingClient.runNumberArrived(runNumber == null ? null : runNumber.toString());
                }
                    
                // LSSTCCSRAFTS-678 We collect here the daq metadata and forward it to the imageHandlingClient
                imageHandlingClient.addMetaData(keywords.getImageName(), keywords.getKeywords());
                    
            }
        }
    }

    private class OCSBridgeDataListener implements StatusMessageListener {

        @Override
        public void onStatusMessage(StatusMessage msg) {

            if (msg instanceof StatusSubsystemData) {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                if (ssd.getDataKey().equals(ImageHeaderData.EVENT_KEY)) {
                    ImageHeaderData data = (ImageHeaderData) ssd.getObject().getValue();
                    // Figure out what image these headers correspond to, and pass to the appropriate client
                    // Unfortnately header service does not provide the OBSID so we need to put the prices together
                    StringBuilder imageName = new StringBuilder("aa_b_cccccccc_dddddd");
                    for (Header header : data.getHeaders()) {
                        if (header.getKeyword() == null) continue;
                        final String value = header.getValue();
                        switch (header.getKeyword()) {
                            case "CAMCODE":
                                imageName.replace(0, 2, value);
                                break;
                            case "CONTRLLR":
                                imageName.replace(3, 4, value);
                                break;
                            case "DAYOBS":
                                imageName.replace(5, 13, value);
                                break;
                            case "SEQNUM":
                                imageName.replace(14, 20, String.format("%06d", Integer.parseInt(value)));
                                break;
                        }
                    }
                    LOG.log(Level.INFO, "Received image headers for {0}: {1}", new Object[]{imageName, data.getHeaders()});
                    imageHandlingClient.headerDataArrived(imageName.toString(), data.getHeaders());
                } else if (ssd.getDataKey().equals(HeaderServiceEnabled.EVENT_KEY)) {
                    HeaderServiceEnabled headerServiceEnabled = (HeaderServiceEnabled) ssd.getObject().getValue();
                    LOG.log(Level.INFO, "Received headerServiceEnabled: {0}", headerServiceEnabled.isEnabled());
                    imageHandlingClient.setHeaderServiceEnabled(headerServiceEnabled.isEnabled());
                }
            }
        }
    }
}
