package org.lsst.ccs.subsystem.imagehandling;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData.Header;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;
import org.lsst.ccs.utilities.location.LocationSet;

public class ImageHandlingClient implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageHandlingClient.class.getName());

    @LookupPath
    private String path;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageHandlingConfig imageHandlingConfig;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final List<RebNode> rebs = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService statusAggregator;
    
    private ExecutorService executor;
    private volatile Map<String, ImageHandler> imageHandlers = new ConcurrentHashMap<>();
    private volatile Map<String, Map<String,Serializable>> imageMetaDataSets = new ConcurrentHashMap<>();
    private volatile ImageHandler imageHandlerPendingDarktime = null;
    private volatile boolean darkTimeArrived;
    private volatile String runNumber;
    private String previousRunNumber;
    private CommandExecutor commandExecutor;
    private volatile boolean isHeaderServiceEnabled = true;

    @Override
    public void build() {
    }

    @Override
    public void start() {
        LOG.log(Level.FINE, "Starting ImageHandling client {0})", path);
    }
    
    @Override
    public void postStart() {
        try {
            commandExecutor = new CommandExecutor(imageHandlingConfig, agent);
            Semaphore semaphore = new Semaphore(imageHandlingConfig.getDaqThreads());
            int nLocations = imageHandlingConfig.getLocationsToProcess().size();
            final int nThreads = Math.max(20, nLocations+2); // Temporary auxtel+comcam fix
            // pre-create all the stores we will need
            ConcurrentLinkedDeque<Store> stores = new ConcurrentLinkedDeque<>();
            for (int i=0; i<nThreads; i++) {
                stores.add(new Store(imageHandlingConfig.getDaqPartition()));
            }
            ThreadFactory readThreadFactory = (Runnable r) -> new ReadThread(r, stores, semaphore);

            executor = new ThreadPoolExecutor(nThreads, nThreads, 60L,
                    TimeUnit.SECONDS, new SynchronousQueue<>(), readThreadFactory);

            Store store = new Store(imageHandlingConfig.getDaqPartition());
            store.addImageListener(new ImageListener() {
                
                @Override
                public void imageCreated(Image image) {
                    if (imageHandlingConfig.isUseStreaming()) {
                        if (!checkFolder(image)) return;
                        handleImage(image, true);
                    }
                }

                @Override
                public void imageComplete(Image image) {
                    KeyValueData kvd = new KeyValueData(ImageReceivedEvent.EVENT_KEY, new ImageReceivedEvent(image.getMetaData()));
                    agent.publishSubsystemDataOnStatusBus(kvd);
                    if (!imageHandlingConfig.isUseStreaming()) {
                        if (!checkFolder(image)) return;
                        handleImage(image, false);
                    }
                }

                private void handleImage(Image image, boolean isStreaming) {
                    LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
                    locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
                    ImageHandler imageHandler;
                    synchronized (ImageHandlingClient.this) {
                        if (!Objects.equals(runNumber, previousRunNumber)) {
                            LOG.log(Level.INFO, () -> String.format("Run number changed from %s to %s, clearing status aggregators", previousRunNumber, runNumber));
                            previousRunNumber = runNumber;
                            Instant cutoff = Instant.now().minus(Duration.ofMinutes(1));
                            for (String subsystemToClear : imageHandlingConfig.getSubsystemsToClear()) {
                                long start = System.nanoTime();
                                statusAggregator.clearAgentDataOlderThan(subsystemToClear, cutoff);
                                LOG.log(Level.INFO, () -> String.format("Clearing status aggregator for %s took %,dns", subsystemToClear, System.nanoTime() - start));  
                            }
                            runNumber = null;
                        }
                        
                        imageHandler = new ImageHandler(getMetaDataSet(image.getMetaData().getName()), image, executor, imageHandlingConfig, rebs, isStreaming, isHeaderServiceEnabled, commandExecutor);
                        if (darkTimeArrived) {
                            imageHandler.darkTimeArrived();
                            darkTimeArrived = false;
                        } else {
                            imageHandlerPendingDarktime = imageHandler;
                        }
                        imageHandlers.put(image.getMetaData().getName(), imageHandler);
                    }
                    Future<FileList> future = executor.submit(imageHandler);
                    executor.submit(() -> {
                        try {
                            FileList filelist = future.get(Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.FitsTimeoutSeconds", 30), TimeUnit.SECONDS);
                            ImageEventSender sender = new ImageEventSender(filelist, null, agent, locationsWritten, image.getMetaData().getName());
                            sender.run();
                        } catch (InterruptedException | ExecutionException | TimeoutException x) {
                            ImageEventSender sender = new ImageEventSender(null, x, agent, locationsWritten, image.getMetaData().getName());
                            sender.run();                            
                        } finally {
                            synchronized (ImageHandlingClient.this) {
                                imageHandlers.remove(image.getMetaData().getName());
                            }
                        }
                    });
                }

                private boolean checkFolder(Image image) {
                    // Check if image is in expected folder
                    final String daqFolder = imageHandlingConfig.getDaqFolder();
                    if (daqFolder != null && !"".equals(daqFolder)) {
                        if (!imageHandlingConfig.getDaqFolder().equals(image.getMetaData().getCreationFolderName())) {
                            return false;
                        }
                    }
                    return true;
                }
            });
        } catch (DAQException x) {
            // TODO: Is this the right thing to do? unfortunately HasLifecycle 
            // javadoc does not say what happens if an exception is thrown.
            throw new RuntimeException("Failed to connect to DAQ", x);
        }
    }

    ExecutorService getExecutorService() {
        return executor;
    }

    FileList fetchImage(String imageName) throws DAQException, InterruptedException, ExecutionException {
        Store store = new Store(imageHandlingConfig.getDaqPartition());
        final String daqFolder = imageHandlingConfig.getDaqFolder();
        Folder imageFolder = store.getCatalog().find(daqFolder);
        if (imageFolder == null) {
            throw new RuntimeException("Folder " + daqFolder + " not found");
        }
        Image image = imageFolder.find(imageName);
        if (image == null) {
            throw new RuntimeException("Image " + imageName + " not found");
        }
        LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
        locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
        Future<FileList> future = executor.submit(new ImageHandler(getMetaDataSet(imageName), image, executor, imageHandlingConfig, rebs, false, false, commandExecutor));
        return future.get();
    }
    
    /**
     * Called when darktime arrives. The message dose not include an image name, so 
     * for now we will have to make some assumptions. In future we may need to 
     * include the image name in the event.
     */
    void darkTimeArrived() {
        synchronized (ImageHandlingClient.this) {
            if (imageHandlerPendingDarktime != null) {
                imageHandlerPendingDarktime.darkTimeArrived();
                imageHandlerPendingDarktime = null;
            } else {
                darkTimeArrived = true;
            }
        } 
    }

    void runNumberArrived(String runNumber) {
        synchronized (ImageHandlingClient.this) {
            this.runNumber = runNumber;
        }
    }

    /** 
     * Called if and when header service data arrives
     * @param imageName
     * @param headers 
     */
    void headerDataArrived(String imageName, List<Header> headers) {
        synchronized (ImageHandlingClient.this) {
            ImageHandler imageHandler = imageHandlers.get(imageName);
            if (imageHandler == null) {
                LOG.log(Level.WARNING, "Image headers received for non-existent image {0}", imageName);
            } else {
                imageHandler.headerDataArrived(headers);
            }
        }
    }

    void setHeaderServiceEnabled(boolean enabled) {
        this.isHeaderServiceEnabled = enabled;
    }
    

    // LSSTCCSRAFTS-678 Store the DAQ metadata for a give imageName.
    // The method below is used to retrieve this metadata when the fits file is
    // written
    void addMetaData(String imageName, Map<String,Serializable> headerMap) {
        LOG.log(Level.INFO,"Adding metadata keys for image {0}: {1}", new Object[]{imageName,headerMap.keySet()});
        Map<String,Serializable> existingMap = getMetaDataSet(imageName);
        existingMap.putAll(headerMap);
    }
    
    // LSSTCCSRAFTS-678 Get the DAQ metadata for a given imageName and remove it.
    Map<String,Serializable> getMetaDataSet(String imageName) {
        return imageMetaDataSets.computeIfAbsent(imageName, (in) -> new ConcurrentHashMap<>());
    }
    
    
    
    
}
