package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;

/**
 * A call for executing commands on a FITS file
 *
 * @author tonyj
 */
class CommandExecutor {

    private static final Logger LOG = Logger.getLogger(CommandExecutor.class.getName());
    private static final Pattern TELEMETRY_PATTERN = Pattern.compile("Telemetry\\:\\s*(.*)\\:\\s*(.*)");
    private static final Pattern SLOT_PATTERN = Pattern.compile(".*_S(.*)\\.fits");

    private final ImageHandlingConfig config;
    private final Queue<Process> runningProcesses = new ConcurrentLinkedQueue<>();
    private final Agent agent;

    CommandExecutor(ImageHandlingConfig config, Agent agent) {
        this.config = config;
        this.agent = agent;
    }

    void execute(FileList fileList, Map<String, String> env) {
        for (File file : fileList) {
            execute(file, env);
        }
    }

    void execute(File fitsFile, Map<String, String> env) {
        // TODO: Chain multiple commands together
        List<String> commands = config.getCommands();
        List<String> command = List.of("/bin/bash", "-c", commands.stream().collect(Collectors.joining(";")));
        String date = env.getOrDefault("DATE", "UNKNOWN");
        final File logFileDir = new File(config.getLogDirectory(), date);
        File logFile = new File(logFileDir, fitsFile.getName() + ".log");
        Matcher matcher = SLOT_PATTERN.matcher(fitsFile.getName());
        String slot = matcher.matches() ? "S" + matcher.group(1) : "S??";
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.environment().putAll(config.getEnvironment());
        processBuilder.environment().putAll(env);
        processBuilder.environment().put("LOGFILE", logFile.getAbsolutePath());
        processBuilder.environment().put("FITSFILE", fitsFile.getAbsolutePath());
        processBuilder.environment().put("SLOT", slot);
        processBuilder.directory(config.getCurrentDirectory());
        processBuilder.redirectErrorStream(true);
        processBuilder.redirectOutput(logFile);
        try {
            Files.createDirectories(logFileDir.toPath());
            Process process = processBuilder.start();
            runningProcesses.add(process);
            LOG.log(Level.INFO, () -> String.format("Process for file %s started", fitsFile));
            process.onExit().thenAccept((p) -> {
                runningProcesses.remove(p);
                int rc = p.exitValue();
                if (rc != 0) {
                    LOG.log(Level.WARNING, () -> String.format("Process for file %s terminated with non-zero return code %d", fitsFile, rc));
                } else {
                    LOG.log(Level.INFO, () -> String.format("Process for file %s complete", fitsFile));
                    String raft = processBuilder.environment().getOrDefault("RAFT", "R??");
                    KeyValueDataList keyValueDataList = scanFileForTelemetry(logFile, "scripts/" + raft + "/" + slot);
                    if (!keyValueDataList.getListOfKeyValueData().isEmpty()) {
                        agent.publishSubsystemDataOnStatusBus(keyValueDataList);
                    }

                }
            });
        } catch (IOException x) {
            LOG.log(Level.WARNING, "Unable to create process for file " + fitsFile, x);
        }
    }

    private KeyValueDataList scanFileForTelemetry(File logFile, String root) {
        KeyValueDataList kvdl = new KeyValueDataList("");
        try {
            try ( BufferedReader in = Files.newBufferedReader(logFile.toPath())) {
                for (;;) {
                    String line = in.readLine();
                    if (line == null) {
                        break;
                    }
                    Matcher matcher = TELEMETRY_PATTERN.matcher(line);
                    if (matcher.matches()) {
                        String key = matcher.group(1);
                        String value = matcher.group(2);
                        if (key.startsWith("/")) {
                            kvdl.addData(key, value);
                        } else {
                            kvdl.addData(root + "/" + key, value);

                        }
                    }

                }
            }
        } catch (IOException x) {
            LOG.log(Level.WARNING, "IO Error scanning log file: " + logFile, x);
        }
        return kvdl;
    }

}
