package org.lsst.ccs.subsystem.imagehandling;

import java.util.LinkedHashSet;
import java.util.Set;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.imagehandling.states.ImageHandlingState;
import org.lsst.ccs.utilities.ccd.Reb;

/**
 * A Class containing Reb information and the FitsService associated to the
 * give Reb.
 * This object is created in groovy.
 * 
 * This node can be enabled or disabled via the configuration service.
 * When this node is disabled fits files for the corresponding Reb will
 * not be written out.
 * 
 * @author The LSST CCS Team
 */
class RebNode implements HasLifecycle {

    // TODO: I don't understand why this field cannot be final.
    @SuppressWarnings("FieldMayBeFinal")
    private Reb reb;
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private FitsService fitsService;
    
    @ConfigurationParameter(category = "FitsHandling", units="unitless")
    @SuppressWarnings("FieldMayBeFinal")
    private volatile boolean enabled = true;            
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    
    @LookupPath
    private String path;
    
    private final Set<String> imagesInFlight = new LinkedHashSet<>();
            
    RebNode(Reb reb) {
        this.reb = reb;
    }

    boolean isEnabled() {
        return enabled;
    }
    
    FitsService getFitsService() {
        return fitsService;
    }
    
    Location getLocation() {
        return Location.of(reb.getFullName());
    }
    
    Reb getReb() {
        return reb;
    }

    @Override
    public void build() {
        agentStateService.registerState(ImageHandlingState.class, "The Image Handling State for a Reb", this);
        agentStateService.updateAgentComponentState(this, ImageHandlingState.IDLE);
    }
    
    @ConfigurationParameterChanger(propertyName = "enabled")
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
        updateState(enabled ? ImageHandlingState.IDLE : ImageHandlingState.DISABLED);            
    }
    
    protected final void updateState(ImageHandlingState state) {
        synchronized (agentStateService.getStateLock() ) {        
            
            ImageHandlingState currentState = getState();
            
            switch (state) {
                case DISABLED:
                    //We can go to DISABLED only from IDLE. 
                    //If we are in any data processing state
                    //we delay the transition to DISABLED to after we go back to IDLE.
                    if (currentState != ImageHandlingState.IDLE) {
                        return;
                    }
                    break;
                case STREAMING:
                case READING:
                    //We can transition to the image processing states only from
                    //the IDLE state
                    if ( currentState != ImageHandlingState.IDLE ) {
                        throw new RuntimeException("Illegal state transition. Cannot go to "+state+" while in state: "+getState());                        
                    }
                    break;                    
            }            
            agentStateService.updateAgentComponentState(this, state);

            //If we are in the IDLE state and the node has been disabled,
            //then transition to the DISABLED state.
            if ( state == ImageHandlingState.IDLE && !isEnabled() ) {
                agentStateService.updateAgentComponentState(this, ImageHandlingState.DISABLED);
            }
        }
    }
    
    private ImageHandlingState getState() {
        return (ImageHandlingState)agentStateService.getComponentState(path, ImageHandlingState.class);
    }

    synchronized void incrementImageCount(String name) {
        boolean wasEmpty = imagesInFlight.isEmpty();
        imagesInFlight.add(name);
        if (wasEmpty) {
            updateState(ImageHandlingState.READING);
        }
    }

    synchronized void decrementImageCount(String name) {
        boolean removed = imagesInFlight.remove(name);
        if (removed && imagesInFlight.isEmpty()) {
            updateState(ImageHandlingState.IDLE);
        }
    }
}
