package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.util.AsciiFuncs;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.daq.ims.Source;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.utilities.image.FitsCheckSum;

/**
 * Coordinates the processing of a single image. Note since it is possible that
 * there are multiple images in flight at once, more than one instance of this
 * class may be running simultaneously.
 *
 * @author tonyj
 */
class ImageHandler implements Callable<FileList> {

    private static final Logger LOG = Logger.getLogger(ImageHandler.class.getName());

    private final Image image;
    private final ImageHandlingConfig config;
    private final ExecutorService executor;
    private final List<RebNode> rebs;
    private final boolean isStreaming;
    private final CountDownLatch darkTime = new CountDownLatch(1);
    private final CommandExecutor commandExecutor;
    private final CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData;
    private final boolean isHeaderServiceEnabled;

    ImageHandler(Image image, ExecutorService executor, ImageHandlingConfig config, List<RebNode> rebs, boolean isStreaming, boolean isHeaderServiceEnabled, CommandExecutor commandExecutor) {
        this.image = image;
        this.config = config;
        this.executor = executor;
        this.rebs = rebs;
        this.isStreaming = isStreaming;
        this.isHeaderServiceEnabled = isHeaderServiceEnabled;
        this.commandExecutor = commandExecutor;
        this.futureHeaderData = new CompletableFuture<List<ImageHeaderData.Header>>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15), TimeUnit.SECONDS);
    }

    @Override
    public FileList call() throws IOException, DAQException, InterruptedException, ExecutionException {
        // Image has one source for each REB associated with the image
        List<Source> sources = image.listSources();
        Set<Location> locationsToProcess = config.getLocationsToProcess();
        List<CompletableFuture<FileList>> lffl = new ArrayList<>();
        sources.stream()
                .filter((source) -> (locationsToProcess.contains(source.getLocation())))
                .forEach((source) -> {
                    Location location = source.getLocation();
                    for (RebNode r : rebs) {
                        if (r.isEnabled() && r.getLocation().equals(location)) {
                            final SourceHandler sourceHandler = new SourceHandler(darkTime, image.getMetaData().getName(), source, config, r, isStreaming);
                            CompletableFuture<FileList> futureFilelist = CompletableUtils.asyncCallable(executor, sourceHandler);
                            if (config.getWaitForHeaderService() == ImageHandlingConfig.WaitForHeaderService.ALWAYS || 
                               (config.getWaitForHeaderService() == ImageHandlingConfig.WaitForHeaderService.AUTO && isHeaderServiceEnabled)) {
                                futureFilelist = futureFilelist.thenCombine(futureHeaderData, (fl, headerData) -> {
                                    if (headerData == null) {
                                        LOG.log(Level.WARNING, "Header service data did not arrive in a timely manner");
                                    } else {
                                        try {
                                            fixupFitsFiles(fl, headerData);
                                        } catch (FitsException | IOException x) {
                                            LOG.log(Level.WARNING, "Error while fixing FITS headers", x);
                                        }
                                    }
                                    return fl;
                                });
                            }
                            lffl.add(futureFilelist);
                            if (!config.getCommands().isEmpty()) {
                                futureFilelist.thenAccept(fl -> {
                                    Map<String, String> env = new HashMap<>();
                                    env.put("BOARD", location.getBoardName());
                                    env.put("RAFT", location.getRaftName());
                                    String imageName = image.getMetaData().getName();
                                    env.put("IMAGENAME", imageName);
                                    try {
                                        ImageName in = new ImageName(imageName);
                                        env.put("SEQNO", in.getNumberString());
                                        env.put("DATE", in.getDateString());
                                        env.put("CONTROLLER", in.getController().getCode());
                                        env.put("SOURCE", in.getSource().getCode());
                                    } catch (IllegalArgumentException x) {
                                        env.put("SEQNO", "UNKNOWN");
                                        env.put("DATE", "UNKNOWN");
                                        env.put("CONTROLLER", "UNKNOWN");
                                        env.put("SOURCE", "UNKNOWN");
                                    }
                                    commandExecutor.execute(fl, env);
                                });
                            }
                        }
                    }
                });
        FileList result = new FileList();
        for (Future<FileList> ffl : lffl) {
            FileList fl = ffl.get();
            result.addAll(fl);
        }
        return result;
    }

    void darkTimeArrived() {
        darkTime.countDown();
    }

    void headerDataArrived(List<ImageHeaderData.Header> headers) {
        futureHeaderData.complete(headers);
    }

    private void fixupFitsFiles(FileList fileList, List<ImageHeaderData.Header> headerData) throws FitsException, IOException {
        for (File file : fileList) {
            Fits fits = new Fits(file);
            final BasicHDU<?> hdu = fits.getHDU(0);
            Header header = hdu.getHeader();
            long deltaCheckSum = 0;
            for (ImageHeaderData.Header headerServiceHeader : headerData) {
                String keyword = headerServiceHeader.getKeyword();
                if (keyword == null || "COMMENT".equals(keyword)) {
                    continue;
                }
                if (keyword.startsWith("HIERARCH")) keyword = keyword.replace(" ",".");
                HeaderCard card = header.findCard(keyword);
                String newValue = headerServiceHeader.getValue();

                if (card != null) {
                    String oldValue = card.getValue();

                    if ((oldValue == null || oldValue.isEmpty()) && !(newValue == null || newValue.isEmpty())) {
                        long oldCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        if (!isString(newValue)) {
                            card.setValue(headerServiceHeader.getValue());
                        } else {
                            card = new HeaderCard(card.getKey(), newValue, card.getComment());
                            header.updateLine(card.getKey(), card);
                        }
                        long newCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        deltaCheckSum += newCheckSum - oldCheckSum;
                    } else if (!Objects.equals(oldValue, newValue)) {
                        LOG.log(Level.WARNING, "For card {0} header service={1} but CCS={2}", new Object[]{card.getKey(), newValue, oldValue});
                    }
                } else {
                    LOG.log(Level.WARNING, "For card {0} header service={1} but CCS was not defined", new Object[]{keyword, newValue});
                }
            }
            FitsCheckSum.updateCheckSum(header, deltaCheckSum);
            header.rewrite();
        }
    }

    private boolean isString(String string) {
        if ("true".equalsIgnoreCase(string) || "false".equalsIgnoreCase(string)) {
            return false;
        }
        try {
            Double.parseDouble(string);
            return false;
        } catch (NumberFormatException x) {
            return true;
        }
    }
}
