package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;

/**
 * A call for executing commands on a FITS file
 * @author tonyj
 */
class CommandExecutor {

    private static final Logger LOG = Logger.getLogger(CommandExecutor.class.getName());

    private final ImageHandlingConfig config;
    private final Queue<Process> runningProcesses = new ConcurrentLinkedQueue<>();
    
    CommandExecutor(ImageHandlingConfig config) {
        this.config = config;
    }
    
    void execute(FileList fileList,  Map<String, String> env) {
        for (File file : fileList) {
            execute(file, env);
        }
    }
     
    void execute(File fitsFile,  Map<String, String> env) {
        // TODO: Chain multiple commands together
        List<String> commands = config.getCommands();
        List<String> command = List.of("/bin/bash","-c", commands.stream().collect(Collectors.joining(";")));
        String date = env.getOrDefault("DATE", "UNKNOWN");
        final File logFileDir = new File(config.getLogDirectory(), date);
        File logFile = new File(logFileDir, fitsFile.getName()+".log");
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.environment().putAll(config.getEnvironment());
        processBuilder.environment().putAll(env);
        processBuilder.environment().put("LOGFILE", logFile.getAbsolutePath());
        processBuilder.environment().put("FITSFILE", fitsFile.getAbsolutePath());
        processBuilder.directory(config.getCurrentDirectory());
        processBuilder.redirectErrorStream(true);
        processBuilder.redirectOutput(logFile);
        try {
            Files.createDirectories(logFileDir.toPath());
            Process process = processBuilder.start();
            runningProcesses.add(process);
            LOG.log(Level.INFO, () -> String.format("Process for file %s started", fitsFile));
            process.onExit().thenAccept((p) -> {
                runningProcesses.remove(p);
                int rc = p.exitValue();
                if (rc != 0) {
                   LOG.log(Level.WARNING, () -> String.format("Process for file %s terminated with non-zero return code %d", fitsFile, rc));
                } else {
                   LOG.log(Level.INFO, () -> String.format("Process for file %s complete", fitsFile));
                }
            });
        } catch (IOException x) {
            LOG.log(Level.WARNING, "Unable to create process for file "+fitsFile, x);
        }
    }

}
